package com.liveperson.messaging.commands;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.utils.TokenUtils;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;


/**
 * Created by Akshay on 8/28/19.
 */
public class GetUnreadMessagesCountFromUMSCommand implements Command {

    public static final String TAG = "GetUnreadMessagesCountCommand";

    private static final String UMS_UNREAD_COUNT_URL = "https://%s/api/account/%s/messaging/consumer/conversation/unread-messages-count?v=3";

    public static final String AUTHORIZATION = "Authorization";
    private static final String USER_AGENT = "User-Agent";
    private static final String ANDROID = "ANDROID";
    private static final String ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to make request.";
    private static final String ERROR_SERVER_ERROR = "Server error: ";

    private static final String UNREAD_COUNT = "unread_count";
    private static final String UNREAD_REQUEST_TIMESTAMP = "unread_request_timestamp";
    private static final long LAST_REQUEST_TIMESTAMP_THRESHOLD = 1000 * 10; // 10 seconds
    private static final int REQUEST_TIMEOUT = 30000;

    private final Messaging mController;
    private String mBrandId;
    private ICallback<Integer, Exception> mCallback;
    private String mUmsDomain;
    private PreferenceManager mPreferenceManager;
    private String mLocalToken;

    public GetUnreadMessagesCountFromUMSCommand(Messaging messagingController, String brandId, ICallback<Integer, Exception> callback) {
        mController = messagingController;
        mBrandId = brandId;
        mCallback = callback;
        mPreferenceManager = PreferenceManager.getInstance();

    }

    @Override
    public void execute() {

        //Checking if the time threshold was passed
        if (!isRequestThresholdWasPassed(mBrandId)) {
            LPMobileLog.d(TAG, "Time threshold was not passed yet. Return cached value of unread Messages");
            returnCachedDetails();

        } else {

            LPMobileLog.d(TAG, "Time threshold was passed. Make getUnreadMessageCount request");

            // If the AmsAccount does not exist for the given brandId (e.g. the app was killed and no conversation started yet),
            // we try to get the token from shared preferences since it's still does not exist in the account.
            if (mController.mAccountsController.getAccount(mBrandId) == null) {
                mLocalToken = getTokenFromSharedPreferences();
                if (TokenUtils.isJwtExpired(mLocalToken)) {
                    sendTokenExpired();
                    return;
                }
            } else if (mController.mAccountsController.isTokenExpired(mBrandId)) { // Check token from the account
                sendTokenExpired();
                return;
            }


            LPMobileLog.d(TAG, "JWT is valid - send request to get unread message count");
            mUmsDomain = mController.mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
            if (TextUtils.isEmpty(mUmsDomain)) {
                mUmsDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY, mBrandId, null);
                if (TextUtils.isEmpty(mUmsDomain)) {
                    notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Missing Domain"));
                    return;
                }
            }
            validateStateAndSendRequest(mUmsDomain);
        }
    }

    /**
     * Get the token that is stored in the shared preferences
     *
     * @return
     */
    private String getTokenFromSharedPreferences() {

        String token;
        String decryptedToken = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
        if (TextUtils.isEmpty(decryptedToken)) {
            token = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId, null);
            mPreferenceManager.remove(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId);
        } else {
            token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
        }

        return token;
    }

    /**
     * Send token expired callback and onError callback for this command
     */
    private void sendTokenExpired() {
        LPMobileLog.d(TAG, "JWT is expired - calling to onTokenExpired callback");
        mController.mEventsProxy.onTokenExpired();
        mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Token expired, refresh the token and try again"));
    }


    /**
     * Notify in case of a success with the relevant counter
     *
     * @param counter
     */
    private void notifySuccess(final int counter) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    mCallback.onSuccess(counter);
                }
            });
        }
    }

    /**
     * Notify in case of an error
     *
     * @param exception
     */
    private void notifyError(final Exception exception) {
        if (mCallback != null) {
            Infra.instance.postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    if(exception instanceof SSLPeerUnverifiedException){
                        mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, exception.getMessage());

                    }
                    mCallback.onError(exception);
                }
            });
        }
    }

    /**
     * Checking if we have an active internet connection and send the request to UMS
     *
     * @param umsDomain
     */
    private void validateStateAndSendRequest(final String umsDomain) {

        if (InternetConnectionService.isNetworkAvailable()) {
            sendRequest(umsDomain);

        } else {
            new SynchronizedInternetConnectionCallback(new Runnable() {
                @Override
                public void run() {
                    sendRequest(umsDomain);
                }
            }).execute();
        }
    }

    private String buildRequestURL(String umsDomain) {
        return String.format(UMS_UNREAD_COUNT_URL, umsDomain, mBrandId);
    }


    /**
     * Send a request to UMS to get unread message count
     *
     * @param requestURL
     */
    private void sendRequest(String requestURL) {
        requestURL = buildRequestURL(requestURL);
        String token = mController.mAccountsController.getToken(mBrandId);

        token = (token == null) ? mLocalToken : token;
        if (token == null || requestURL == null) {
            notifyError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " Error: Authorization failed. Token is missing or is invalid"));
            return;
        }

        List<String> certificates = mController.mAccountsController.getCertificatePinningKeys(mBrandId);

        HttpGetRequest httpGetRequest = new HttpGetRequest(requestURL);
        httpGetRequest.addHeader(AUTHORIZATION, token);
        httpGetRequest.addHeader(USER_AGENT, ANDROID);
        httpGetRequest.setCertificatePinningKeys(certificates);

        httpGetRequest.setTimeout(REQUEST_TIMEOUT);

        LPMobileLog.i(TAG, "GetUnreadMessageCount url " + requestURL);

        httpGetRequest.setCallback(new ICallback<String, Exception>() {
            @Override
            public void onSuccess(final String count) {
                try {
                    if (!TextUtils.isEmpty(count)) {
                        LPMobileLog.i(TAG, "onSuccess " + count);
                        if (mCallback != null) {
                            int unreadCount = Integer.parseInt(count);
                            updateCachedDetails(unreadCount);
                            notifySuccess(unreadCount);
                            return;
                        }
                    }
                    notifyError(new Exception(ERROR_SERVER_ERROR + "Empty response"));
                } catch (Exception error) {
                    LPMobileLog.e(TAG, "Failed to parse unread message count: " + error);
                    notifyError(error);
                }
            }

            @Override
            public void onError(Exception exception) {
                if (exception != null) {
                    LPMobileLog.e(TAG, "onError " + exception.getMessage());
                    if (mCallback != null) {
                        notifyError(new Exception(ERROR_SERVER_ERROR + exception.getMessage()));
                    }
                }
            }
        });
        HttpHandler.execute(httpGetRequest);
    }

    /**
     * Updating the counter and timestamp
     *
     * @param counter
     */
    private void updateCachedDetails(int counter) {
        PreferenceManager.getInstance().setLongValue(UNREAD_REQUEST_TIMESTAMP, mBrandId, System.currentTimeMillis());
        PreferenceManager.getInstance().setIntValue(UNREAD_COUNT, mBrandId, counter);
    }


    /**
     * Return cached count
     */
    private void returnCachedDetails() {
        LPMobileLog.d(TAG, "Return cached unread message count");
        int counter =  PreferenceManager.getInstance().getIntValue(UNREAD_COUNT, mBrandId, 0);
        notifySuccess(counter);
    }


    /**
     * Checking if the threshold was passed
     *
     * @param brandId
     * @return true - threshold was passed, false otherwise
     */
    private boolean isRequestThresholdWasPassed(String brandId) {

        long lastTimestamp = PreferenceManager.getInstance().getLongValue(UNREAD_REQUEST_TIMESTAMP, brandId, 0L);

        long delta = System.currentTimeMillis() - lastTimestamp;

        // If delta is greater than threshold or lastTimestamp never set
        if ((lastTimestamp == 0) || delta > LAST_REQUEST_TIMESTAMP_THRESHOLD) {
            return true;
        }
        return false;
    }
}