package com.liveperson.messaging.commands.tasks;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.LPAuthenticationParams;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.utils.TokenUtils;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsConnectionAnalytics;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.network.http.IdpRequest;
import com.liveperson.messaging.network.http.UnAuthRequest;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A task to get the token from the IDP service
 */
public class IdpTask extends BaseAmsAccountConnectionTask {

    public static final String TAG = "IdpTask";
    private final AmsUsers mAmsUsers;
    private AccountsController mAccountsController;

    private String mHostVersion;

    public IdpTask(AccountsController accountsController, AmsUsers amsUsers, String hostVersion){
        mHostVersion = hostVersion;
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
    }

    /**
     *
     */
    @Override
    public void execute() {
        LPMobileLog.d(TAG, "Running IDP task...");

        AmsConnectionAnalytics.idpTaskStart();

        // Try to get a locally existing token
        String token = mAccountsController.getToken(mBrandId);
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        boolean tokenExpired = mAccountsController.isTokenExpired(mBrandId);

        //if JWT exists and valid (not expired) // could be only in un-auth flow, cause otherwise we would stop connecting flow in ConnectionStateMachine.
        if (!TextUtils.isEmpty(token) && !tokenExpired) {

            LPMobileLog.d(TAG, "execute - token exists and valid: " + token);
            AmsConnectionAnalytics.idpTaskEnd();
            mCallback.onTaskSuccess();

        } else { // Token does not exist. Send a request to IDP service

            String idpDomain = getIdpDomain();

            LPAuthenticationParams auth = mAccountsController.getLPAuthenticationParams(mBrandId);
            if (auth.getAuthType() == LPAuthenticationParams.LPAuthenticationType.UN_AUTH) {

                if (tokenExpired) {
                    //clear LP JWT, we are going to generate new one!
                    //we won't clean Non Auth code, cause we need to for refreshing the user.
                    mAccountsController.setToken(mBrandId, null);
                }

                String connectorId = mAccountsController.getAccount(mBrandId).getConnectorId();//mCampaignInfo == null? "" : mCampaignInfo.getConnectorId();

                new UnAuthRequest(mAccountsController.getAccount(mBrandId), idpDomain, mBrandId, auth, mHostVersion, getIdpCallback(), certificates, connectorId).execute();
            } else {
                //Auth and SignUp:
                new IdpRequest(mAccountsController.getAccount(mBrandId), idpDomain, mBrandId, auth, mHostVersion, getIdpCallback(), certificates).execute();
            }
        }
    }

    @NonNull
    private IDPExceptionICallback getIdpCallback() {
        return new IDPExceptionICallback();
    }

    /**
     * Get the IDP domain from CSDS. If the "IDP" does not exist in the CSDS domains, we get the AsyncMessagingEnt
     * and replace the "msg" to "idp
     *
     * @return
     */
    private String getIdpDomain() {

        String idpDomain;

		// Get the IDP domain from CSDS
		idpDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_IDP_DOMAIN_KEY);

		// The IDP domain is not always in CSDS. If we got an empty string we go get the AccountsController.AMS_CSDS_DOMAIN_KEY instead
		if (TextUtils.isEmpty(idpDomain)) {
			String asyncMessagingEnt = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);

            // Replace the "msg" to "idp"
            idpDomain = asyncMessagingEnt.replaceFirst("msg", "idp");
        }

        return idpDomain;
    }


    @Override
    public String getName() {
        return TAG;
    }

    public class IDPExceptionICallback implements ICallback<String, Exception> {

        @Override
        public void onSuccess(String token) {
            LPMobileLog.d(TAG, "onSuccess: got token ");
            mAccountsController.setToken(mBrandId, token);
            mAccountsController.getAccount(mBrandId).setOriginalConsumerId(TokenUtils.getOriginalConsumerIdFromJWT(token));
            //set consumer user id
            mAmsUsers.updateConsumerId(mBrandId, TokenUtils.getConsumerUserId(token));
            AmsConnectionAnalytics.idpTaskEnd();

            mAccountsController.getAccount(mBrandId).sendAuthenticationCompletedStatus();

            mCallback.onTaskSuccess();
        }

        @Override
        public void onError(Exception exception) {
            LPMobileLog.d(TAG, "onError: failed to connect to idp!");
            if(exception instanceof SSLPeerUnverifiedException){
                mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, exception);
            }else {
                mCallback.onTaskError(TaskType.IDP, exception);
            }
        }

        public void onError(TaskType taskType, Exception exception) {
            LPMobileLog.d(TAG, "onError: failed to connect to idp! taskType = " + taskType + ", error: " + exception.getMessage());
            mCallback.onTaskError(taskType, exception);
        }
    }
}
