package com.liveperson.messaging.utils;

import android.text.TextUtils;
import android.util.Base64;

import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.messaging.R;

import org.json.JSONObject;

import java.util.Date;

/**
 * Created by nirni on 11/7/17.
 */
public class TokenUtils {

	public static final String TAG = TokenUtils.class.getSimpleName();

	private static int jwtExpirationBuffer = -1;


	public static boolean isJwtExpired(String jwt) {

		// Get it once
		if(jwtExpirationBuffer != -1) {
			jwtExpirationBuffer = Configuration.getInteger(R.integer.lp_messaging_buffer_expiration_seconds);
		}

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return false;
		} else {
			LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "jwt exists: " + jwt + " checking if expired..");

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return false;
			}

			//take the second segment, decode it (base64), convert it to json object, and find "exp" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, "UTF-8");
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);
				Long expirationInSeconds = secondSegmentsJson.getLong("exp");

				Long iat = secondSegmentsJson.getLong("iat");
				long expirationInMillis = expirationInSeconds * 1000;
				LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "expiration = " + new Date(expirationInMillis).toString());
				LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "iat = " + new Date(iat*1000).toString());

				if (isGoingToExpire(expirationInMillis)){
					LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "JWT is expired or about to.. ");
					return true;
				}else{
					LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "JWT is still valid ");
				}

			} catch (Exception e) {
				LPMobileLog.e(TAG, e);
			}
			return false;

		}
	}

	private static boolean isGoingToExpire(long expirationInMillis) {
		long bufferTime = jwtExpirationBuffer*1000;
		long time = System.currentTimeMillis() + bufferTime;
		if( time > expirationInMillis ){
			return true;
		}
		return false;
	}


	public static String getOriginalConsumerIdFromJWT(String jwt) {

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return "";
		} else {
			LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "Getting original consumerId from jwt: " + jwt);

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return "";
			}
			//example:
			// 			{
			//				"sub": "b555ec39a12c560c2c76b43f5e218ccbd46c981318bd0df1c94db72aa48b9d95",
			//					"aud": "acc:le22225965",
			//					"acr": "0",
			//					"iss": "https://idp.liveperson.net",
			//					"lp.ext": {
			//							"sub": "1fcbeb5e-a245-4096-8beb-5ea24560964e",
			//							"iss": "https://idp.liveperson.net"
			//					},
			//					"exp": 1524988657,
			//					"iat": 1524988057
			//			}
			//take the second segment, decode it (base64), convert it to json object, and find "lp.ext"."sub" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, "UTF-8");
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);

				JSONObject lpExt = secondSegmentsJson.optJSONObject("lp.ext");
				if (lpExt != null){
					return lpExt.optString("sub");
				}

			} catch (Exception e) {
				LPMobileLog.e(TAG, e);
			}
			return "";

		}
	}


	public static String getConsumerUserId(String jwt) {

		if (TextUtils.isEmpty(jwt)) {
			//no jwt exists, we'll run normal flow
			return "";
		} else {
			LPMobileLog.d(TAG, LPMobileLog.FlowTags.LOGIN, "Getting original consumerId from jwt: " + jwt);

			String [] segments = jwt.split("\\.");
			if (segments.length < 3){
				return "";
			}
			//example:
			// 			{
			//				****"sub": "b555ec39a12c560c2c76b43f5e218ccbd46c981318bd0df1c94db72aa48b9d95",*****
			//					"aud": "acc:le22225965",
			//					....
			//					"exp": 1524988657,
			//					"iat": 1524988057
			//			}
			//take the second segment, decode it (base64), convert it to json object, and find "lp.ext"."sub" value.
			try {
				byte[] secondSegmentsByte = Base64.decode(segments[1], Base64.DEFAULT);
				String secondSegments = new String(secondSegmentsByte, "UTF-8");
				JSONObject secondSegmentsJson = new JSONObject(secondSegments);
				String sub = secondSegmentsJson.optString("sub");
				return sub;
			} catch (Exception e) {
				LPMobileLog.e(TAG, e);
			}
			return "";

		}
	}
}
