package com.liveperson.messaging.background.filesharing.document;

import androidx.annotation.Nullable;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.background.DownloadFileTask;
import com.liveperson.messaging.background.filesharing.DownloadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.FileSharingType;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

public class DownloadDocumentTask extends DownloadFileTask {

    public static final String TAG = "DownloadDocumentTask";

    private FileSharingType mSubFileType;
    private File mFilesFolder;
    private static String DOCUMENTS_FOLDER = "documents/";
    private String mBrandId;

    public DownloadDocumentTask(DownloadFileTaskBundle params, FileSharingType subFileType) {
        super(params);
        mSubFileType = subFileType;
        mFilesFolder = Infra.instance.getApplicationContext().getFilesDir();
        mBrandId = params.getBrandId();
    }

    @Override
    protected String saveFileToDisk(byte[] byteArray) {
        return saveByteArrayToDisk(byteArray);
    }

    /**
     * Save the given byte array to disk to the given file
     *
     * @param fileByteArray
     * @return path to saved file
     */
    private String saveByteArrayToDisk(byte[] fileByteArray) {

        // Generate filename
        File filePath = new File(getDocumentsFolder(), generateDocumentName(mSubFileType));


        LPMobileLog.d(TAG, "saveByteArrayToDisk: filePath: " + filePath.getAbsolutePath());

        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(filePath);

            fos.write(fileByteArray);

        } catch (FileNotFoundException e) {

            LPMobileLog.e(TAG,"saveByteArrayToDisk: File not found", e);
            return null;

        } catch (IOException e) {
            LPMobileLog.e(TAG, "saveByteArrayToDisk: IOException", e);

        } finally {
            try {
                if (fos != null) {

                    fos.close();
                }
            } catch (IOException e) {
                LPMobileLog.e(TAG, "saveByteArrayToDisk: error closing file", e);
                return null;
            }
        }

        LPMobileLog.d(TAG, "saveByteArrayToDisk: file absolute path: " + filePath.getAbsolutePath());
        return filePath.getAbsolutePath();
    }


    private String generateDocumentName(FileSharingType fileType) {

        switch (fileType){
        case DOCX:
            return UniqueID.createUniqueMessageEventId() + ".docx";
        case PPTX:
            return UniqueID.createUniqueMessageEventId() + ".pptx";
        case XLSX:
            return UniqueID.createUniqueMessageEventId() + ".xlsx";
        default:
            return UniqueID.createUniqueMessageEventId() + ".pdf";
        }
    }


    /**
     * Return the  folder, create it if not exist
     * @return the documents folder path or null if does not exist and cannot be created
     */
    @Nullable
    private String getDocumentsFolder() {
        File outgoingDocumentFolder = new File(mFilesFolder + "/" + DOCUMENTS_FOLDER + "/" + mBrandId);
        if (!outgoingDocumentFolder.exists()) {
            if (!outgoingDocumentFolder.mkdirs()) {
                LPMobileLog.e(TAG, "getDocumentsFolder: /documents folder could not be created");
                return null;
            }
        }

        return outgoingDocumentFolder.getPath();
    }


}
