package com.liveperson.messaging.commands;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.UnregisterPushRequest;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.model.MessagingUserProfile;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import java.util.List;

/**
 * Command to unregister from push notifications for a given brandId.
 * <p>
 * Created by nirni on 1/27/16.
 */
public class UnregisterPusherCommand implements Command {

    public static final String TAG = UnregisterPusherCommand.class.getSimpleName();

    private static final String PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister";
    private final AccountsController mAccountsController;
    private final AmsUsers mAmsUsers;

    private String mBrandId;
    private String mAppId;
    private ICallback<Void, Exception> listener;
    private boolean runImmediately;

    public UnregisterPusherCommand(AccountsController accountsController, AmsUsers amsUsers, String brandId, String appId, ICallback<Void, Exception> listener, boolean immediately) {
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
        mBrandId = brandId;
        mAppId = appId;

        this.listener = listener;
        runImmediately = immediately;
    }


    @Override
    public void execute() {

        LPMobileLog.i(TAG, "execute unregister for brandId " + mBrandId);

        // Create pusher unregister URL
        String pusherDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
        if (TextUtils.isEmpty(pusherDomain)) {
            pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
            if (TextUtils.isEmpty(pusherDomain)) {
                LPMobileLog.w(TAG, "pusherDomain does not exists. Quit unregister push");
                notifySuccess();
                return;
            }
        }
        final String pusherURL = String.format(PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);

        // Get consumerId from memory
        String consumerId = mAmsUsers.getConsumerId(mBrandId);

        // If consumerId is not available in memory try to get from DB
        if (TextUtils.isEmpty(consumerId)) {
            LPMobileLog.d(TAG, "execute: consumerId is not available. Trying to get from DB...");
            mAmsUsers.getUserByBrandIDFromDB(mBrandId).setPostQueryOnBackground(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
                @Override
                public void onResult(MessagingUserProfile userProfile) {
                    if (userProfile != null) {
                        String originatorId = userProfile.getOriginatorId();
                        if (!TextUtils.isEmpty(originatorId)) {
                            runUnregisterPush(originatorId, pusherURL);
                            LPMobileLog.d(TAG, "onResult: got  consumerId from DB (" + originatorId + "). Unregister push with it...");
                        } else {
                            LPMobileLog.w(TAG, "onResult: Cannot get consumerId from DB. Quit unregister push");
                        }
                    } else {
                        LPMobileLog.w(TAG, "onResult: Cannot get user profile from DB. Quit unregister push");
                    }

                }


            }).setPostQueryOnUI(new DataBaseCommand.QueryCallback<MessagingUserProfile>() {
                @Override
                public void onResult(MessagingUserProfile data) {
                    if (data == null || TextUtils.isEmpty(data.getOriginatorId())) {
                        //if we don't have a consumer id it counts as success.
                        notifySuccess();
                    }
                }
            }).execute();
        } else {
            runUnregisterPush(consumerId, pusherURL);
        }
    }

    private void notifySuccess() {
        if (listener != null) {
            listener.onSuccess(null);
        }
    }

    private void runUnregisterPush(final String consumerId, final String pusherURL) {
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        if (runImmediately) {
            LPMobileLog.d(TAG, "run: Unregister push immediately");
            if (InternetConnectionService.isNetworkAvailable()) {
                // If available execute immediately
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(listener).execute();
            } else {
                if (listener != null) {
                    listener.onError(new Exception("No network available"));
                }
            }
        } else {
            new SynchronizedInternetConnectionCallback(new Runnable() {

                @Override
                public void run() {
                    LPMobileLog.d(TAG, "run: Unregister push for consumerId: " + consumerId);
                    new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(listener).execute();
                }
            }).execute();
        }
    }
}
