package com.liveperson.messaging.controller;

import android.net.Uri;
import android.text.TextUtils;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by ofira on 07/12/2015.
 */
public class AccountsController implements Clearable{

	private static final String TAG = AccountsController.class.getSimpleName();

    public static final String QA_CSDS_DOMAIN = "hc1n.dev.lprnd.net"; // QA
    public static final String ALPHA_CSDS_DOMAIN = "adminlogin.liveperson.net"; // Production

    protected static final String KEY_TOKEN = "brand_token";
    //    protected static final String SOCKET_URL = "wss://%1$s/api/messaging/ws/2/consumer/%2$s";
    protected static final String SOCKET_URL = "wss://%1$s/ws_api/account/%2$s/messaging/consumer";
    protected static final String DOMAIN_VERSION_QUERY = "v";
    protected static final int DOMAIN_VERSION_VALUE = 3;

	protected Map<String, AmsAccount> mAccounts;
    private ClientProperties mClientProperties;


    public AccountsController(ClientProperties clientProperties) {
        mAccounts = new HashMap<>();
        mClientProperties = clientProperties;
    }


	/**
     * @param brand
     */
    public void addNewAccount(String brand) {
        if (getAccount(brand) == null) {
            LPMobileLog.i(TAG, "Adding new account: " + brand);
            mAccounts.put(brand, new AmsAccount(brand));
        }
    }


    /**
     * @param brand
     * @return
     */
    public String getToken(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return null;
        }
        return account.getToken();
    }

    /**
     * Certificate Pinning public key
     * @param brand
     * @return
     */
    public List<String> getCertificatePinningKeys(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return null;
        }
        return account.getCertificatePinningKeys();
    }


    /**
     * @param brand
     * @return
     */
    public boolean isTokenExpired(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return false;
        }
        return account.isTokenExpired() || account.isHostAppJWTExpired();
    }

    /**
     * @param brand
     * @return
     */
    public LPAuthenticationParams getLPAuthenticationParams(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return null;
        }
        return account.getLPAuthenticationParams();
    }

    /**
     * @param brand
     * @return
     */
    public void setLPAuthenticationParams(String brand, LPAuthenticationParams lpAuthenticationParams) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return;
        }
        account.setLPAuthenticationParams(lpAuthenticationParams);
    }



    /**
     * @param brand
     * @param token
     */
    public void setToken(String brand, String token) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return;
        }
        account.setToken(token);
    }

	/**
	 * Set CSDS values to persistent memory
	 * @param brand
	 * @param csdsUrls
	 * @return true - if one of the relevant values were changed, false - if non were changed
	 */
    public boolean setCSDSMap(String brand, HashMap<String, String> csdsUrls) {

		boolean dataUpdatedWithNewValue;

		AmsAccount account = getAccount(brand);
        if (account == null) {
            return false;
        }

		dataUpdatedWithNewValue = account.setCSDSMap(csdsUrls);

		initLoggos(brand);

		return dataUpdatedWithNewValue;
	}

	private void initLoggos(String brand) {
		// Init Loggos with domain and brand
		String loggosDomain = getServiceUrl(brand, ConnectionParamsCache.CSDS_LOGGOS_DOMAIN_KEY);
		List<String> certificates = getCertificatePinningKeys(brand);
		Infra.instance.getLoggos().init(brand, loggosDomain, certificates);
	}

	public boolean isCsdsDataMissing(String brand) {
        AmsAccount account = getAccount(brand);
		if (account == null) {
			return false;
		}

		return account.isCsdsDataMissing();
    }

    public void setAppId(String brand, String appId){
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return;
        }
        account.setAppId(appId);
    }

    /**
     * @param brand
     * @return
     */
    public String getDomain(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return null;
        }
        return account.getDomain();
    }

    public String getServiceUrl(String brand, String serviceName) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
            return null;
        }
        return account.getServiceUrl(serviceName);
    }

	/**
	 * Get the {@link com.liveperson.messaging.controller.connection.ConnectionParamsCache} for the given brandId
	 * @param brandId
	 * @return
	 */
	public IConnectionParamsCache getConnectionParamsCache(String brandId) {
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			return null;
		}
		return account.getConnectionParamsCache();

	}

	/**
	 * Check whether AutoMessages is enabled on the given account (according to LPTag)
	 * @param brandId
	 * @return true - enabled, false - disabled
	 */
	public boolean isAutoMessagesEnabled(String brandId){
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			return false;
		}

		return account.isAutoMessagesEnabled();
	}
	/**
     * Return connection url for socket based on brand id
     *
     * @param brandId
     * @return
     */
    public String getConnectionUrl(String brandId) {

        AmsAccount account = getAccount(brandId);
        if (account == null) {
            return null;
        }
		String connectionUrl = account.getServiceUrl(ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
		String token = account.getToken();
		if (TextUtils.isEmpty(connectionUrl) || TextUtils.isEmpty(token)) {
			return null;
		}

		String path = String.format(SOCKET_URL, connectionUrl, brandId);
        Uri fullUri = Uri.parse(path).buildUpon()
                .appendQueryParameter(DOMAIN_VERSION_QUERY, String.valueOf(DOMAIN_VERSION_VALUE)).build();

        //Header: authorization : jwt token
        //Header: client properties : json
        return fullUri.toString();//path + mQueriesParams;
	}

	public String getTokenizerUrl(String brandId) {
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			return null;
		}
		return account.getServiceUrl(ConnectionParamsCache.CSDS_AMS_TOKENIZER_DOMAIN_KEY);
	}

    public String getConnectionUrlForLogs(String brandId) {
        AmsAccount account = getAccount(brandId);
        if (account == null) {
            return null;
        }
        String connectionUrl = account.getServiceUrl(ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
        String token = account.getToken();
        if (TextUtils.isEmpty(connectionUrl) || TextUtils.isEmpty(token)) {
            return null;
        }
        return String.format(SOCKET_URL, connectionUrl, brandId, "private_token");
    }

    /**
     * Return the {@link AmsAccount} of specific brand
     *
     * @param brand
     * @return
     */
    public AmsAccount getAccount(String brand) {
        return mAccounts.get(brand);
    }

    @Override
    public void clear() {
        mAccounts.clear();
    }

    public ClientProperties getClientProperties() {
        return mClientProperties;
    }

    public boolean isInUnAuthMode(String brandId){
        AmsAccount account = getAccount(brandId);
        if (account == null) {
            return false;
        }
        return account.isInUnAuthMode();
    }
}
