package com.liveperson.messaging.network.http;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPostRequest;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.commands.tasks.IdpTask;
import com.liveperson.messaging.model.AmsAccount;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Created by shiranr on 3/27/18.
 */

public class UnAuthRequest extends IdpRequest {

    public static final String TAG = "UnAuthRequest";
    ///api/account/{accountId}/app/anonymous/authorize
    //This code is also in Monitoring module, be careful in case you are changing this code.
    public static final String IDP_ANONYMOUS_URL = "https://%s/api/account/%s/anonymous/authorize";
    protected String mConnectorId;

    public UnAuthRequest(AmsAccount account, String idpDomain, String brandId, LPAuthenticationParams lpAuthenticationParams,
                         String hostVersion, IdpTask.IDPExceptionICallback callback, List<String> certificates, String connectorId) {
        super(account, idpDomain, brandId, lpAuthenticationParams, hostVersion, callback, certificates);
        mConnectorId = connectorId;
    }

    @Override
    public void execute() {
        sendUnAuthRequest();
    }

    public void sendUnAuthRequest() {

        ///api/account/{accountId}/app/anonymous/authorize
        String idpUrl = String.format(IDP_ANONYMOUS_URL, mIdpDomain, mBrandId);
        final HttpPostRequest unAuthJwtRequest = new HttpPostRequest(idpUrl);

        // Add the body if exist - case of refresh
        String nonAuthCode = mAccount.getNonAuthCode();

        if (!TextUtils.isEmpty(nonAuthCode)){
            LPMobileLog.d(TAG, "Refresh anonymous token");
            //if there is non auth code exists, we want to refresh it, we'll send it to be refreshed
            JSONObject jsonBody = new JSONObject();
            try {
                jsonBody.put("id_token", nonAuthCode);
            } catch (JSONException e) {
            }
            HttpRequestBody body = new LPJSONObjectBody(jsonBody);
            unAuthJwtRequest.setBody(body);
        }else{
            LPMobileLog.d(TAG, "New anonymous token");
        }

        unAuthJwtRequest.setCallback(new ICallback<String, Exception>() {

            @Override
            public void onSuccess(String idpResponse) {
                LPMobileLog.d(TAG, "onSuccess " + idpResponse);

                if (!TextUtils.isEmpty(idpResponse)) {

                    try {
                        JSONObject idpJson = new JSONObject(idpResponse);
                        String token = idpJson.getString("token");
                        mAccount.setNonAuthCode(token);

                        //After we have the anonymous token, we are following the usual idp request
                        HttpPostRequest httpPostRequest = getHttpPostRequestForIDPV3(token);
                        sendGeneralRequest(httpPostRequest);

                    } catch (JSONException je) {
                        LPMobileLog.d(TAG, "JSONException: ", je);
                        String errorMsg = "Error parsing onSuccess response: idp url = " + unAuthJwtRequest.getUrl() + ". Exception " + je.getMessage();
                        sendErrorCallback(new Exception(errorMsg));
                    }
                }
            }

            @Override
            public void onError(Exception exception) {
                LPMobileLog.w(TAG, "Error: idp url = " + unAuthJwtRequest.getUrl() + ". Exception ", exception);
                String message = exception.getMessage();
                if (!TextUtils.isEmpty(message) && message.contains(USER_EXPIRED_ERROR)
                        // The following check is super safety - we can get 2001-
                        // 'USER EXPIRED' error only when trying to refresh the current non auth code, so we must have it
                        && !TextUtils.isEmpty(mAccount.getNonAuthCode())){

                    mCallback.onError(TaskType.USER_EXPIRED, exception);
                }else{
                    sendErrorCallback(exception);
                }
            }
        });

        sendGeneralRequest(unAuthJwtRequest);
    }


    @NonNull
    protected HttpPostRequest getHttpPostRequestForIDPV3(String anonymousJwt) {
        JSONObject jsonBody = null;

        if (!TextUtils.isEmpty(anonymousJwt)) {
            jsonBody = getBodyForIDPJwtFlow(anonymousJwt);

        }
        LPMobileLog.d(TAG, "Idp json body: " + jsonBody);

        String IDP_URL = "https://%s/api/account/%s/app/%s/%s?v=3.0";
        String idpUrl = String.format(IDP_URL, mIdpDomain, mBrandId,mConnectorId, AUTHENTICATE);
        final HttpPostRequest httpPostRequest = new HttpPostRequest(idpUrl);

        // Add the body if exist
        HttpRequestBody body = new LPJSONObjectBody(jsonBody);
        httpPostRequest.setBody(body);

        httpPostRequest.setCallback(new ICallback<String, Exception>() {

            @Override
            public void onSuccess(String idpResponse) {
                LPMobileLog.d(TAG, "onSuccess " + idpResponse);

                if (!TextUtils.isEmpty(idpResponse)) {

                    try {
                        JSONObject idpJson = new JSONObject(idpResponse);
                        String token = idpJson.getString("token");
                        mCallback.onSuccess(token);
                    } catch (JSONException je) {
                        LPMobileLog.d(TAG, "JSONException: ", je);
                        String errorMsg = "idp url = " + httpPostRequest.getUrl() + ". Exception " + je.getMessage();
                        sendErrorCallback(new Exception(errorMsg));
                    }
                }
            }

            @Override
            public void onError(Exception e) {
                LPMobileLog.d(TAG, "Error: idp url = " + httpPostRequest.getUrl() + ". Exception ", e);
                sendErrorCallback(e);
            }
        });


        return httpPostRequest;
    }

}
