package com.liveperson.messaging.commands.tasks;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.model.LptagData;
import com.liveperson.infra.network.http.requests.LptagRequest;
import com.liveperson.infra.utils.DispatchQueue;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.model.AmsConnectionAnalytics;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 9/11/17.
 * <p/>
 * Base connection task to fetch LPTag
 */
public class LptagTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "LptagTask";
    private final AccountsController mAccountsController;

    /**
     * This {@link DispatchQueue} is used in case we received a Retry reply from LPTag
     */
    private DispatchQueue mDispatchQueue;
    private int mRetryCounter;

    private LptagRequest mLptagRequest = null;

    public LptagTask(AccountsController accountsController) {
        mAccountsController = accountsController;
    }

    /**
     * Execute this LptagTask
     * <p>
     * This task should not fail. In case we don't get a response we still return a success
     * <p>
     * LPTag can return in some cases
     */
    @Override
    public void execute() {
        LPMobileLog.d(TAG, "Running LPTag task...");

        AmsConnectionAnalytics.lptagTaskStart();

        // Initialize the retry mechanism
        mDispatchQueue = new DispatchQueue("Lptag");
        mRetryCounter = 0;

        List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);

        // Get the LPTag domain from Infra
        String domain = Infra.instance.getLptagEnvironment().getLptagDomain();

        mLptagRequest = new LptagRequest(domain, mBrandId, certificates, new ICallback<LptagData, Exception>() {
            @Override
            public void onSuccess(LptagData lptagData) {

                if (lptagData != null) { // If we got a retry response

                    if (lptagData.getMaxRetries() > 0) { // If need a retry
                        if (mRetryCounter < lptagData.getMaxRetries()) { // Check current retry

                            LPMobileLog.d(TAG, "onSuccess: Need retry. MaxRetries = " + lptagData.getMaxRetries() + ", current retry = " + mRetryCounter + ", RetryTimeout = " + lptagData.getRetryTimeout());
                            mRetryCounter++;
                            mDispatchQueue.postRunnable(new Runnable() {
                                @Override
                                public void run() {
                                    mLptagRequest.execute();
                                }
                            }, lptagData.getRetryTimeout());
                        } else { // Done with retries. Return error
                            LPMobileLog.d(TAG, "Done all retries but still didn't get reply. AutoMessages is off. Return success");
                            mCallback.onTaskSuccess();
                        }

                    } else { // LPTag data received
                        mAccountsController.getConnectionParamsCache(mBrandId).setAutoMessagesFeatureEnabled(lptagData.isAutoMessagesFeatureEnabled());
                        LPMobileLog.d(TAG, "onSuccess: Got AutoMessages feature: " + lptagData.isAutoMessagesFeatureEnabled());
                        AmsConnectionAnalytics.lptagTaskEnd();
                        mCallback.onTaskSuccess();
                    }
                } else { // No data received from LPTag, task success. Todo: what should we return here
                    LPMobileLog.d(TAG, "onSuccess: No AutoMessages feature received from LPTag");
                    AmsConnectionAnalytics.lptagTaskEnd();
                    mCallback.onTaskSuccess();
                }
            }

            @Override
            public void onError(Exception e) {
                AmsConnectionAnalytics.lptagTaskEnd();
                if (e instanceof SSLPeerUnverifiedException) {
                    mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, e);
                } else {
                    mCallback.onTaskError(TaskType.CSDS, e); // TODO: 9/7/17 Currently, we use TaskType.CSDS so we won't change the "API". In any case we intend to replace the CSDS call with LPTag in the future
                }
            }
        });

        // Execute the request
        mLptagRequest.execute();
    }

    @Override
    public String getName() {
        return TAG;
    }
}