package com.liveperson.messaging.controller;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.util.SparseArray;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.BuildConfig;
import com.liveperson.infra.messaging.R;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.TimeZone;

/**
 * Created by shiranr on 30/06/2016.
 */
public class ClientProperties implements Clearable {

	private static final String TAG = "ClientProperties";

	private static final String APP_ID_PREFERENCE_KEY = "APP_ID_PREFERENCE_KEY";
	private static final String SDK_VERSION_PREFERENCE_KEY = "SDK_VERSION_PREFERENCE_KEY";


	private static final String APP_ID = "appId";
	private static final String DEVICE_FAMILY = "deviceFamily";
	private static final String OS = "os";
	private static final String IP_ADDRESS = "ipAddress";
	private static final String OS_NAME = "osName";
	private static final String OS_VERSION = "osVersion";
	private static final String INTEGRATION = "integration";
	private static final String INTEGRATION_VERSION = "integrationVersion";
	//	private static final String BROWSER = "browser";
    //	private static final String BROWSER_VERSION = "browserVersion";
	private static final String TIME_ZONE = "timeZone";
	private static final String DEVICE_MANUFACTURER = "deviceManufacture"; // Incorrect spelling required
	private static final String DEVICE_MODEL = "deviceModel";
	private static final String APP_VERSION = "appVersion";
	private static final String FEATURES = "features";
	private static final String TYPE = "type";
	private static final String TYPE_VALUE = ".ClientProperties";


	private String mAppId;
	private String mSdkVersion;
	private final String ipAddress;

	private JSONObject mJson;

	public ClientProperties(String appId, String appVersion) {
		String _ipAddress = "unknown";
		try {
			_ipAddress = InetAddress.getLocalHost().getHostAddress();
		} catch (UnknownHostException e) {
			LPMobileLog.e(TAG, "Failed to get ip address, unknown host exception.", e);
		}

		ipAddress = _ipAddress;
		saveAppId(appId);
		saveSDKVersion(appVersion);
		createJson();

		LPMobileLog.v(TAG, getLoggablePropertyDump(
				Infra.instance.getApplicationContext(), "Created ClientProperties:", appId));
	}

	public ClientProperties() {
		this(null, null);
	}

	private void createJson() {
		mJson = new JSONObject();
		try {
			mJson.put(APP_ID, mAppId);
			mJson.put(APP_VERSION, getAppVersion(Infra.instance.getApplicationContext()));
			mJson.put(DEVICE_FAMILY, DeviceFamily.MOBILE.name());
			mJson.put(OS, OperatingSystem.ANDROID.name());
			mJson.put(OS_NAME, getOsName());
			mJson.put(OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
			mJson.put(DEVICE_MODEL, Build.MODEL);
			mJson.put(DEVICE_MANUFACTURER, Build.MANUFACTURER);
			mJson.put(INTEGRATION, Integration.MOBILE_SDK.name());
			mJson.put(INTEGRATION_VERSION, mSdkVersion);
			mJson.put(TYPE, TYPE_VALUE);

			// Build the features array
			JSONArray featuresArray = new JSONArray();
			featuresArray.put(Features.PHOTO_SHARING.name())
					.put(Features.CO_APP.name())
					.put(Features.RICH_CONTENT.name())
					.put(Features.SECURE_FORMS.name())
					.put(Features.AUTO_MESSAGES.name())
					.put(Features.QUICK_REPLIES.name())
					.put(Features.MULTI_DIALOG.name());

			if (Configuration.getBoolean(R.bool.enable_file_Sharing)) {
				featuresArray.put(Features.FILE_SHARING.name());
			}

			mJson.put(FEATURES, featuresArray);

			mJson.put(TIME_ZONE, TimeZone.getDefault().getID());

		} catch (JSONException e) {
			LPMobileLog.e(TAG, "JSONException while adding properties to JSON Object.", e);
		}
	}

	public String getJsonString() {
		return mJson.toString();
	}

	public void addToUriBuilder(Uri.Builder builder) {
		builder.appendQueryParameter(APP_ID, mAppId)
				.appendQueryParameter(APP_VERSION,  getAppVersion(Infra.instance.getApplicationContext()))
				.appendQueryParameter(DEVICE_FAMILY,  DeviceFamily.MOBILE.name())
				.appendQueryParameter(IP_ADDRESS, ipAddress)
				.appendQueryParameter(OS, OperatingSystem.ANDROID.name())
				.appendQueryParameter(OS_NAME, getOsName())
				.appendQueryParameter(OS_VERSION, String.valueOf(Build.VERSION.SDK_INT))
				.appendQueryParameter(DEVICE_MODEL, Build.MODEL)
				.appendQueryParameter(DEVICE_MANUFACTURER, Build.MANUFACTURER)
				.appendQueryParameter(INTEGRATION, Integration.MOBILE_SDK.name())
				.appendQueryParameter(INTEGRATION_VERSION, mSdkVersion)
				.appendQueryParameter(FEATURES, Features.PHOTO_SHARING.name() + "," +
						Features.CO_APP.name() + "," +
						Features.RICH_CONTENT.name() + "," +
						Features.SECURE_FORMS.name() + "," +
						Features.AUTO_MESSAGES.name() + "," +
						Features.QUICK_REPLIES.name() + "," +
						Features.MULTI_DIALOG.name() )

				.appendQueryParameter(TIME_ZONE, TimeZone.getDefault().getID());
	}

	public void saveAppId(String newAppId) {
		if (TextUtils.isEmpty(newAppId)) {
			mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
		} else {
			mAppId = newAppId;
			PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mAppId);
		}
	}

	public void saveSDKVersion(String newSdkVersion) {
		if (TextUtils.isEmpty(newSdkVersion)) {
			mSdkVersion = PreferenceManager.getInstance().getStringValue(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
		} else {

			final String oldSdkVersion = PreferenceManager.getInstance().getStringValue(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");

			if (!TextUtils.isEmpty(oldSdkVersion) && (!oldSdkVersion.equals(newSdkVersion))) {
				PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SDK_VERSION_CHANGED_CLEAR_DATABASE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
			}

			mSdkVersion = newSdkVersion;
			PreferenceManager.getInstance().setStringValue(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mSdkVersion);
		}
	}

	@Override
	public void clear() {
		PreferenceManager.getInstance().remove(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
		PreferenceManager.getInstance().remove(SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
		PreferenceManager.getInstance().remove(PreferenceManager.SDK_VERSION_CHANGED_CLEAR_DATABASE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
	}

	@NonNull
	private static String getAppVersion(@NonNull Context appContext) {
		try {
			PackageInfo packageInfo = appContext.getPackageManager().getPackageInfo(appContext.getPackageName(), 0);
			return packageInfo.versionName;
		} catch (PackageManager.NameNotFoundException nnfe) {
			LPMobileLog.w(TAG, "Could not find package for our own package name!", nnfe);
		} catch (NullPointerException npe) {
			LPMobileLog.w(TAG, "NPE while trying to fetch our app's version number!", npe);
		}
		return "";
	}

	@NonNull
	private static String getOsName() {
		AndroidVersion ver = AndroidVersion.fromSdkVersionCode(Build.VERSION.SDK_INT);
		if (ver != null) {
			return ver.logName;
		} else {
			return String.format("Unknown (%s)", Build.VERSION.RELEASE);
		}
	}

	@NonNull
	public static String getLoggablePropertyDump(@NonNull Context applicationContext, @NonNull String dumpTitle, @Nullable String appId) {
		return "\n========================================\n" +
				dumpTitle + "\n" +
				"----------------------------------------\n" +
				"Device Family:      " + DeviceFamily.MOBILE.name() + "\n" +
				"OS:                 " + OperatingSystem.ANDROID.name() + "\n" +
				"OS Name:            " + getOsName() + "\n" +
				"OS Version:         " + Build.VERSION.RELEASE+ "\n" +
				"OS Codename:        " + Build.VERSION.CODENAME + "\n" +
				"API Version:        " + Build.VERSION.SDK_INT + "\n" +
				"Device Model:       " + Build.MODEL + "\n" +
				"Device Maker:       " + Build.MANUFACTURER + "\n" +
				(appId != null ? "Host App ID:        " + appId + "\n" : "") +
				"Host App Version:   " + getAppVersion(applicationContext) + "\n" +
				"LP Client:          " + Integration.MOBILE_SDK.name() + "\n" +
				"LP Client Version:  " + BuildConfig.VERSION_NAME + "\n" +
				"========================================";
	}

	/* https://lpgithub.dev.lprnd.net/Architecture/api-ums/blob/314b3b72eaf0c276a35118414c79ad16b4ad7c6a/3.0.0.9/scheme.json
	  "appId" : "appIdExample",
	  "ipAddress" : "127.321.22.1",
	  "deviceFamily" : "MOBILE_PHONE",
	  "os" : "ANDROID",
	  "osVersion" : "18",
	  "integration" : "MOBILE_SDK",
	  "integrationVersion" : "1.2",
	  "browser" : "OTHER",
	  "browserVersion" : "",
	  "timeZone" : "Europe/Athens"
	  */
	private enum DeviceFamily {
		MOBILE,
		TABLET,
		PERSONAL_COMPUTER,
		PDA,
		OTHER,
		UNKNOWN
	}

	private enum OperatingSystem {
		WINDOWS,
		LINUX,
		OSX,
		ANDROID,
		IOS,
		OTHER
	}

	private enum AndroidVersion {
		BASE(1, "Base (1.0)"),
		BASE_1_1(2,"Base (1.1)"),
		CUPCAKE(3, "Cupcake (1.5)"),
		DONUT(4, "Donut (1.6)"),
		ECLAIR(5, "Eclair (2.0)"),
		ECLAIR_0_1(6, "Eclair (2.0.1)"),
		ECLAIR_MR1(7, "Eclair (2.1)"),
		FROYO(8, "Froyo (2.2)"),
		GINGERBREAD(9, "Gingerbread (2.3)"),
		GINGERBREAD_MR1(10, "Gingerbread (2.3.3)"),
		HONEYCOMB(11, "Honeycomb (3.0)"),
		HONEYCOMB_MR1(12, "Honeycomb (3.1)"),
		HONEYCOMB_MR2(13, "Honeycomb (3.2)"),
		ICE_CREAM_SANDWICH(14, "Ice Cream Sandwich (4.0)"),
		ICE_CREAM_SANDWICH_MR1(15, "Ice Cream Sandwich (4.0.3)"),
		JELLY_BEAN(16, "Jelly Bean (4.1)"),
		JELLY_BEAN_MR1(17, "Jelly Bean (4.2)"),
		JELLY_BEAN_MR2(18, "Jelly Bean (4.3)"),
		KITKAT(19, "Kitkat (4.4)"),
		KITKAT_WATCH(20, "Kitkat Watch (4.4W)"),
		LOLLIPOP(21, "Lollipop (5.0)"),
		LOLLIPOP_MR1(22, "Lollipop (5.1)"),
		MARSHMALLOW(23, "Marshmallow (6.0)"),
		NOUGAT(24, "Nougat (7.0)"),
		NOUGAT_MR1(25, "Nougat (7.1)"),
		OREO(26, "Oreo (8.0)"),
		OREO_MR1(27, "Oreo (8.1)"),
		PIE(28, "Pie (9.0)"),
		Q(29, "Q (10.0)");

		public final int verCode;
		@NonNull
		public final String logName;

		AndroidVersion(final int verCode, @NonNull final String logName) {
			this.verCode = verCode;
			this.logName = logName;
		}

		private static final SparseArray<AndroidVersion> verCodeMap = new SparseArray<>();

		static {
			for (AndroidVersion version : AndroidVersion.values()) {
				verCodeMap.put(version.verCode, version);
			}
		}

		public static AndroidVersion fromSdkVersionCode(final int verCode) {
			return verCodeMap.get(verCode);
		}
	}

	private enum Integration {
		/** In case of connecting via regular browser */
		WEB_SDK,

		/** In case of connecting via native application */
		MOBILE_SDK,

		/** In case of connecting via singe brand connection (GW) */
		BRAND_SDK
	}

	private enum Browser {
		CHROME,
		FIREFOX,
		IE,
		OPERA,
		SAFARI,
		OTHER
	}

	public enum Features {
		CO_BROWSE,
		CO_APP,
		PHOTO_SHARING,
		SECURE_FORMS,
		RICH_CONTENT,
		AUTO_MESSAGES,
		QUICK_REPLIES,
		MULTI_DIALOG,
		FILE_SHARING
	}
}
