package com.liveperson.messaging.controller;

import android.text.TextUtils;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;

/**
 * Created by nirni on 11/26/15.
 * <p/>
 * This controller is used to send on startup/view change a read ack on messages send by the agent
 * and a read ack was not sent yet.
 * It listens on two broadcasts: Foreground and Connection and only if a view is in the foreground
 * and there is an active connection we send the ack on the messages
 */
public class AmsReadController implements ShutDown {

    private static final String TAG = "AmsReadController";

    private LocalBroadcastReceiver mForegroundConnectionReceiver;
    private final Messaging mController;

    public AmsReadController(Messaging messagingController) {
        mController = messagingController;
    }

    /**
     * Register a broadcast receiver to monitor any views that go into foreground or background.
     * Since this is called from the connection command when connection is on, it first check if there
     * is anything in the foreground and sends the read ack.
     *
     * @param brandId
     */
    public void registerForegroundConnectionReceiver(String brandId) {

        // Send read ack if there is anything in the foreground
        onConnected(brandId);

        // Register the foreground receiver
        if (mForegroundConnectionReceiver == null) {
            mForegroundConnectionReceiver = new LocalBroadcastReceiver.Builder()
                    .addAction(ForegroundService.SCREEN_FOREGROUND_ACTION)
                    .build((context, intent) -> {

                        // Get the info from the intent
                        String brandId1 = intent.getStringExtra(ForegroundService.SCREEN_FOREGROUND_EXTRA_BRAND_ID);
                        String targetId = intent.getStringExtra(ForegroundService.SCREEN_FOREGROUND_EXTRA_TARGET_ID);
                        boolean isForeground = intent.getBooleanExtra(ForegroundService.SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND, false);

                        // If the view got in foreground we check if the connection for the given brand is on and send the ack
                        if (isForeground) {

                            // If the connection is on and it is on the same brand as the foreground view we send the read ack
                            if (mController.mConnectionController.isSocketReady(brandId1)){

	                            LPLog.INSTANCE.d(TAG, "Sending read ack for targetId " + targetId);
                                String originatorId = mController.getOriginatorId(targetId);
                                mController.amsMessages.sendReadAckOnMessages(brandId1, targetId, originatorId);
                            }
                        }
                    });
        } else {
            mForegroundConnectionReceiver.register();
        }
    }

    private void unregisterForegroundConnectionReceiver() {
        if (mForegroundConnectionReceiver != null){
            mForegroundConnectionReceiver.unregister();
            mForegroundConnectionReceiver = null;
        }
    }

    /**
     * Validate that the connection is on and that there is a view in the foreground and sent the read ack
     *
     * @param brandId
     */
    private void onConnected(String brandId) {

        if(!TextUtils.isEmpty(brandId)
                && mController.mConnectionController.isSocketReady(brandId)
                && ForegroundService.getInstance().isBrandForeground(brandId)) {

            LPLog.INSTANCE.d(TAG, " (onConnected) Sending read ack for brandId " + brandId);
            String originatorId = MessagingFactory.getInstance().getController().getOriginatorId(brandId); // TODO: Perry: "Why not using 'mController.getOriginatorId(brandId);'?"
            mController.amsMessages.sendReadAckOnMessages(brandId, null, originatorId);
        }
    }

    @Override
    public void shutDown() {
        unregisterForegroundConnectionReceiver();
    }
}
