package com.liveperson.messaging.model;

import android.database.Cursor;
import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.database.tables.FilesTable;

import java.util.Objects;

/**
 * Created by shiranr on 30/06/2016.
 */
public class FileMessage {

    public static final String EXTRA_LOCAL_URL = "EXTRA_LOCAL_URL";
    public static final String EXTRA_PREVIEW = "EXTRA_PREVIEW" ;
    public static final String EXTRA_LOAD_STATUS = "EXTRA_LOAD_STATUS";
    public static final String EXTRA_FULL_IMAGE_EXISTS = "EXTRA_FULL_IMAGE_EXISTS";
    public static final String EXTRA_PREVIEW_IMAGE_EXISTS = "EXTRA_PREVIEW_IMAGE_EXISTS";
    public static final String EXTRA_FILE_ROW_ID = "EXTRA_FILE_ROW_ID";
    public static final String EXTRA_FILE_TYPE = "EXTRA_FILE_TYPE";
    private long mFileRowId;
    private String mPreview;
    private String mSwiftPath;
    private String mFileType;
    private String mLocalUrl;
    private long mRelatedMessageRowID;
    private FilesTable.LoadStatus mLoadStatus;

    public FileMessage(String preview, String fileType, String localUrl, String swiftPath, long relatedMessageRowID) {
        this.mPreview = preview;
        this.mFileType = fileType;
        this.mLocalUrl = localUrl;
        this.mLoadStatus = FilesTable.LoadStatus.NOT_STARTED;
        this.mRelatedMessageRowID = relatedMessageRowID;
        this.mSwiftPath = swiftPath;
    }

    public FileMessage(
            FileMessage delegated,
            long localFileRowId
    ) {
        this(
                delegated.getPreview(),
                delegated.getFileType(),
                delegated.getLocalUrl(),
                delegated.getSwiftPath(),
                delegated.getRelatedMessageRowID()
        );
        this.mFileRowId = localFileRowId;
    }

	private FileMessage(Cursor cursor) {
        int columnIndex = cursor.getColumnIndex(FilesTable.KEY_ID_AS_VALUE);
        if (columnIndex != -1){
            mFileRowId = cursor.getLong(columnIndex);
        }else{
            mFileRowId = cursor.getLong(cursor.getColumnIndex(FilesTable.KEY_ID));
        }
        mRelatedMessageRowID = cursor.getInt(cursor.getColumnIndex(FilesTable.KEY_RELATED_MESSAGE_ROW_ID));
		mPreview = cursor.getString(cursor.getColumnIndex(FilesTable.KEY_PREVIEW));
		mFileType = cursor.getString(cursor.getColumnIndex(FilesTable.KEY_FILE_TYPE));
		mLocalUrl = cursor.getString(cursor.getColumnIndex(FilesTable.KEY_LOCAL_URL));
		mLoadStatus = FilesTable.LoadStatus.values()[cursor.getInt(cursor.getColumnIndex(FilesTable.KEY_LOAD_STATUS))];
		mSwiftPath = cursor.getString(cursor.getColumnIndex(FilesTable.KEY_SWIFT_PATH));
	}

	public static FileMessage fromCursor(Cursor cursor) {
		if (cursor == null){
			return null;
		}

		int relatedMessageRowIDColumn = cursor.getColumnIndex(FilesTable.KEY_RELATED_MESSAGE_ROW_ID);
		if (relatedMessageRowIDColumn == -1){
			return null;
		}

		return new FileMessage(cursor);
	}

	public String getPreview() {
        return mPreview;
    }

    public String getSwiftPath() {
        return mSwiftPath;
    }

    public String getFileType() {
        return mFileType;
    }

    public String getLocalUrl() {
        return mLocalUrl;
    }

    public long getFileRowId() {
        return mFileRowId;
    }

    public long getRelatedMessageRowID() {
        return mRelatedMessageRowID;
    }

    public  FilesTable.LoadStatus getLoadStatus() {
        return mLoadStatus;
    }


    /**
     * Update file message and get Diff
     * @param newFileMessage
     * @return
     */
    public Bundle update(FileMessage newFileMessage) {
        Bundle extras = new Bundle();
        if (newFileMessage != null) {

            if (mFileRowId != newFileMessage.getFileRowId()) {
                mFileRowId = newFileMessage.getFileRowId();
                extras.putLong(EXTRA_FILE_ROW_ID, mFileRowId);
            }

            boolean imageChanges = false;
            if (!TextUtils.equals(mPreview, newFileMessage.getPreview())) {
                mPreview = newFileMessage.getPreview();
                imageChanges = true;
            }
            extras.putString(EXTRA_PREVIEW, mPreview);

            if (!TextUtils.equals(mLocalUrl, newFileMessage.getLocalUrl())) {
                mLocalUrl = newFileMessage.getLocalUrl();
                imageChanges = true;
            }
            extras.putString(EXTRA_LOCAL_URL, mLocalUrl);

            if (mLoadStatus != newFileMessage.getLoadStatus()) {
                mLoadStatus = newFileMessage.getLoadStatus();
                extras.putInt(EXTRA_LOAD_STATUS, mLoadStatus.ordinal());

            }

            if (!TextUtils.equals(mSwiftPath, newFileMessage.getSwiftPath())) {
                mSwiftPath = newFileMessage.getSwiftPath();
            }

            if (mFileType != null) {
                extras.putString(EXTRA_FILE_TYPE, mFileType);
            }


            //if there were changed (!extras.isEmpty()) but no changes in the preview url or the local url (!imageChanges)
            //we should notify the ui if the current state is downloading/history flow or we are on uploading flow.
            if (!extras.isEmpty() && !imageChanges) {
                if (!TextUtils.isEmpty(mLocalUrl)) {
                    extras.putBoolean(EXTRA_FULL_IMAGE_EXISTS, true);
                } else if (!TextUtils.isEmpty(mPreview)) {
                    extras.putBoolean(EXTRA_PREVIEW_IMAGE_EXISTS, true);
                }
            }

        }
        return extras;
    }

    public Bundle getAllFileChanges() {
        Bundle extras = new Bundle();
        extras.putString(EXTRA_PREVIEW, mPreview);
        extras.putString(EXTRA_LOCAL_URL, mLocalUrl);
        extras.putLong(EXTRA_FILE_ROW_ID, mFileRowId);
        if (!TextUtils.isEmpty(mLocalUrl)) {
            extras.putBoolean(EXTRA_FULL_IMAGE_EXISTS, true);
        }else if (!TextUtils.isEmpty(mPreview)) {
            extras.putBoolean(EXTRA_PREVIEW_IMAGE_EXISTS, true);
        }
        extras.putInt(EXTRA_LOAD_STATUS, mLoadStatus.ordinal());
        return extras;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        FileMessage that = (FileMessage) o;
        return mFileRowId == that.mFileRowId
                && mRelatedMessageRowID == that.mRelatedMessageRowID
                && Objects.equals(mPreview, that.mPreview)
                && Objects.equals(mSwiftPath, that.mSwiftPath)
                && Objects.equals(mFileType, that.mFileType)
                && Objects.equals(mLocalUrl, that.mLocalUrl)
                && mLoadStatus == that.mLoadStatus;
    }

    @Override
    public int hashCode() {
        return Objects.hash(mFileRowId, mPreview, mSwiftPath, mFileType, mLocalUrl, mRelatedMessageRowID, mLoadStatus);
    }
}
