package com.liveperson.messaging.network.http;

import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.liveperson.api.request.QueryMessages;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.infra.utils.SdkConst;
import com.liveperson.infra.utils.Synchronizer;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.SocketTaskType;
import com.liveperson.messaging.commands.BasicQueryMessagesCommand;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.Dialog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000CB;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000CC;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000CD;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000CE;

/**
 * Created by shiranr on 14/01/2016.
 */
public class QueryMessagesINCACommand extends BasicQueryMessagesCommand {

    private static final String TAG = "QueryMessagesINCACommand";

    //https://<domain>/messaging_history/api/account/{accountID}/conversations/conversation/content/
    // search?conversationId=4a6ce154-a086-4bfb-8ab2-2658fd88157


    private static final String INCA_GET_MESSAGES_URL = "https://%s/messaging_history/api/account/%s/conversations/conversation/content/%s";
    private static final String SEARCH_QUERY = "search";
    private static final int REQUEST_TIMEOUT = 30000;


    public QueryMessagesINCACommand(Messaging controller, String brandId, String conversationId, String dialogId, boolean updateUI) {
        super(controller, brandId, conversationId, dialogId, updateUI);
    }

    @Override
    public void execute() {

        String incaDomain = controller.mAccountsController.getServiceUrl(brandID, ConnectionParamsCache.CSDS_INCA_KEY);

        if (!TextUtils.isEmpty(incaDomain)) {
            String historyUrl = String.format(INCA_GET_MESSAGES_URL, incaDomain, brandID, SEARCH_QUERY);

            Uri uri = Uri.parse(historyUrl).buildUpon()
                    .appendQueryParameter("conversationId", conversationId)
                    .appendQueryParameter("source", SdkConst.SOURCE_NAME)
                    .build();

            LPLog.INSTANCE.d(TAG, "Getting inca messages url " + uri.toString());
            HttpGetRequest httpGetRequest = new HttpGetRequest(uri.toString());
            //["Content-Type": "application/json", "Authorization": "Bearer eyJraWQiOiIwMDAwMSIsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2In0.eyJzdWIiOiJhNDRiOGZiNWU1NjVkZWE5MTZlMWYzYmQ1MGNkMGU1ZmMzY2Q5OTJmMzhjZjc3ZGEwYTM5NjZkMzk5YTVkMTFiIiwiYXVkIjoiYWNjOjM2MzY2Nzk2IiwiaXNzIjoiaHR0cHM6XC9cL2lkcC5saXZlcGVyc29uLm5ldCIsImxwLmV4dCI6eyJzdWIiOiJscFRlc3R3b3ciLCJpc3MiOiJodHRwczpcL1wvY3VzdG9tZXJXZWJTaXRlLmNvbSIsImxwX3NkZXMiOiJlMmY3OTU4YmI2NTdkYWM4YTJjY2IwYTU4Y2I0ZGRiNDVkZmJiZmYzODQzMzVkYmRiYzAwY2Y1YzZiYWVmY2UyNTMyYjViY2M1MTU1N2IxZThlNWUxNzExNTg0NmY4Zjk3YWFhNmM0ZTM0ZDQwNjE4ZjJmMTgzZTg5M2EzZTFlN2YwYmU5N2VhNjA3MzI1YWE2NzY0NzZmMmYzNDhjYjVkZWNmZTIyYzMzYzQ0Y2JjZDFhYTBkMTU4ZTViZTFkYjIwNDU3MDQ5NWE5ZmIwNjQ0NmU0Zjk1MThmZDBiYTg4MCJ9LCJleHAiOjE1MTg2ODI4NTgsImlhdCI6MTUxNjA5MDg1OH0.OacsjIdHThLrsn6df1OdlSGqeSCDEtVOwFQOXHs-aqfDuSOQfjhPK95h33rKKDZaxBSqXJi23arCNaoVNRgF9PZna4oIrjqtRRqPPzTVm0sg0JNJKLtSRhgFGPEJOa2BnYaZlRROn5xLtglqAFWt5rTNZ9G4D4r3ew94ZXuhZ8ILEjzAH3_bG4rY1pRrkdhM9o_5m045Oslb4Tj11viY6l3U07O9qfPa3DUdyax4-ghCnZy5fmb0AFumPOubFRTRYf9WxsVeIA-llwXxpHX22et-o-riAtT5ENBFS72LzHwcExU4gf7ipp3BR4S1Lm6J3NCFrmscmqAb3PhLIPvKUw"]
            AmsAccount account = controller.mAccountsController.getAccount(brandID);
            String token = account == null ? "" : account.getToken();
            httpGetRequest.addHeader("Authorization", "Bearer "+ token);
            httpGetRequest.setCertificatePinningKeys(controller.mAccountsController.getCertificatePinningKeys(brandID));
            httpGetRequest.setTimeout(REQUEST_TIMEOUT);
            httpGetRequest.setCallback( new ICallback<String, Exception>() {
                @Override
                public void onSuccess(String res) {
                    LPLog.INSTANCE.d(TAG, "onSuccess with INCA history response details " + res);
                    if (TextUtils.isEmpty(res)) {
                        return;
                    }
                    try {
                        JSONObject jsonObject = new JSONObject(res);

                        JSONArray array = jsonObject.optJSONArray("messageEventRecords");
                        final QueryMessages.Response response = new QueryMessages.Response(array);

                        final ArrayList<ContentEventNotification> messagesResult = response.getBody();

                        saveMessagesResults(messagesResult);
                    } catch (JSONException e) {
                        LPLog.INSTANCE.e(TAG, ERR_000000CB, "JSONException while parsing " + res, e);
                    }

                }

                @Override
                public void onError(Exception exception) {
                    LPLog.INSTANCE.e(TAG, ERR_000000CC, "Exception", exception);
                    onHandleResponseCompletedFailed();
                }
            });

            HttpHandler.execute(httpGetRequest);

            onRequestSent();
        }
    }

    private void saveMessagesResults(final ArrayList<ContentEventNotification> messagesResult) {
        Map<String, ArrayList<ContentEventNotification>> messagesByDialogIdMap = new HashMap<>();
        messagesByDialogIdMap.put(dialogId, new ArrayList<>());
        // Splitting the array to multiple arrays that are divided by dialogId
        for (ContentEventNotification notification : messagesResult) {

            // The dialog IDs should be inside the result
            if (TextUtils.isEmpty(notification.dialogId)) {
                LPLog.INSTANCE.e(TAG, FlowTags.DIALOGS, ERR_000000CD, "Dialog ID of message is empty after fetching from INCA");
                notification.dialogId = conversationId;
            }

            if (!messagesByDialogIdMap.containsKey(notification.dialogId)) {
                messagesByDialogIdMap.put(notification.dialogId, new ArrayList<>());
            }
            //noinspection ConstantConditions
            messagesByDialogIdMap.get(notification.dialogId).add(notification);
        }

        final Synchronizer<Boolean> synchronizer = new Synchronizer<>(extra -> {
            boolean allDoneSuccessfully = true;
            for (Boolean doneSuccessfully : extra) {
                if (doneSuccessfully != null) {
                    allDoneSuccessfully &= doneSuccessfully;
                }
            }

            if (allDoneSuccessfully) {
                onHandleResponseCompletedSuccessfully();
            } else {
                onHandleResponseCompletedFailed();
            }
        });

        for (Map.Entry<String, ArrayList<ContentEventNotification>> entry : messagesByDialogIdMap.entrySet()) {
            final Synchronizer<Boolean>.Holder holder = synchronizer.createHolder();

            if (entry.getValue().isEmpty()) {
                LPLog.INSTANCE.e(TAG, FlowTags.DIALOGS, ERR_000000CE, "Received empty messages list.");
            }

            controller.amsDialogs.saveMessagesResult(entry.getKey(), entry.getValue(), true, mUpdateUI, new ICallback<Dialog, Exception>() {
                @Override
                public void onSuccess(Dialog value) {
                    holder.release(true);
                }

                @Override
                public void onError(Exception exception) {
                    holder.release(false);
                }
            });
        }
    }

    private void onRequestSent() {
        if (!TextUtils.isEmpty(conversationId)){
            controller.amsConversations.addUpdateRequestInProgress(conversationId);
        }

    }

    private void onRequestCompleted() {
        if (!TextUtils.isEmpty(conversationId)){
            controller.amsConversations.removeUpdateRequestInProgress(conversationId);
        }
    }


    private void onHandleResponseCompletedFailed() {
        onRequestCompleted();
        if (mResponseCallBack != null){
            mResponseCallBack.onTaskError(SocketTaskType.QUERY_MESSAGES, new Exception("unexpected QueryMessages"));
        }
    }

    private void onHandleResponseCompletedSuccessfully() {
        onRequestCompleted();
        if (mResponseCallBack != null){
            mResponseCallBack.onTaskSuccess();
        }
    }
}
