package com.liveperson.messaging.background;

import android.net.Uri;

import com.liveperson.api.response.BaseGenerateURLResponse;
import com.liveperson.api.response.IOnUrlReady;
import com.liveperson.api.response.model.QueryParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.filesharing.DownloadFileTaskBundle;
import com.liveperson.messaging.network.http.DownloadFileRequest;
import com.liveperson.messaging.network.http.IncaGetFileUrlRequest;
import com.liveperson.messaging.network.socket.requests.GetUrlForDownloadRequest;

/**
 * Created by shiranr on 7/24/16.
 */
public abstract class DownloadFileTask {

    private static final String TAG = "DownloadFileTask";

    private DownloadFileTaskCallback mDownloadFileTaskCallback;
    protected DownloadFileTaskBundle mDownloadFileTaskParams;
    private String mRelativePath;
    private FilesTable.LoadStatus mStatus = FilesTable.LoadStatus.NOT_STARTED;
    private long mFileRowId;

	/**
	 * Save the file to disk and return its path
	 * @return
	 */
	protected abstract String saveFileToDisk(byte[] byteArray);

    /**
     * creating bitmaps (thumbnail and full size)
     * @param params
     */
    public DownloadFileTask(DownloadFileTaskBundle params) {
        mDownloadFileTaskParams = params;
        mFileRowId = mDownloadFileTaskParams.getFileRowId();
    }

    /**
     * Start downloading task.
     * Task will first get the file url from the relevant end point and then will download the actual file.
     * @param fromInca true - get file url from INCA, false - get url from UMS.
     */
    public void startDownload(boolean fromInca) {
        if (fromInca) {
            getFileUrlFromInca();

        } else {
            getFileUrlFromUMS();
        }
    }


    /**
     * Request file url from INCA
     */
    private void getFileUrlFromInca() {
        setStatus(FilesTable.LoadStatus.REQUESTING_URL);

        // The file ID as is after the first ‘/‘ from the relativePath we receive from INCA
        String relativePath = mDownloadFileTaskParams.getRelativePath();
        if (relativePath != null && !relativePath.isEmpty()){
            if (relativePath.contains("/")) {
                relativePath = relativePath.substring(relativePath.lastIndexOf("/") + 1);
            }
        } else {
            setStatus(FilesTable.LoadStatus.FAILED);
            return;
        }

        new IncaGetFileUrlRequest(MessagingFactory.getInstance().getController(),
                mDownloadFileTaskParams.getBrandId(),
                mDownloadFileTaskParams.getConversationId(),
                relativePath,
                new ICallback<Uri, Exception>() {
                    @Override
                    public void onSuccess(Uri uri) {
                        mRelativePath = uri.getPath();
                        MessagingFactory.getInstance().getController().amsFiles.updateRelativePath(mFileRowId, mRelativePath);

                        if (isDownloadFailed()) {
                            return;
                        }

                        QueryParams queryParams = new QueryParams(uri);

                        downloadFromSwift(queryParams);
                    }

                    @Override
                    public void onError(Exception exception) {
                        if (isDownloadFailed()) {
                            return;
                        }
                        onDownloadFailed(new Exception("failed to generate url. " + exception.toString()));
                    }
                }).execute();
    }

    public void onConnectionAvailable() {
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "onConnectionAvailable");
        if (mStatus == FilesTable.LoadStatus.NOT_STARTED){
            setStatus(FilesTable.LoadStatus.PROCESSING);
            mDownloadFileTaskCallback.onReadyToGetUrl();
        }
    }

    public void setCallBack(DownloadFileTaskCallback downloadFileTaskCallback) {
        mDownloadFileTaskCallback = downloadFileTaskCallback;
    }

    /**
     * Request file url from UMS
     * */
    private void getFileUrlFromUMS() {
        setStatus(FilesTable.LoadStatus.REQUESTING_URL);

        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "generateUrlRunnable");
        SocketManager.getInstance().send(new GetUrlForDownloadRequest(MessagingFactory.getInstance().getController(),
                mDownloadFileTaskParams.getBrandId(), mDownloadFileTaskParams.getRelativePath(),
                new IOnUrlReady() {
                    @Override
                    public void onUrlReady(final BaseGenerateURLResponse body) {
                        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "URL ready!!" + body.relativePath );
                        //updating file relativePath on DB
                        mRelativePath = body.relativePath;
                        MessagingFactory.getInstance().getController().amsFiles.updateRelativePath(mFileRowId, mRelativePath);

                        if (isDownloadFailed()){
                            return;
                        }
                        downloadFromSwift(body.queryParams);
                    }

                    @Override
                    public void onUrlError(String errorMessage) {
                        if (isDownloadFailed()){
                            return;
                        }
                        onDownloadFailed(new Exception("failed to generate url." + errorMessage));
                    }
                }));
    }

    private void downloadFromSwift(QueryParams queryParams) {
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "Downloading from swift..");

        setStatus(FilesTable.LoadStatus.DOWNLOADING);

        DownloadFileRequest mDownloadFileRequest = new DownloadFileRequest(mDownloadFileTaskParams.getSwiftDomain(), mRelativePath, queryParams, mDownloadFileTaskParams.getRestParams().mCertificates, new ICallback<byte[], Exception>() {
            @Override
            public void onSuccess(byte[] byteArray) {
                if (isDownloadFailed()) {
                    return;
                }
                LPLog.INSTANCE.d(TAG + Thread.currentThread(), "Downloading from swift succeeded! saving file");

				final String path = saveFileToDisk(byteArray);
                MessagingFactory.getInstance().getController().amsFiles.updateLocalPath(mFileRowId, path)
                        .setPostQueryOnBackground(data -> onDownloadSucceed(path)).execute();
            }

            @Override
            public void onError(Exception exception) {
                if (isDownloadFailed()) {
                    return;
                }
                LPLog.INSTANCE.w(TAG, "failed to Download from swift ", exception);
                onDownloadFailed(exception);
            }
        });
        mDownloadFileRequest.execute();
    }

    private void notifyChange() {
        MessagingFactory.getInstance().getController().amsMessages.updateFileMessageByRowId(
                mDownloadFileTaskParams.getMessageRowId(), mDownloadFileTaskParams.getFileRowId()).execute();
    }

    private void onDownloadSucceed(String fullImageLocalPath) {
        setStatus(FilesTable.LoadStatus.COMPLETED);
        if (mDownloadFileTaskCallback != null) {
            mDownloadFileTaskCallback.onDownloadFinishedSuccessfully(fullImageLocalPath);
        }
    }

    public void onConnectionUnavailable() {
        onDownloadFailed(new Exception("Failed to Download. connection unavailable"));
    }
    private void onDownloadFailed(Throwable exception) {
        LPLog.INSTANCE.d(TAG , "onDownloadFailed. ", exception);
        setStatus(FilesTable.LoadStatus.FAILED);

        if (mDownloadFileTaskCallback != null) {
            mDownloadFileTaskCallback.onDownloadFailed(this, exception);
        }
    }

    public void setStatus(FilesTable.LoadStatus status) {
        this.mStatus = status;
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "set file status: "+ status +" mFileRowId = " + mFileRowId);
        if (mFileRowId != -1){
            MessagingFactory.getInstance()
                    .getController()
                    .amsFiles
                    .updateStatus(mFileRowId, mStatus, data -> {
                        notifyChange();
                    });
        } else {
            notifyChange();
        }
    }

    private boolean isDownloadFailed() {
        return mStatus == FilesTable.LoadStatus.FAILED;
    }
}
