package com.liveperson.messaging.commands.tasks;

import android.text.TextUtils;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.SiteSettingsRequest;
import com.liveperson.infra.utils.VersionUtils;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;
import com.liveperson.messaging.model.AmsConnectionAnalytics;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 12/29/15.
 * This is a task class used to fetch site settings for specific site.
 * It stores relevant settings in shared preference to be used later.
 */
public class SiteSettingsFetcherTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "SiteSettingsFetcherTask";

	// JSON property keys
    public static final String SITE_SETTINGS_PROPERTY_NAME_ID = "id";
    public static final String SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE = "propertyValue";
    public static final String SITE_SETTINGS_PROPERTY_NAME_VALUE = "value";

	// ID of the image sharing in the JSON (todo nn: check this with Ariel)
    public static final String SITE_SETTINGS_PHOTO_SHARING_ID_VALUE = "messaging.file.sharing.enabled";
	// ID of the voice sharing in the JSON
	public static final String SITE_SETTINGS_AUDIO_SHARING_ID_VALUE = "messaging.audio.sharing.enabled";
	// ID of the min version in the JSON
	public static final String SITE_SETTINGS_MIN_VERSION_ID_VALUE = "messaging.android.sdk.min.version";
	// ID of sdk analytics flag
	public static final String SITE_SETTINGS_SDK_ANALYTICS_ID_VALUE = "messaging.android.sdk.analytics.enabled";
	// ID of event manager domain
	public static final String SITE_SETTINGS_EVENT_MANAGER_DOMAIN_ID_VALUE = "messaging.sdk.event.manager.domain";
	// ID controller bot
	public static final String SITE_SETTINGS_CONTROLLER_BOT_ID_VALUE = "le.site.cobrowse.controllerBotPreferredForSystemMessages";
	// ID to add thumbnail blur effect
	public static final String SITE_SETTINGS_FILE_SHARING_BLUR_THUMBNAILS_ID_VALUE = "messaging.file.sharing.blurThumbnails";

    private final AccountsController mAccountsController;
	private boolean mPhotoSharingEnabled = true; // Default is true (in case we don't get the value from remote configuration)
	private boolean mVoiceSharingEnabled = true; // Default is true (in case we don't get the value from remote configuration)
	private boolean mSDKAnalyticsEnabled = true; // Default is true (in case we don't get the value from remote configuration)
	private boolean enableTranscendentMessages = true; // Default is true (in case we don't get the value from remote configuration)
	private boolean enableThumbnailBlur = false; // Default is false (in case we don't get the value from remote configuration)

	private String mMinSdkVersionFromServer;
	private String mEventManagerDomain;

    public SiteSettingsFetcherTask(AccountsController accountsController) {
        mAccountsController = accountsController;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.d(TAG, "Running site settings check task...");

		AmsConnectionAnalytics.siteSettingsFetcherTaskStart();

		// Get the AC-CDN domain
        final String siteSettingsServiceDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_AC_CDN_DOMAIN_KEY);
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);

		// Create and send the request
        new SiteSettingsRequest(siteSettingsServiceDomain, mBrandId, certificates, new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String siteSettingsJsonString) {

				if (!TextUtils.isEmpty(siteSettingsJsonString)) {

					// Reset the minSdkVersionFromServer
					mMinSdkVersionFromServer = null;

					// Parse json and get site settings' enable photo sharing
					boolean success = parseConfigurationData(siteSettingsJsonString);

					if (success) {
						LPLog.INSTANCE.d(TAG, "onSuccess: Got photo sharing enable value from site settings: " + mPhotoSharingEnabled);
					}
					else {
						LPLog.INSTANCE.w(TAG, "onSuccess: Cannot get photo sharing enable value from site settings. Use default (true)");
					}

					// SiteSettings kill switch
					// Checking the version only if we succeeded in getting it from the server
					if(mMinSdkVersionFromServer != null) {

						LPLog.INSTANCE.d(TAG, "onSuccess: Checking SDK version against version from site-settings");

						// Write server version to persistent memory
						final IConnectionParamsCache cache = mAccountsController.getConnectionParamsCache(mBrandId);
						if (cache != null) {
							cache.updateAcCdnVersion(mMinSdkVersionFromServer);
						} else  {
							String message = "onSuccess: " +
									"did not get connection cache params " +
									"for brand with id: " + mBrandId;
							LPLog.INSTANCE.d(TAG, message);
						}


						// If the sdk version is smaller than the version in the server we abort
						if (!VersionUtils.isValidSdkVersion(mMinSdkVersionFromServer)) {

							// Send an error
							String detailMessage = "Current SDK version is smaller than the one from the server (" + mMinSdkVersionFromServer + "). SDK will not connect. Please upgrade SDK. ";
							mCallback.onTaskError(TaskType.VERSION, LpError.INVALID_SDK_VERSION, new Exception(detailMessage));
							return;
						}
					}
					else {
						LPLog.INSTANCE.w(TAG, "onSuccess: did not get min SDK version from site-settings. Ignore and continue as usual");
					}

					// Succeeded in getting configuration
					AmsConnectionAnalytics.siteSettingsFetcherTaskEnd();
					mCallback.onTaskSuccess();
				}
            }

            @Override
            public void onError(Exception exception) {
				if(exception instanceof SSLPeerUnverifiedException) {
					mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, LpError.INVALID_CERTIFICATE, exception);
				}else {
					// Currently, if we cannot get to the configuration server we don't fail
					mCallback.onTaskSuccess();
				}
            }
        }).execute();

    }

	/**
	 * Get configuration from the given json string and store it in shared preferences
	 * @param siteSettingsJsonString
	 * @return true - if success in getting values from site settings, false - otherwise
	 */
	private boolean parseConfigurationData(String siteSettingsJsonString) {

		try {
			// We're default mPhotoSharingEnabled to true since we want it to be on in any case unless it
			// exist in site settings AND it's false
			mPhotoSharingEnabled = true;
			JSONArray jsonarray = new JSONArray(siteSettingsJsonString);
			for (int i = 0; i < jsonarray.length(); i++) {
				// Get next item
				JSONObject jsonobject = jsonarray.getJSONObject(i);
				String id = jsonobject.getString(SITE_SETTINGS_PROPERTY_NAME_ID);

				// If the current object is the photo sharing setting
				if (id.equalsIgnoreCase(SITE_SETTINGS_PHOTO_SHARING_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					String enabled = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					// If value is false we sent the enable flag to false, otherwise it'll use the default (true)
					if (enabled.equalsIgnoreCase("false")) {
						mPhotoSharingEnabled = false;
						LPLog.INSTANCE.d(TAG, "parseConfigurationData: photo sharing is disabled in SiteSettings");
					}
				}
				// If the current object is the photo sharing setting
				else if (id.equalsIgnoreCase(SITE_SETTINGS_AUDIO_SHARING_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					String audioEnabled = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					// If value is false we sent the enable flag to false, otherwise it'll use the default (true)
					if (audioEnabled.equalsIgnoreCase("false")) {
						mVoiceSharingEnabled = false;
						LPLog.INSTANCE.d(TAG, "parseConfigurationData: audio sharing is disabled in SiteSettings");
					}
				}
				else if (id.equalsIgnoreCase(SITE_SETTINGS_MIN_VERSION_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					mMinSdkVersionFromServer = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					LPLog.INSTANCE.d(TAG, "onSuccess: minSdkVersion from site-settings: " + mMinSdkVersionFromServer);
				}
				else if (id.equalsIgnoreCase(SITE_SETTINGS_SDK_ANALYTICS_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					String analyticsEnabled = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					// If value is false we sent the enable flag to false, otherwise it'll use the default (true)
					if (analyticsEnabled.equalsIgnoreCase("false")) {
						mSDKAnalyticsEnabled = false;
						LPLog.INSTANCE.d(TAG, "parseConfigurationData: sdk analytics is disabled in SiteSettings");
					}
				}
				else if (id.equalsIgnoreCase(SITE_SETTINGS_EVENT_MANAGER_DOMAIN_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					mEventManagerDomain = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
				}
				else if (id.equalsIgnoreCase(SITE_SETTINGS_CONTROLLER_BOT_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					String controllerBotEnabled = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					enableTranscendentMessages = controllerBotEnabled.equalsIgnoreCase("false");
					LPLog.INSTANCE.d(TAG, "enableTranscendentMessages: " + enableTranscendentMessages);
				}
				else if (id.equalsIgnoreCase(SITE_SETTINGS_FILE_SHARING_BLUR_THUMBNAILS_ID_VALUE)) {
					JSONObject propertyValue = jsonobject.getJSONObject(SITE_SETTINGS_PROPERTY_NAME_PROPERTY_VALUE);
					String blurThumbnail = propertyValue.getString(SITE_SETTINGS_PROPERTY_NAME_VALUE);
					// If value is true we sent the enable flag to true, otherwise it'll use the default (false)
					if (blurThumbnail.equalsIgnoreCase("true")) {
						enableThumbnailBlur = true;
						LPLog.INSTANCE.d(TAG, "parseConfigurationData: File sharing thumbnail blur is enabled in SiteSettings");
					}
				}
			}

			LPLog.INSTANCE.d(TAG, "onSuccess: site settings enablePhotoSharing is: " + mPhotoSharingEnabled);

		} catch (JSONException e) {
			LPLog.INSTANCE.w(TAG, "parseConfigurationData: error fetching photo sharing enabled from site settings. Use default (true)", e);
			return false; // Error fetching/parsing json data
		}

		PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SITE_SETTINGS_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mPhotoSharingEnabled);
		PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SITE_SETTINGS_VOICE_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mVoiceSharingEnabled);
		PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SITE_SETTINGS_SDK_ANALYTICS_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mSDKAnalyticsEnabled);
		PreferenceManager.getInstance().setBooleanValue(PreferenceManager.ENABLE_TRANSCENDENT_MESSAGES_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, enableTranscendentMessages);
		PreferenceManager.getInstance().setStringValue(PreferenceManager.SITE_SETTINGS_EVENT_MANAGER_DOMAIN_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mEventManagerDomain);
		PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SITE_SETTINGS_THUMBNAIL_BLUR_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, enableThumbnailBlur);

		Infra.instance.getAnalyticsService().setIsAnalyticsEnabled(mSDKAnalyticsEnabled);

		// Fetch json data was successful
		return true;
	}

	@Override
    public String getName() {
        return TAG;
    }
}
