package com.liveperson.messaging.model;

import static com.liveperson.messaging.model.AmsMessages.QUICK_REPLY_MESSAGE_SEQUENCE;

import android.content.Context;
import android.database.Cursor;
import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.UniqueID;


/**
 * Created by shiranr on 11/3/16.
 */
public class FullMessageRow {

    private static final String TAG = "FullMessageRow";

    public static final String EXTRA_FILE_CONTENT = "EXTRA_FILE_CONTENT";
    public static final String EXTRA_AGENT_AVATAR = "EXTRA_AGENT_AVATAR";
    private MessagingChatMessage messagingChatMessage;
    private String agentAvatar;
    private String agentNickName;
    private FileMessage fileMessage;


    public FullMessageRow(MessagingChatMessage messagingChatMessage, String agentAvatar, FileMessage fileMessage) {
        this.messagingChatMessage = messagingChatMessage;
        this.agentAvatar = agentAvatar;
        this.fileMessage = fileMessage;
    }

    public FullMessageRow(Cursor cursor) {
        long messageId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
        String originatorId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_ORIGINATOR_ID));
        String message = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT));
        long timeStamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
        int typeInt = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_MESSAGE_TYPE));
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.values()[typeInt];
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_STATUS))];
        String dialogId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID));
        String eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
        int serverSequence = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));

        EncryptionVersion messageEncryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE)));

        EncryptionVersion userEncryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE)));
        agentAvatar = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_PROFILE_IMAGE));

        agentAvatar = DBEncryptionHelper.decrypt(userEncryptionVersion, agentAvatar);
        agentNickName = cursor.getString(cursor.getColumnIndex(UsersTable.KEY_NICKNAME));
        agentNickName = DBEncryptionHelper.decrypt(userEncryptionVersion, agentNickName);
        messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, dialogId, eventId, type, state, messageEncryptionVersion);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        if (messagingChatMessage.isFileType()) {
            fileMessage = FileMessage.fromCursor(cursor);
        }
    }

    public void setFileMessage(FileMessage fileMessage) {
        this.fileMessage = fileMessage;
    }

    public FileMessage getFileMessage() {
        return fileMessage;
    }

    public MessagingChatMessage getMessagingChatMessage() {
        return messagingChatMessage;
    }

    public void setAgentAvatar(String agentAvatar) {
        this.agentAvatar = agentAvatar;
    }

    public String getAgentAvatar() {
        return agentAvatar;
    }

    public String getAgentNickName() {
        return agentNickName;
    }

    public void setAgentNickName(String agentNickName) {
        this.agentNickName = agentNickName;
    }

    /**
     * @param newMessageRow
     * @return 0 if it's the same item.
     * 1 if newMessageRow is a new item older than this current msg.
     * -1 if newMessageRow is a new item newer than this current msg.
     */
    public int newerThan(FullMessageRow newMessageRow) {
        boolean isSameLocalId = messagingChatMessage.getLocalId() == newMessageRow.messagingChatMessage.getLocalId();

        boolean isSameSequence = messagingChatMessage.getServerSequence() == newMessageRow.getMessagingChatMessage().getServerSequence();
        isSameSequence &= messagingChatMessage.getServerSequence() != -1; // requried to prevent duplication while fast sending of messages

        boolean isSameDialogId = TextUtils.equals(messagingChatMessage.getDialogId(), newMessageRow.getMessagingChatMessage().getDialogId());
        boolean isSameEventId = TextUtils.equals(messagingChatMessage.getEventId(), newMessageRow.getMessagingChatMessage().getEventId());
        boolean isNewerTime = messagingChatMessage.getTimeStamp() > newMessageRow.getMessagingChatMessage().getTimeStamp();

        if (isSameLocalId && Math.max(messagingChatMessage.getLocalId(), newMessageRow.messagingChatMessage.getLocalId()) > 0) {
            return 0;
        }
        // if we have the same event id or (we need to check sequence and it's the same conv id and sequence - its the same message.
        if (isSameEventId || (isSameSequence && isSameDialogId)) {
            return 0;
        }
        if (isNewerTime) {
            return 1;
        }
        return -1;
    }

    /**
     * Updating current message, and creating diff bundle to update ui.
     *
     * @param newMessageRow
     * @return
     */
    public Bundle update(FullMessageRow newMessageRow) {
        Bundle bundle = new Bundle();
        bundle.putAll(messagingChatMessage.update(newMessageRow.getMessagingChatMessage()));
        if (fileMessage != null) {
            bundle.putAll(fileMessage.update(newMessageRow.getFileMessage()));
        } else if (newMessageRow.getFileMessage() != null) {
            bundle.putAll(newMessageRow.getFileMessage().getAllFileChanges());
            fileMessage = newMessageRow.getFileMessage();
        }
        if (!TextUtils.equals(agentAvatar, newMessageRow.getAgentAvatar())) {
            agentAvatar = newMessageRow.getAgentAvatar();
            bundle.putString(EXTRA_AGENT_AVATAR, agentAvatar);
        }

        return (bundle.isEmpty() ? null : bundle);
    }

    public static FullMessageRow generateUnreadIndicatorMessage(Context context, int numUnreadMessages, long timeStamp) {
        long messageId = -1;
        String originatorId = null;
        String unreadMessagesBase = context.getResources().getQuantityString(R.plurals.lp_unread_message, numUnreadMessages);
        String message = String.format(unreadMessagesBase, numUnreadMessages);
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.UNREAD_INDICATOR;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String convId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, convId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }

    public static Bundle getUnreadIndicatorMessageDiff(FullMessageRow newUnreadIndicatorMessage) {
        Bundle bundle = new Bundle();
        bundle.putString(MessagingChatMessage.EXTRA_MESSAGE_TEXT, newUnreadIndicatorMessage.getMessagingChatMessage().getMessage());
        return bundle;
    }


    public static FullMessageRow getLoadingIndicatorMessage(Context context, long timeStamp, boolean setText) {
        long messageId = -1;
        String originatorId = null;
        String message = setText ? context.getString(R.string.lp_still_loading_message) : "";// at the beginning there is no text of loading. only after 10 seconds we'll show text with loading
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.LOADING;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String dialogId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, dialogId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }


    public static Bundle getLoadingIndicatorMessageDiff(Context context) {
        Bundle bundle = new Bundle();
        // at the beginning there is no text of loading. only after 10 seconds we'll show text with loading
        bundle.putString(MessagingChatMessage.EXTRA_MESSAGE_TEXT, context.getString(R.string.lp_still_loading_message));
        return bundle;
    }

    /**
     * At the beginning there is no text of loading. only after 10 seconds we'll show text with loading
     *
     * @param message Still loading messaging.
     */
    public static FullMessageRow getLoadingHistoryIndicatorMessage(long timeStamp, String message) {
        long messageId = -1;
        String originatorId = null;
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.LOADING;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.RECEIVED;
        String convId = "";
        String eventId = "";
        int serverSequence = -1;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(originatorId, message, timeStamp, convId, eventId, type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);

        return new FullMessageRow(messagingChatMessage, null, null);
    }

    /**
     * Method used to create a date header message instance to represent
     * date header as a native view (as list item).
     *
     * @param dayTimestamp  - beginning of day when messages were sent.
     * @param formattedDate - String representation of date header (Today, Yesterday, date format, etc)
     * @return constructed message row to represent.
     */
    public static FullMessageRow createDateHeader(long dayTimestamp, String formattedDate) {
        long messageId = -11;
        int serverSequence = -9;

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(
                null,
                formattedDate,
                dayTimestamp,
                "",
                "",
                MessagingChatMessage.MessageType.DATE_HEADER,
                MessagingChatMessage.MessageState.RECEIVED,
                EncryptionVersion.NONE
        );
        messagingChatMessage.setServerSequence(serverSequence);
        messagingChatMessage.setMessageId(messageId);
        return new FullMessageRow(messagingChatMessage, null, null);
    }

    /**
     * Generate FullMessageRow object with AGENT_QUICK_REPLIES type that contains the given quickReplyJsonString.
     * This create a message with empty dialogId, eventId, serverSequence, messageId since it is a temporary message that needs to be added to the
     * conversation and then removed. We add dialogId only when Quick reply question is of PCS survey since we want to send metadata to UMS. (See MessagesAsListLoader, case AGENT_QUICK_REPLIES:)
     * The timestamp of the generated message is one millisecond after the given relatedAgentMessageTimestamp since we want this message
     * to appear right after the related agent message
     *
     * @param quickReplyJsonString         the required QuickReplies JSON string
     * @param relatedAgentMessageTimestamp the related agent message's timestamp
     * @param dialogId                     currently active dialog's id. This is Empty unless PCS dialog.
     * @return
     */
    public static FullMessageRow generateAgentQuickReplyMessage(String quickReplyJsonString, long relatedAgentMessageTimestamp, String dialogId) {
        MessagingChatMessage.MessageType type = MessagingChatMessage.MessageType.AGENT_QUICK_REPLIES;
        MessagingChatMessage.MessageState state = MessagingChatMessage.MessageState.READ;

        long quickReplyMessageTimestamp = relatedAgentMessageTimestamp + 1; // Put the QR message right after the related agent message
        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(null, quickReplyJsonString, quickReplyMessageTimestamp, dialogId, UniqueID.createUniqueMessageEventId(), type, state, EncryptionVersion.NONE);
        messagingChatMessage.setServerSequence(QUICK_REPLY_MESSAGE_SEQUENCE);
        messagingChatMessage.setMessageId(-1);

        FullMessageRow message = new FullMessageRow(messagingChatMessage, null, null);

        LPLog.INSTANCE.d(TAG, "generateAgentQuickReplyMessage: add the quick replies json to the message");
        return message;
    }

    @Override
    public String toString() {
        return "FullMessageRow{" +
                "messagingChatMessage=" + messagingChatMessage +
                ", agentAvatar='" + agentAvatar + '\'' +
                ", agentNickName='" + agentNickName + '\'' +
                ", fileMessage=" + fileMessage +
                '}';
    }
}
