package com.liveperson.messaging.commands.tasks

import com.liveperson.infra.ICallback
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.network.http.requests.ACCDNConnectorsRequest
import com.liveperson.messaging.LpError
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.controller.AccountsController
import com.liveperson.messaging.controller.connection.ConnectionParamsCache
import org.json.JSONArray
import kotlin.Exception

/**
 * This task is intended to be executed when performing step up authentication.
 * The purpose is to fetch connector ids associated with a given account.
 * We filters and cache only auth connector id.
 */
class ACCDNConnectorsTask(val accountsController: AccountsController) :
    BaseAmsAccountConnectionTask() {
    private val TAG = "ACCDNConnectorsTask"

    private val CONFIGURATION_KEY = "configuration"
    private val ID_KEY = "id"
    private val JS_METHOD_NAME_KEY = "jsMethodName"
    private val JS_METHOD_CONTEXT_KEY = "jsContext"
    private val AUTH_TYPE_KEY = "type"
    private val ISSUER_DISPLAY_NAME_KEY = "issuerDisplayName"
    var performStepUp: Boolean? = false

    override fun execute() {
        performStepUp = accountsController.getAccount(mBrandId)?.isPerformStepUp
        if (performStepUp == false && getIssuerName() == null) {
            mCallback.onTaskSuccess()
            return
        }
        val certificates: List<String>? = accountsController.getCertificatePinningKeys(mBrandId)
        val acCdnDomain = accountsController.getServiceUrl(
            mBrandId,
            ConnectionParamsCache.CSDS_AC_CDN_DOMAIN_KEY)

        i(TAG, "Starting ACCDNConnectors Task....")

        ACCDNConnectorsRequest(acCdnDomain, mBrandId, certificates, object : ICallback<String, Exception> {
            override fun onSuccess(value: String?) {
                i(TAG, "Successfully got connector information: $value")
                if (!value.isNullOrEmpty()) {
                    try {
                        extractConnectorId(JSONArray(value))
                    } catch (exception: Exception) {
                        onTaskError(exception)
                    }
                }
                mCallback.onTaskSuccess()
            }

            override fun onError(exception: Exception?) {
                onTaskError(exception)
            }

        }).execute()
    }

    private fun onTaskError(exception: Exception?) {
        e(TAG, ErrorCode.ERR_00000164, "Failed to get connector information: $exception")
        mCallback.onTaskError(TaskType.ACCDN_CONNECTORS, LpError.UNKNOWN, exception)
    }

    override fun getName(): String {
        return TAG
    }

    /**
     * Extract auth connector Id from response data.
     */
    private fun extractConnectorId(data: JSONArray) {
        val issuerDisplayName = getIssuerName()

        for (i in 0 until data.length()) {
            val connector = data.getJSONObject(i)
            val connectorType = getRequiredConnectorType()

            if (connector.has(ID_KEY) && connector.has(CONFIGURATION_KEY)) {
                val configuration = connector.getJSONObject(CONFIGURATION_KEY)

                if ((performStepUp == true ) &&
                    (!configuration.has(JS_METHOD_NAME_KEY) || (!configuration.getString(JS_METHOD_NAME_KEY).equals("lpTag.taglets.unAuthMessaging.lpUnauthFunction"))) &&
                    (!configuration.has(JS_METHOD_CONTEXT_KEY) || !configuration.getString(JS_METHOD_CONTEXT_KEY).equals("lpTag.taglets.unAuthMessaging")) &&
                    connector.getInt(AUTH_TYPE_KEY) == connectorType) {
                        val connectorId = connector.get(ID_KEY).toString() // as Long
                        i(TAG, "Found valid connector id: $connectorId")
                        PreferenceManager.getInstance().setStringValue(PreferenceManager.KEY_AUTH_CONNECTOR_ID, mBrandId, connectorId)
                    return
                }
                if (issuerDisplayName != null && configuration.has(ISSUER_DISPLAY_NAME_KEY) &&
                    issuerDisplayName.equals(configuration.getString(ISSUER_DISPLAY_NAME_KEY), true)) {
                    val connectorId = connector.get(ID_KEY).toString()
                    i(TAG, "Found valid connector id of $issuerDisplayName: $connectorId")
                    PreferenceManager.getInstance().setStringValue(PreferenceManager.KEY_AUTH_CONNECTOR_ID, mBrandId, connectorId)
                    return
                }
            }
        }
        w(TAG, "Valid auth connector id not found.")
    }

    /**
     * Based on Authentication type (code or implicit), return the desired connector type (2 or 1)
     */
    private fun getRequiredConnectorType(): Int {
        val lpAuthenticationParams = accountsController.getLPAuthenticationParams(mBrandId)
        if (!lpAuthenticationParams?.authKey.isNullOrEmpty()) {
            return 2
        } else if (!lpAuthenticationParams?.hostAppJWT.isNullOrEmpty()) {
            return 1
        }
        return -1
    }

    /**
     * Get Issuer Display Name if it's available
     * return null if it's not
     */
    private fun getIssuerName(): String? {
        return accountsController.getLPAuthenticationParams(mBrandId)?.issuerDisplayName?.takeIf {
            it.isNotBlank() }?.apply {
                // issuerDisplayName is not blank
            } ?: run {
                null
            }
    }
}