package com.liveperson.messaging.commands.tasks;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.callbacks.AuthCallBack;
import com.liveperson.infra.errors.ErrorCode;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.model.Consumer;
import com.liveperson.infra.model.errors.AuthError;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.AmsConnectionAnalytics;
import com.liveperson.messaging.model.AmsUsers;

import org.jetbrains.annotations.NotNull;

import java.util.List;

import static com.liveperson.messaging.controller.connection.ConnectionParamsCache.CSDS_IDP_DOMAIN_KEY;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A task to get the token from the IDP service
 */
public class IdpTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "IdpTask";
    private final AmsUsers mAmsUsers;
    private AccountsController mAccountsController;

    private String mHostVersion;

    public IdpTask(AccountsController accountsController, AmsUsers amsUsers, String hostVersion){
        mHostVersion = hostVersion;
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.d(TAG, "Running IDP task...");

        AmsConnectionAnalytics.idpTaskStart();

        LPAuthenticationParams auth = mAccountsController.getLPAuthenticationParams(mBrandId);
        // If we already have authentication data for SignUp user, do not send IDP request
        // INC0040824: if user has moved from background to foreground, do not send IDP request (isConnectionResumed = true)
        if (auth == null) {
            LPLog.INSTANCE.e(
                    TAG,
                    ErrorCode.ERR_000000C7,
                    "getLPAuthenticationParams: getLPAuthenticationParams returns null for brand with id: " + mBrandId
            );
            mCallback.onTaskError(
                    TaskType.IDP,
                    LpError.IDP,
                    new NullPointerException("Auth of defined brand id: " + mBrandId + " is null")
            );
            return;
        } else if (Infra.instance.getConsumerManager().getActiveConsumer() != null
                && (auth.getAuthType() == LPAuthenticationType.SIGN_UP
                || Infra.instance.getConsumerManager().isAuthenticated())
        ) {
            Infra.instance.getConsumerManager().setIsAuthenticated();
            onTaskComplete();
            LPLog.INSTANCE.d(TAG, "Not sending Idp request -> mark IdpTask completed");
            return;
        }

        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        AmsAccount account = mAccountsController.getAccount(mBrandId);

        String unAuthConnectorId = account == null ? null : account.getConnectorId();
        String authConnectorId = account == null ? null : account.getAuthConnectorId();

        boolean performSteUp = account != null && account.isPerformStepUp();
        if (performSteUp) {
            backupOldConsumerId();
        }

        // This code block handles case: Use multiple IDPs auth then go back to use default auth (same code or implicit in both cases)
        if (!performSteUp && account != null && account.getLPAuthenticationParams() != null
                && TextUtils.isEmpty(account.getLPAuthenticationParams().getIssuerDisplayName())) {
            authConnectorId = null;
        }

        Infra.instance.getConsumerManager().login(auth, getIdpDomain(), mHostVersion, certificates, unAuthConnectorId, authConnectorId, performSteUp, new AuthCallBack() {
            @Override
            public void onAuthSuccess(@NonNull Consumer consumer) {
                LPLog.INSTANCE.d(TAG, "onAuthSuccess: Consumer: " + LPLog.INSTANCE.mask(consumer));
                mAmsUsers.updateConsumerId(mBrandId, consumer.getConsumerId());
                onTaskComplete();
            }

            @Override
            public void onConsumerSwitch(@NotNull Consumer oldConsumer, @NotNull Consumer newConsumer) {
                switchUser(oldConsumer, newConsumer);
                mAmsUsers.setStepUpConsumerId(oldConsumer.getConsumerId());
                mAmsUsers.updateConsumerId(mBrandId, newConsumer.getConsumerId());
                onTaskComplete();
            }

            @Override
            public void onAuthFailed(@NotNull AuthError error) {
                if (error != null) {
                    LPLog.INSTANCE.d(TAG, "onAuthFailed: error: " + error.toString());
                    mCallback.onTaskError(TaskType.IDP, LpError.IDP, error.getFailureReason(), error.getException());
                }
            }
        });
    }

    /**
     * Get the IDP domain from CSDS. If the "IDP" does not exist in the CSDS domains, we get the AsyncMessagingEnt
     * and replace the "msg" to "idp
     *
     * @return idp domain string
     */
    private String getIdpDomain() {

        String idpDomain;

        // Get the IDP domain from CSDS
        idpDomain = mAccountsController.getServiceUrl(mBrandId, CSDS_IDP_DOMAIN_KEY);
        // If idp domain is missing, try to get it from preferences.
        if (TextUtils.isEmpty(idpDomain)) {
            idpDomain = PreferenceManager.getInstance().getStringValue(CSDS_IDP_DOMAIN_KEY, mBrandId, null);
        }

        // The IDP domain is not always in CSDS. If we got an empty string we go get the AccountsController.AMS_CSDS_DOMAIN_KEY instead
        if (TextUtils.isEmpty(idpDomain)) {
            String asyncMessagingEnt = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
            // Replace the "msg" to "idp"
            idpDomain = asyncMessagingEnt.replaceFirst("msg", "idp");
        }

        return idpDomain;
    }

    /**
     * To perform step up, we need to backup old consumerId.
     */
    private void backupOldConsumerId() {
        Consumer oldConsumer = Infra.instance.getConsumerManager().getActiveConsumer();
        if (oldConsumer != null) {
            mAmsUsers.setStepUpConsumerId(oldConsumer.getConsumerId());
        }
    }

    /**
     * Switch consumer If old consumer id is different than new one
     *
     * @param oldConsumer Old consumer object
     * @param newConsumer new Consumer object
     */
    private void switchUser(Consumer oldConsumer, Consumer newConsumer) {
        String consumerId = newConsumer.getConsumerId();
        String oldConsumerId = oldConsumer.getConsumerId();
        MessagingFactory.getInstance().getController().liteLogout(mBrandId, oldConsumerId, consumerId);
    }

    /**
     * Notify Idp task completion via callback
     */
    private void onTaskComplete() {
        AmsAccount account = mAccountsController.getAccount(mBrandId);
        if (account != null) {
            account.sendAuthenticationCompletedStatus();
            // To make sure we are not skipping authentication when running pusher commands from background, reset auth state.
            if (!ForegroundService.getInstance().isBrandForeground(mBrandId)){
                account.resetConsumerAuthState();
            }
        }

        AmsConnectionAnalytics.idpTaskEnd();
        mCallback.onTaskSuccess();
    }

    @Override
    public String getName() {
        return TAG;
    }
}
