package com.liveperson.messaging.controller;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.SocketTaskType;
import com.liveperson.messaging.model.AmsConnection;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Created by ofira on 06/12/2015.
 */
public class ConnectionsController implements ShutDownAsync, Clearable{

    private static final String TAG = "ConnectionsController";

    public static final String KEY_PREF_LAST_UPDATE_TIME = "KEY_PREF_LAST_UPDATE_TIME";

    //<Brand ID, Ams Connection>
	private Map<String, AmsConnection> mBrandConnection;
    //<Subscription ID, brand ID>
    private Map<String, String> mSubscriptionBrand;
    protected Messaging mController;
    private LocalBroadcastReceiver mConnectionReceiver = null;

    public ConnectionsController(Messaging messagingController) {
        mController = messagingController;
        mBrandConnection = new HashMap<>();
        mSubscriptionBrand = new HashMap<>();
    }

    public void initConnectionReceiver() {
        if (mConnectionReceiver == null){
            LPLog.INSTANCE.d(TAG, "Registering Connection Receiver");
            mConnectionReceiver = new LocalBroadcastReceiver.Builder()
                    .addAction(InternetConnectionService.BROADCAST_INTERNET_CONNECTION_CONNECTED)
                    .addAction(InternetConnectionService.BROADCAST_INTERNET_CONNECTION_DISCONNECTED)
                    .build((context, intent) -> {
	                    LPLog.INSTANCE.d(TAG, "-----Broadcast----- " + intent.getAction());
						String action = intent.getAction();
						if (action != null) {
							switch (action) {
								case InternetConnectionService.BROADCAST_INTERNET_CONNECTION_CONNECTED:
									onResumeConnection();
									break;

								//TODO: Need to check if we need it here?
								case InternetConnectionService.BROADCAST_INTERNET_CONNECTION_DISCONNECTED:
									// Send the NetworkLost event to all connected brands
									for (String brandId : mBrandConnection.keySet()) {
										networkLost(brandId);
									}
									break;
							}
						}
					});
        }
    }

    private void onResumeConnection() {
		// Send the NetworkAvailable event to all connected brands
		for(String brandId : mBrandConnection.keySet()){
            networkAvailable(brandId);
        }
    }

    private void unregisterConnectionReceiver(){
        if (mConnectionReceiver != null) {
            LPLog.INSTANCE.d(TAG, "Unregistering Connection Receiver");
            mConnectionReceiver.unregister();
            mConnectionReceiver = null;
        }
    }
    /**
     * @param brand
     */
    public void addNewConnection(String brand) {
        if (getConnection(brand) == null) {
            LPLog.INSTANCE.i(TAG, "Adding new AmsConnection: " + brand);
            mBrandConnection.put(brand, new AmsConnection(mController, brand));
        }else{
            getConnection(brand).init();
        }
    }

    /**
     * Check if the connection process has been finished for specific brand
     *
     * @param brand
     * @return
     */
    public boolean isSocketOpen(String brand) {
        AmsConnection connection = getConnection(brand);
        return connection != null && connection.isSocketOpen();
    }
    /**
     * Check if the connection process has been finished for specific brand
     *
     * @param brand
     * @return
     */
    public boolean isSocketReady(String brand) {
        AmsConnection connection = getConnection(brand);
        return connection != null && connection.isSocketReady();
    }
    /**
     * Check if the connection process has been finished for specific brand
     *
     * @param brand
     * @return
     */
    public boolean isConnecting(String brand) {
        AmsConnection connection = getConnection(brand);
        return connection != null && connection.isConnecting();
    }

    /**
     * Check if the we are updated- means if we got first ExConversationChangeNotification for specific brand
     *
     * @param brand
     * @return
     */
    public boolean isUpdated(String brand) {
        AmsConnection connection = getConnection(brand);
        return connection != null && connection.isUpdated();
    }

    /**
     *
     * @param brand
     */
    public AmsConnection.AmsSocketState registerSocket(String brand){
        AmsConnection connection = getConnection(brand);
        if (connection == null) {
            return null;
        }
        return connection.registerSocket();
    }

   	/**
	 * Connect to server
	 * @param brand
	 */
    public void connect(String brand, boolean connectInBackground) {

        AmsConnection connection = getConnection(brand);
        if (connection == null) {
            return;
        }
        connection.startConnecting(connectInBackground);
    }
    /**
     * Connect to server
     * @param brand
     */
    public void connect(String brand) {

        connect(brand, false);
    }

	/**
	 * Network was lost
	 * @param brandId
	 */
	private void networkLost(String brandId) {
		AmsConnection connection = getConnection(brandId);
		if (connection == null) {
			return;
		}
		connection.networkLost();
	}

	/**
	 * Network is available
	 * @param brandId
	 */
	private void networkAvailable(String brandId) {
		AmsConnection connection = getConnection(brandId);
		if (connection == null) {
			return;
		}
		connection.networkAvailable();
	}

	/**
	 * Set that the conversation of the given brand was moved to foreground
	 * @param brandId
	 */
	public void moveToForeground(String brandId) {
		AmsConnection connection = getConnection(brandId);
		if (connection != null) {
			connection.moveToForeground();
		}
	}

	/**
	 * Set that the conversation of the given brand was moved to background
	 * @param brandId
	 */
	public void moveToBackground(String brandId, long timeout) {
		AmsConnection connection = getConnection(brandId);
		if (connection != null) {
			connection.moveToBackground(timeout);
		}
	}

	/**
	 * Indicate service has started
	 * @param brandId
	 */
	public void serviceStarted(String brandId) {
		AmsConnection connection = getConnection(brandId);
		if (connection != null) {
			connection.serviceStarted();
		}
	}

	/**
	 * Indicate service has stopped
	 * @param brandId
	 */
	public void serviceStopped(String brandId) {
		AmsConnection connection = getConnection(brandId);
		if (connection != null) {
			connection.serviceStopped();
		}
	}


	/**
     * Get clock diff for specific brand
     *
     * @param brand
     * @return
     */
    public long getClockDiff(String brand) {
        AmsConnection connection = getConnection(brand);
        if (connection == null) {
            return 0L;
        }
        return connection.getClockDiff();
    }


    /**
     * set clock diff for specific brand
     *
     * @param brand
     * @param clock
     */
    public void setClockDiff(String brand, long clock) {
        AmsConnection connection = getConnection(brand);
        if (null != connection) {
            connection.setClock(clock);
        }
    }

    public String getBrandIDForSubscription(String subscriptionID){
        return mSubscriptionBrand.get(subscriptionID);
    }

    public void setLastUpdateTime(String subscriptionID, long updatedTime) {
        String brand = mSubscriptionBrand.get(subscriptionID);
        AmsConnection connection = getConnection(brand);
        if(connection == null){
            return;
        }
        connection.setLastUpdateTime(updatedTime);
    }

    public void clearLastUpdateTime(String brandId) {
        PreferenceManager.getInstance().setLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, brandId, 0);
    }

    public boolean isLastUpdateTimeExists(String subscriptionID) {
        String brand = mSubscriptionBrand.get(subscriptionID);
        AmsConnection connection = getConnection(brand);
        return connection != null && connection.isLastUpdateTimeExists();
    }
    public void setSubscription(String brandId, String subscriptionId) {
        Iterator<Map.Entry<String, String>> it = mSubscriptionBrand.entrySet().iterator();

        while (it.hasNext()) {
            Map.Entry subscription = it.next();
            //remove old subscription for the same brand (brand should have only 1 subscription)
            if (subscription.getValue().equals(brandId)) {
                it.remove();
            }
        }

        mSubscriptionBrand.put(subscriptionId, brandId);
    }

	/**
	 * Get the current subscription ID for the given brand
	 * @param brandId
	 * @return the subscription ID, or null if subscriptionId does not exist for the given brand
	 */
	public String getSubscriptionId(String brandId) {

		for (Map.Entry<String, String> subscription : mSubscriptionBrand.entrySet()) {
			//remove old subscription for the same brand (brand should have only 1 subscription)
			if (subscription.getValue().equals(brandId)) {

				return subscription.getKey();
			}
		}

		// No subscriptionId
		return null;
	}

	public long getLastUpdateTime(String brand) {
        AmsConnection connection = getConnection(brand);
        return connection == null ? 0L : connection.getLastUpdateTime();
    }


    public boolean isFirstNotificationAfterSubscribe(String brand) {
        AmsConnection connection = getConnection(brand);
        if (connection == null) {
            return false;
        }
        return connection.isFirstNotificationAfterSubscribe();
    }

    public void setFirstNotificationAfterSubscribe(String brand, boolean firstNotificationAfterSubscribe) {
        AmsConnection connection = getConnection(brand);
        if (connection == null) {
            return;
        }
        connection.setFirstNotificationAfterSubscribe(firstNotificationAfterSubscribe);
    }

    public void notifySocketTaskFailure(String brandId, LpError error, Throwable exception) {
        AmsConnection connection = getConnection(brandId);
        if (connection == null) {
            return;
        }
        connection.notifySocketTaskFailure(error, exception);
    }


    /**
     * Return the {@link AmsConnection} of specific brand
     *
     * @param brand
     * @return
     */
    public AmsConnection getConnection(String brand) {
        return mBrandConnection.get(brand);
    }

    /**
     * Shutting down all the open connections.
     * @param listener will be called after all the connections were terminated.
     */
    @Override
    public void shutDown(ShutDownCompletionListener listener) {

        unregisterConnectionReceiver();

        Collection<AmsConnection> connections = mBrandConnection.values();
        int numOfConnections = connections.size();
        if (numOfConnections > 0) {
            //one listener that count the number of open connection.
            //when no more open connection available, it calls listener callback
            ShutDownCompletionListener connectionCounterListener =
                    new AmsConnectionShutDownCompletionListener(numOfConnections, listener);

            for (AmsConnection amsConnection : connections) {
                // shutting down connection with the connection counter listener
                amsConnection.shutDown(connectionCounterListener);
            }
        }else{
            //no open connections
            listener.shutDownCompleted();
        }
    }


    @Override
    public void clear(){
        mSubscriptionBrand.clear();
        mBrandConnection.clear();
    }


    /**
     * ShutDownCompletionListener that counts the number of open connection and notifying
     *   the given listener when all the connections are closed.
     * Get numOfConnection as the beginning number of open connection.
     * Decrease the counter when each connection calls shutDownCompleted().
     * When no more open connection available, it calls the given listener callback
     */
    private static class AmsConnectionShutDownCompletionListener implements ShutDownCompletionListener {
        private ShutDownCompletionListener listener;
        int numOfConnections;

        public AmsConnectionShutDownCompletionListener(int numOfConnections, ShutDownCompletionListener listener) {
            this.listener = listener;
            this.numOfConnections = numOfConnections;
        }

        @Override
        public void shutDownCompleted() {
            numOfConnections--;
            if (numOfConnections == 0){
                listener.shutDownCompleted();
            }
        }

		@Override
		public void shutDownFailed() {

		}
	}
}
