package com.liveperson.messaging.network.socket;

import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.events.ExConversationChangeNotification;
import com.liveperson.api.response.model.Change;
import com.liveperson.api.response.model.CobrowseDialogData;
import com.liveperson.api.response.model.ConversationINCADetails;
import com.liveperson.api.response.model.ConversationUMSDetails;
import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.MultiDialog;
import com.liveperson.api.response.model.Result;
import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.sdk.LPConversationData;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.socket.BaseResponseHandler;
import com.liveperson.infra.network.socket.BaseSocketRequest;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.utils.DateUtils;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.infra.utils.ThreadPoolExecutor;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.SocketTaskType;
import com.liveperson.messaging.background.CoBrowseManager;
import com.liveperson.messaging.commands.BasicQueryMessagesCommand;
import com.liveperson.messaging.commands.QueryMessagesUMSCommand;
import com.liveperson.messaging.commands.tasks.BaseAmsSocketConnectionCallback;
import com.liveperson.messaging.commands.tasks.FetchConversationManager;
import com.liveperson.messaging.model.AmsDialogs;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.ConversationUtils;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.DialogUtils;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.http.IncaGetConversationsListRequest;
import com.liveperson.messaging.network.http.QueryMessagesINCACommand;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000BD;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000BE;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000BF;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000C0;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000C1;

/**
 * Created by eyalv on 11/11/15.
 */
public class ExConversationChangeNotificationResponseHandler extends BaseResponseHandler<List<ConversationData>, BaseSocketRequest> {

    private static final String TAG = "ExConversationChangeNotificationResponseHandler";
    public static final String PREF_HIDE_CLOSED_CONVERSATIONS = "hide_closed_conversations";
	protected final Messaging mController;
    private ConversationUtils mConversationUtils;
    private DialogUtils mDialogUtils;
    private int mNumOpenConversations = 0;
    private int mNumCloseConversations = 0;
    private int mNumOfUpdatedConversations = 0;
    private boolean isFirstNotificationForBrand ;
    private boolean isFirstNotificationAfterSubscribe ;
    private String mBrandID;
    private String mSubscriptionId;
    private int queryMessageRequestCounter = 0;
    private boolean firstClosedConversation = true;
    private long mLastUpdateTime = 0;

    public ExConversationChangeNotificationResponseHandler(Messaging messaging) {
        mController = messaging;
        init();
    }

    @Override
    public String getAPIResponseType() {
        return ExConversationChangeNotification.CONVERSATION_CHANGE_NOTIFICATION_TYPE;
    }

    protected void init() {
        mConversationUtils = new ConversationUtils(mController);
        mDialogUtils = new DialogUtils(mController);
    }

    @Override
    protected List<ConversationData> parse(JSONObject jsonObject) {
        ExConversationChangeNotification responseBody = null;
        try {
            responseBody = new ExConversationChangeNotification(jsonObject);
        } catch (JSONException e) {
            LPLog.INSTANCE.d(TAG, "Exception parsing JSON: ", e);
        }
        ArrayList<ConversationData> conversations = new ArrayList<>();
        if (responseBody != null) {

            mSubscriptionId = responseBody.notificationBody.getSubscriptionId();
            isFirstNotificationForBrand = mController.mConnectionController.isLastUpdateTimeExists(mSubscriptionId);
            mBrandID = mController.mConnectionController.getBrandIDForSubscription(responseBody.notificationBody.getSubscriptionId());
            mLastUpdateTime = mController.mConnectionController.getLastUpdateTime(mBrandID);
            mController.mConnectionController.setLastUpdateTime(responseBody.notificationBody.getSubscriptionId(), System.currentTimeMillis());
            isFirstNotificationAfterSubscribe = mController.mConnectionController.isFirstNotificationAfterSubscribe(mBrandID);
            //First time after subscribe we go check if we have to update from INCA
            mController.mConnectionController.setFirstNotificationAfterSubscribe(mBrandID, false);

            for (Change change : responseBody.notificationBody.getChanges()) {
                notifyDialogsStateChanges(change.result.conversationDetails);

                if (!isValidResponse(change.result)) {
                    continue;
                }
                if (!TextUtils.equals(Change.KEYS.UPSERT, change.type) && !TextUtils.equals(Change.KEYS.DELETE, change.type)) {
                    continue;
                }

                Result result = change.result;

                if (result.conversationDetails.getState() == null) {
                    continue;
                }

                ConversationState conversationState = result.conversationDetails.getState();

                switch (conversationState) {
                    case OPEN:
                        mNumOpenConversations++;
                        break;
                    case LOCKED:
                        /*
                         * As a workaround to UMS change, we are replacing every LOCKED conversation to CLOSE
                         * This change doesn't need to cause any problems on our side
                         */
                        conversationState = ConversationState.CLOSE;
                    case CLOSE: {
                        if (TextUtils.equals(Change.KEYS.DELETE, change.type)) {
                            // In UnAuthenticated flow we might receive DELETE type when closing a conversation.
                            // Set PREF_HIDE_CLOSED_CONVERSATIONS to true and apply the needed HistoryControl filter in ConversationFragment.
                            // This will override default HistoryControl filter behaviour and hide all closed conversations from the UI.
                            LPLog.INSTANCE.d(TAG, "Parsing CLOSE event with type DELETE - Setting PREF_HIDE_CLOSED_CONVERSATIONS shared preferences key to TRUE");
                            PreferenceManager.getInstance().setBooleanValue(PREF_HIDE_CLOSED_CONVERSATIONS, mBrandID, true);

                        } else if (TextUtils.equals(Change.KEYS.UPSERT, change.type)) {

                            if (PreferenceManager.getInstance().contains(PREF_HIDE_CLOSED_CONVERSATIONS, mBrandID)) {
                                // Remove PREF_HIDE_CLOSED_CONVERSATIONS shared preferences key if it was previously set.
                                // This will restore default HistoryControl filter behaviour.
                                LPLog.INSTANCE.d(TAG, "Parsing CLOSE event with type UPSERT - Clearing PREF_HIDE_CLOSED_CONVERSATIONS key from shared preferences");
                                PreferenceManager.getInstance().remove(PREF_HIDE_CLOSED_CONVERSATIONS, mBrandID);
                            }
                        }
                        mNumCloseConversations++;
                        break;
                    }
                }

                ConversationData conversationData = new ConversationData(conversationState, result, mBrandID);

                conversations.add(conversationData);

                for(DialogData dialog : conversationData.dialogs) {
                    if(dialog.channelType == MultiDialog.ChannelType.COBROWSE) {
                        CoBrowseManager.instance.cobrowseMessageReceived(mBrandID, mController, dialog, result, conversationData.getAssignedAgentId());
                    }
                }
            }

            Collections.sort(conversations);
        }

        return conversations;
    }


    private boolean isValidResponse(Result res) {
        return res != null && res.conversationDetails != null && !TextUtils.isEmpty(res.conversationId);
    }


    @Override
    protected boolean handle(final List<ConversationData> umsConversations) {

        // Protect a case where we don't have brandId. This usually occurs when there is a disconnection
        // and we get from the server notifications with old subscription that does not exist in the system
        // any more. In this case we skip this handle.
        if ((mBrandID == null) || (mController.mConnectionController.getConnection(mBrandID) == null)) {
            LPLog.INSTANCE.w(TAG, "handle: Cannot get connection for brand " + mBrandID + ". Exit handle");
            return true;
        }

        mNumOfUpdatedConversations = umsConversations.size();
        LPLog.INSTANCE.d(TAG, "umsConversations: " + umsConversations.toString());

        //get most recent closed conversation end time
        //get oldest conversation start time from umsConversations array
        //ask inca to get list of conversation ids between those times
        //getFetchConversationManager().fetchConversationsFirstTime(mBrandID, umsConversations);

        //get list of ConversationsFromInca
        Conversation newestClosedConversation = mController.amsConversations.getNewestClosedConversation(mBrandID);
        //the close time of the most recent closed conversation in our DB, is the last time that we know for sure we were fully updated.
        //Since INCA api filter by start time- we'll check if we need to call INCA by mostUpdatedDBTimeTS, but send startFrom in the API

        // If newestClosedConversation is null  - there are 2 options - first time in app OR after clear history. mLastUpdateTime can help us with history option.
        long mostUpdatedDBTimeTS = newestClosedConversation == null ? mLastUpdateTime : newestClosedConversation.getEndTimestamp();
        long startFrom = newestClosedConversation == null ? 0 : newestClosedConversation.getStartTimestamp(); // newest time in our DB
        long startTo = umsConversations.isEmpty() ? System.currentTimeMillis() : umsConversations.get(0).startTs; // newest time from dataFromUMS
        if (isFirstNotificationAfterSubscribe && !DateUtils.isInTheLast24hours(mostUpdatedDBTimeTS)) {
            //need to check inca, cause the newest closed conversation we have started more than 24 hours ago
            //or its first time and there is no conversation in DB - we ask for all conversations available in the last 13 months.

            for (ConversationData conv : umsConversations) {
                LPLog.INSTANCE.d("ums conversation list: ", conv.conversationId +
                        " startTs: " + new Date(conv.startTs) + " , " +
                        " endTs: " + new Date(conv.endTs));
            }

            IncaGetConversationsListRequest moreConv =
                    new IncaGetConversationsListRequest(mController, mBrandID, startFrom, startTo, 0,
                            new ICallback<ArrayList<ConversationINCADetails>, Exception>() {

                                private static final String TAG = "IncaGetConversationsListRequest::ICallback";

                                @Override
                                public void onSuccess(final ArrayList<ConversationINCADetails> conversationHistoryResponse) {

                                    ThreadPoolExecutor.execute(() -> {
                                        List<ConversationData> incaConversations = new ArrayList<>(conversationHistoryResponse.size());
                                        for (ConversationINCADetails conv : conversationHistoryResponse) {
                                            LPLog.INSTANCE.d(TAG, conv.conversationId +
                                                    " startTs: " + new Date(conv.startTs) + " , " +
                                                    " endTs: " + new Date(conv.endTs));
                                            ConversationData conversationData = new ConversationData(conv, mBrandID);
                                            incaConversations.add(conversationData);
                                        }
                                        LPLog.INSTANCE.d(TAG, "incaConversations: " + incaConversations);
                                        mNumOfUpdatedConversations = umsConversations.size() + incaConversations.size();
                                        fetchHistoryMessages(umsConversations, incaConversations);
                                    });
                                }

                                @Override
                                public void onError(final Exception exception) {
                                    ThreadPoolExecutor.execute(() -> {
                                        LPLog.INSTANCE.w(TAG, "Error while trying to receive conversation list from INCA. error: ", exception);
                                        fetchHistoryMessages(umsConversations, null);
                                    });
                                }
                            });
            moreConv.execute();
        } else {
            fetchHistoryMessages(umsConversations, null);
        }

        return true;
    }

    private void fetchHistoryMessages(List<ConversationData> conversationsFromUMS, List<ConversationData> conversationsFromInca) {
        // Protect a case where this method gets called after logout.
        // In this case logout has already cleared the connection and we should ignore this call.
        if ((mBrandID == null) || (mController.mConnectionController.getConnection(mBrandID) == null)) {
            LPLog.INSTANCE.w(TAG, "fetchHistoryMessages: Cannot get connection for brand " + mBrandID + ". Exit handle");
            return;
        }
        if (isFirstNotificationForBrand) {
            LPLog.INSTANCE.d(TAG, "Start FetchConversationManager - Got " + mNumOfUpdatedConversations + " conversations");
            getFetchConversationManager().fetchConversationsFirstTime(mBrandID, conversationsFromUMS, conversationsFromInca);
            saveLastUpdateTime();
        } else {
            LPLog.INSTANCE.d(TAG, "Start updateConversations - Got " + mNumOfUpdatedConversations + " conversations");
            if (mNumOfUpdatedConversations == 0) {
                onHandleConversationCompleted();
            } else {
                for (ConversationData conversationData : conversationsFromUMS) {
                    updateConversationAndDialogs(conversationData);
                }
                if (conversationsFromInca != null) {
                    for (ConversationData conversationData : conversationsFromInca) {
                        updateConversationAndDialogs(conversationData);
                    }
                }
            }

            // Must subscribe to the open conversation exists, cause its not sure we'll get it in the ExConversation
            subscribeToCurrentOpenDialogIfExists();

            if (conversationsFromUMS.size() == 0) {
                // If no conversation updated- check if we are in off hours.
                MessagingFactory.getInstance().getController().amsConversations.notifyOffHoursStatus(mBrandID);
            }
        }
    }

    private void subscribeToCurrentOpenDialogIfExists() {
        MessagingFactory.getInstance().getController().amsDialogs.queryActiveDialog(mBrandID)
                .setPostQueryOnBackground(data -> {
                    if (data != null) {

                        int lastCurrentSequence = data.getLastServerSequence();
                        if (lastCurrentSequence == -1){
                            lastCurrentSequence = 0;
                        }

                        mController.getMessagingEventSubscriptionManager()
                                .addSubscription(mController, mBrandID, data.getConversationId(), data.getDialogId(), lastCurrentSequence, true);
                    }
                }).execute();
    }

    /**
     * Dispatches Android Broadcast Intents based on the {@link MultiDialog.ChannelType} type of the
     * changes contained in {@link ExConversationChangeNotification}
     * @param details
     */
    private void notifyDialogsStateChanges(ConversationUMSDetails details) {
        if (details != null) {
            for (MultiDialog dialog : details.dialogs /* non-null by src/spec */) {
                if (MultiDialog.ChannelType.COBROWSE.equals(dialog.getType())) {
                    Bundle bundle = new Bundle();
                    bundle.putString("message", ((CobrowseDialogData) dialog).metaData);
                    LocalBroadcast.sendBroadcast("LPMessagingDialog\\COBROWSE", bundle);
                }
            }
        }
    }

    @NonNull
    private FetchConversationManager getFetchConversationManager() {
        return new FetchConversationManager(mController);
    }

    private void updateConversationAndDialogs(final ConversationData data) {
        Dialog currentDialog = null;

        switch (data.state) {
            case CLOSE: {
                currentDialog = mController.amsDialogs.getActiveDialog();
                LPLog.INSTANCE.d(TAG, "Closing conversation : " + data.conversationId + ", firstClosedConversation = " + firstClosedConversation + ", mNumCloseConversations = " + mNumCloseConversations + ", mNumOpenConversations = " + mNumOpenConversations);

                //update on resolved conversation only if there is no open conversation yet.
                //boolean updateUI = false;
                //if(mNumCloseConversations >= 1 && mNumOpenConversations == 0){
                boolean shouldUpdateUI = false;

                //we show CSAT (updateUI) only to the first closed conversation (if we have more than 1 closed conversation in a single ExConversationNotification)
                // and only if this conversation didn't close by System (Auto close)
                if (firstClosedConversation) {
                    firstClosedConversation = false;
                    shouldUpdateUI = true;
                }

                //}
                updateClosedConversation(data, shouldUpdateUI);
            }
            break;
            case OPEN: {
                currentDialog = mController.amsDialogs.getActiveDialog();

                ArrayList<Dialog> umsDialogs = AmsDialogs.extractDialogs(data);
                if (currentDialog != null && data.conversationId.equals(currentDialog.getConversationId())) {
                    Dialog dialogToOpen = null;
                    for (Dialog umsDialog : umsDialogs) {
                        DialogState umsDialogState = umsDialog.getState();
                        final Dialog localDialog = mController.amsDialogs.getDialogById(umsDialog.getDialogId());
                        if (localDialog == null && umsDialogState == DialogState.OPEN) {
                            if (dialogToOpen != null) {
                                LPLog.INSTANCE.e(TAG, ERR_000000BD, "Can't be! There are too many open dialogs in the same conversation");
                            }
                            dialogToOpen = umsDialog;
                        }

                        if (localDialog != null) {
                            if (localDialog.getState() != umsDialogState) {
                                switch (umsDialogState) {
                                    case CLOSE:
                                        updateClosedDialog(data, umsDialog, !firstClosedConversation);
                                        break;
                                    case OPEN:
                                        if (dialogToOpen != null) {
                                            LPLog.INSTANCE.e(TAG, ERR_000000BE, "Can't be! There are too many open dialogs in the same conversation");
                                        }
                                        dialogToOpen = umsDialog;
                                        break;
                                    default:
                                        LPLog.INSTANCE.e(TAG, ERR_000000BF, "This scenario can't occur! conversation data: " + LPLog.INSTANCE.mask(data));
                                        break;
                                }
                            }
                        }
                    }

                    if (dialogToOpen != null) {
                        if (dialogToOpen.getChannelType().equals(MultiDialog.ChannelType.COBROWSE)) {
                            dialogToOpen.setState(DialogState.CLOSE);
                        }
                        openDialog(dialogToOpen, data.source);
                        mController.amsDialogs.setActiveDialog(dialogToOpen);
                    }

                    LPLog.INSTANCE.d(TAG, "Updating current conversation TTR In DB . conversation id = " + data.conversationId);
                    queryMessages(currentDialog, data.source);
                    //updating assigned agent profile if we don't have it yet.
                    checkUpdatedOnAssignedAgent(data, currentDialog);
                    //get assign agent profile and run query messages must be BEFORE updating the conversation
                    mController.amsConversations.updateCurrentConversation(data);
                    mController.amsDialogs.updateDialogs(data);
                    // This is implemented this way since in next version we will need to revert it, so left updateTTR the same
                    long effectiveTTR = mConversationUtils.calculateEffectiveTTR(data.brandId, data.ttrValue, data.manualTTR, data.delayTillWhen);
                    mConversationUtils.updateTTR(data.conversationTTRType, effectiveTTR, data.targetId);
                } else {
                    //must be a new conversation the agent started
                    //creating new current conversation in OPEN state than update ttr & send query messages request
                    LPLog.INSTANCE.i(TAG, "New conversation! id = " + data.conversationId + ", time = " + data.startTs);
                    createNewConversation(data);
                }
            }
            break;
        }

        if (currentDialog == null) {
            currentDialog = mController.amsDialogs.getActiveDialog();
        }

        if (currentDialog != null) {
            String dialogId = currentDialog.getDialogId();
            mDialogUtils.updateParticipants(data.targetId, data.participants.ASSIGNED_AGENT, UserProfile.UserType.AGENT, dialogId, true, false);
            mDialogUtils.updateParticipants(data.targetId, data.participants.AGENTS, UserProfile.UserType.AGENT, dialogId, true, false);
            mDialogUtils.updateParticipants(data.targetId, data.participants.MANAGER, UserProfile.UserType.AGENT, dialogId, true, false);
            mDialogUtils.updateParticipants(data.targetId, data.participants.READER, UserProfile.UserType.AGENT, dialogId, true, false);
            mDialogUtils.updateParticipants(data.targetId, data.participants.CONTROLLER, UserProfile.UserType.CONTROLLER, dialogId, true, false);
        } else {
            //LPMobileLog.w(TAG, "Missing current dialog! Conversation data: " + data);

            // Updating with assigned agent. updating other agents (reader/manager/controller type)
            String conversationId = data.conversationId;
            mConversationUtils.updateParticipants(data.targetId, data.participants.ASSIGNED_AGENT, UserProfile.UserType.AGENT, conversationId, true, false);
            mConversationUtils.updateParticipants(data.targetId, data.participants.AGENTS, UserProfile.UserType.AGENT, conversationId, true, false);
            mConversationUtils.updateParticipants(data.targetId, data.participants.MANAGER, UserProfile.UserType.AGENT, conversationId, true, false);
            mConversationUtils.updateParticipants(data.targetId, data.participants.READER, UserProfile.UserType.AGENT, conversationId, true, false);
            mConversationUtils.updateParticipants(data.targetId, data.participants.CONTROLLER, UserProfile.UserType.CONTROLLER, conversationId, true, false);
        }
    }

    private void openDialog(Dialog dialogToOpen, FetchConversationManager.DATA_SOURCE dataSource) {
        mDialogUtils.updateParticipants(dialogToOpen.getTargetId(), new String[]{dialogToOpen.getAssignedAgentId()}, UserProfile.UserType.AGENT, dialogToOpen.getDialogId(), true, true);
        mController.amsDialogs.createNewCurrentDialog(dialogToOpen);
        queryMessages(dialogToOpen, dataSource);
    }

    /**
     * Each conversation marks when handling the update completed.
     * when all conversation were being handled, we can set the timestamp of the
     * notification received as the updated time for the next subscribe.
     */
    private void onHandleConversationCompleted() {
        mNumOfUpdatedConversations--;
        LPLog.INSTANCE.d(TAG, "onHandleConversationCompleted, mNumOfUpdatedConversations = " +mNumOfUpdatedConversations + ", queryMessageRequestCounter:" + queryMessageRequestCounter);
        if (mNumOfUpdatedConversations <= 0){
            saveLastUpdateTime();
            if (queryMessageRequestCounter == 0)
                updateHandledExConversationCompleted();
        }
    }

    private void updateHandledExConversationCompleted() {
        if (null != mController.mConnectionController.getConnection(mBrandID)) {
            mController.mConnectionController.getConnection(mBrandID).setIsUpdated(true);
        }
        mController.amsMessages.updateHandledExConversation(mNumOfUpdatedConversations <= 0);
    }

    private void saveLastUpdateTime() {
        LPLog.INSTANCE.d(TAG, "Saving last notification update for mSubscriptionId:" + mSubscriptionId);
        mController.mConnectionController.setLastUpdateTime(mSubscriptionId, System.currentTimeMillis());
    }

    private void onCurrentConversationServerIDUpdated(ConversationData conversationData){
        mController.amsConversations.deleteTempConversationServerID().executeSynchronously();
        Dialog dialog;
        DataBaseCommand<Dialog> command = mController.amsDialogs.updateCurrentDialogServerId(conversationData);
        if (command == null) {
            return;
        } else {
            dialog = command.executeSynchronously();
        }
        String dialogId = dialog.getDialogId();
        String conversationId = dialog.getConversationId();
        // Update messages with old dialog ID to new dialog ID
        mController.amsMessages.updateMessagesDialogServerID(dialogId).executeSynchronously();
        // Remove temp dialogs after updating the messages with the new dialog ID.
        mController.amsDialogs.deleteDialogById().executeSynchronously();

        LPLog.INSTANCE.d(TAG, "Finished updating messages with server id");
        for (SendMessageRequest sendMessageRequest : dialog.getPendingData().getPendingMessages()) {
            LPLog.INSTANCE.d(TAG, "Finished updating messages with server id, message: " + LPLog.INSTANCE.mask(sendMessageRequest));
            sendMessageRequest.setDialogId(dialogId).setConversationId(conversationId);
            SocketManager.getInstance().send(sendMessageRequest);

            // Add the message to the message queue in order to track if we got ack on it
            mController.amsMessages.mMessageTimeoutQueue.add(MessageTimeoutQueue.MessageType.PUBLISH, (int) sendMessageRequest.getRequestId(), mBrandID, dialogId, sendMessageRequest.getEventId());
        }
    }

    /**
     * Create new current conversation in OPEN state & send query messages request & update ttr
     */
    private void createNewConversation(final ConversationData data) {
        Long requestId = mController.amsConversations.dequeuePendingConversationRequestId();

        // Detects if it's a temp conversation or if it's a conversation created from outside.
        if (requestId != null) {
            final ConversationData conversationData = new ConversationData();
            conversationData.requestId = requestId;
            conversationData.conversationId = data.conversationId;
            conversationData.targetId = data.targetId;
            conversationData.conversationTTRType = data.conversationTTRType;
            conversationData.state = ConversationState.OPEN;
            if (data.dialogs[0] != null) {
                conversationData.startTs = data.dialogs[0].creationTs;
            }

	        LPLog.INSTANCE.d(TAG, "new conversation created. " + conversationData.conversationId);

            DataBaseCommand<Conversation> dataBaseCommand = mController.amsConversations.updateCurrentConversationServerID(conversationData);
            if (dataBaseCommand != null) {
                dataBaseCommand.setPostQueryOnBackground(conversation -> {
                    onCurrentConversationServerIDUpdated(conversationData);
                }).execute();
            } else {
                onCurrentConversationServerIDUpdated(conversationData);
            }

            LPConversationData convData = new LPConversationData(conversationData.conversationId);
            convData.setCloseReason(null);
            mController.mEventsProxy.onConversationStarted(convData);
        } else {
            Conversation conversation = mController.amsConversations.createNewCurrentConversation(data);
            LPLog.INSTANCE.d(TAG, "We have new Current Dialog! " + conversation.getConversationId() + ". Sending request to query messages and update assigned agent details");

            //we want to have the most updated agent profile- listen to response in order to have resume message
            if (!TextUtils.isEmpty(data.getAssignedAgentId())) {
                mConversationUtils.updateParticipants(data.targetId, new String[]{data.getAssignedAgentId()}, UserProfile.UserType.AGENT, conversation.getConversationId(), true, true);
            }

            ArrayList<Dialog> dialogs = AmsDialogs.extractDialogs(data);
            if (!dialogs.isEmpty()) {
                for (Dialog dialog : dialogs) {
                    openDialog(dialog, data.source);
                }

                Dialog openDialog = AmsDialogs.getOpenDialog(dialogs);
                if (openDialog != null) {
                    mController.amsDialogs.setActiveDialog(openDialog);
                }
            } else {
                // This flow should never occur
                LPLog.INSTANCE.e(TAG, ERR_000000C0, "Conversation data has no dialogs!  conversationData: " + LPLog.INSTANCE.mask(data));
            }

            // This is implemented this way since in next version we will need to revert it, so left updateTTR the same
            long effectiveTTR = mConversationUtils.calculateEffectiveTTR(data.brandId, data.ttrValue, data.manualTTR, data.delayTillWhen);
            mConversationUtils.updateTTR(data.conversationTTRType, effectiveTTR, data.targetId);

            LPConversationData convData = new LPConversationData(data.conversationId);
            convData.setCloseReason(null);
            mController.mEventsProxy.onConversationStarted(convData);
        }
    }

    private void updateClosedConversation(final ConversationData data, final boolean shouldUpdateUI) {
        final String assignedAgentId = data.getAssignedAgentId();
        final ArrayList<Dialog> dialogs = new ArrayList<>();

        Dialog activeDialog = mController.amsDialogs.getActiveDialog();
        if (activeDialog != null) {
            if (activeDialog.getConversationId().equals(data.conversationId)) {
                mController.amsDialogs.closeActiveDialog();
            }
        }

        final HashMap<String, Dialog> dialogsFromData = AmsDialogs.extractDialogsToMap(data);

        mController.amsConversations.updateClosedConversation(data, shouldUpdateUI)
                .setPreQueryOnBackground(() -> {
                    ArrayList<Dialog> _dialogs = mController.amsDialogs.queryDialogsByConversationId(data.conversationId).executeSynchronously();

                    dialogs.addAll(_dialogs);
                    for (Dialog dialog : dialogsFromData.values()) {
                        if (!dialogs.contains(dialog)) {
                            dialogs.add(dialog);
                            dialog.setState(DialogState.OPEN);
                        }
                    }

                    for (Dialog dialog : dialogs) {
                        // Update assigned agent if we don't have in db
                        checkUpdatedOnAssignedAgent(data, dialog);
                    }
                })
                .setPostQueryOnBackground(conversation -> {
                    // Return null only if dialog was already closed.
                    if (conversation != null) {
                        for (final Dialog dialog : dialogs) {
                            if (dialog.getState() == DialogState.CLOSE) { // No need to close an already closed dialog
                                continue;
                            }
                            @Nullable Dialog dialogFromData = dialogsFromData.get(dialog.getDialogId());
                            if (dialogFromData != null && dialogFromData != dialog) {
                                dialog.setEndTimestamp(dialogFromData.getEndTimestamp());
                                dialog.setCloseReason(dialogFromData.getCloseReason());
                            }

                            dialog.setAssignedAgentId(assignedAgentId);
                            // We're doing this synchronously since we're already in a background thread
                            Dialog dialogResult = mController.amsDialogs.updateClosedDialog(data, dialog, shouldUpdateUI).executeSynchronously();
	                        LPLog.INSTANCE.d(TAG, "Updated closed dialog: " + dialogResult.getDialogId());

                            // After creating closed dialog, we can send query request & add resolved message
                            queryMessages(dialog, data.source);
                            // If conversation is closed recently, we updated it by pulling it from INCA If not done already.
                            getFetchConversationManager().refreshPendingConversation(dialog.getConversationId());
                            mDialogUtils.addClosedDialogDivider(data.brandId, dialog, assignedAgentId, data.closeReason, true, null);
                        }
                        // If it's system auto close we'll check it inside.
                        String assignedAgentId1 = data.getAssignedAgentId();
                        // Update participants
                        mConversationUtils.updateParticipants(data.targetId, new String[]{assignedAgentId1}, UserProfile.UserType.AGENT, data.conversationId, true, false);
                    } else {
                        onHandleConversationCompleted();
                    }
                })
                .setPostQueryOnUI(dialog -> {
                    if (dialog != null) {
                        Conversation conversation = mController.amsConversations.getConversationById(dialog.getBrandId(), dialog.getConversationId()).executeSynchronously();
                        // need to check if the conversation (where the dialog belongs to) has state (actual value is stage's value) is CLOSE then fire onConversationResolved callback
                        if (conversation!= null && conversation.getState() == ConversationState.CLOSE) {
                            // Update the callback on closed dialog. If the dialog is not null it means this is a real update
                            LPConversationData convData = new LPConversationData(dialog.getConversationId());
                            convData.setCloseReason(dialog.getCloseReason());
                            mController.mEventsProxy.onConversationResolved(convData);
                        }
                    }
                }).execute();
    }

    private void updateClosedDialog(final ConversationData data, final Dialog dialog, final boolean shouldUpdateUI) {
        if (dialog.isOpen() || dialog.getChannelType() == MultiDialog.ChannelType.COBROWSE) { // Is this dialog already closed?
            LPLog.INSTANCE.e(TAG, ERR_000000C1, "Cannot close a closed dialog");
            return;
        }
        final String assignedAgentId = dialog.getAssignedAgentId();

        mController.amsDialogs.updateClosedDialog(data, dialog, shouldUpdateUI)
                .setPreQueryOnBackground(() -> {
                    //update assigned agent if we don't have in db
                    checkUpdatedOnAssignedAgent(data, dialog);
                })
                .setPostQueryOnBackground(queryDialogResult -> {
                    //return null only if dialog was already closed.
                    if (queryDialogResult != null) {
                        dialog.setAssignedAgentId(assignedAgentId);
	                    LPLog.INSTANCE.d(TAG, "Updating closed dialog. " + queryDialogResult.getDialogId());
                        //after creating closed dialog, we can send query request & add resolved message
                        queryMessages(queryDialogResult, data.source);
                        //if it's system auto close we'll check it inside.
                        mDialogUtils.addClosedDialogDivider(data.targetId, queryDialogResult, assignedAgentId, dialog.getCloseReason(), true, null);
                        //update participants
                        mDialogUtils.updateParticipants(data.targetId, new String[]{assignedAgentId}, UserProfile.UserType.AGENT, data.conversationId, true, false);
                    } else {
                        onHandleConversationCompleted();
                    }
                })
				.setPostQueryOnUI(dialog1 -> {
                    if (dialog1 != null) {
                        Conversation conversation = mController.amsConversations.getConversationById(dialog1.getBrandId(), dialog1.getConversationId()).executeSynchronously();
                        // need to check if the conversation (where the dialog belongs to) has state (actual value is stage's value) is CLOSE then fire onConversationResolved callback
                        if (conversation!= null && conversation.getState() == ConversationState.CLOSE) {
                            // Update the callback on closed dialog. If the dialog is not null it means this is a real update
                            LPConversationData convData = new LPConversationData(dialog1.getConversationId());
                            convData.setCloseReason(dialog1.getCloseReason());
                            mController.mEventsProxy.onConversationResolved(convData);
                        }
                    }
                }).execute();
    }

    private void queryMessages(Dialog dialog, FetchConversationManager.DATA_SOURCE source) {
	    LPLog.INSTANCE.d(TAG, "There are some unread messages for conversationId " + dialog.getConversationId() + ", dialogId: " + dialog.getDialogId() +
                " Current last message sequence in db = " + dialog.getLastServerSequence());

        LPLog.INSTANCE.d(TAG, "Sending request to query unread messages... newer than sequence: " + dialog.getLastServerSequence() +" source = " + source);

        BasicQueryMessagesCommand command = null;
        switch (source) {
            case UMS: {
	            LPLog.INSTANCE.d(TAG, "queryMessages UMS: query for dialogId: " + dialog.getDialogId() + ", conversationId: " + dialog.getConversationId());
                command = new QueryMessagesUMSCommand(mController, mBrandID, dialog.getConversationId(), dialog.getDialogId(), dialog.getLastServerSequence(), true);
            }
                break;
            case INCA: {
                LPLog.INSTANCE.d(TAG, "queryMessages INCA: query for dialogId: " + dialog.getDialogId() + ", conversationId: " + dialog.getConversationId());
                command = new QueryMessagesINCACommand(mController, dialog.getBrandId(), dialog.getConversationId(), dialog.getDialogId(), true);
            }
                break;
        }

        queryMessageRequestCounter++;
        command.setResponseCallBack(new BaseAmsSocketConnectionCallback() {
            @Override
            public void onTaskSuccess() {
                //If we got response for all the query request - update on completed
                queryMessageRequestCounter--;
                onHandleConversationCompleted();
            }

            @Override
            public void onTaskError(SocketTaskType type, Throwable exception) {
                //Do nothing - don't notify on ExConversation completed
            }
        });
        command.execute();
    }

    /**
     * updating assigned agent profile if we don't have it yet.
     * @param data
     * @param currentDialog
     */
    private void checkUpdatedOnAssignedAgent(ConversationData data, Dialog currentDialog) {
        String assignedAgent;

        // in case the conversation just got an assigned agent
        // or
        // just removed an assigned agent
        // or
        // the new assign agent is different then the current one - need to update
        if (isNewAssignedAgent(data, currentDialog) ||
				isNewEmptyAssignedAgent(data, currentDialog) ||
				isReplaceAssignedAgent(data, currentDialog)) {

            String assignedAgentId = data.getAssignedAgentId();
            if (TextUtils.isEmpty(assignedAgentId)) {
                LPLog.INSTANCE.i(TAG, "Assigned agent for conversation " + data.conversationId + " was cleared");
                // No assigned agent
                assignedAgent = null;
            } else {
                LPLog.INSTANCE.i(TAG, "new Assigned agent for conversation " + data.conversationId);
                assignedAgent = assignedAgentId;
            }

            //we want to have the most updated agent profile
            mDialogUtils.updateParticipants(data.targetId, new String[]{assignedAgent}, UserProfile.UserType.AGENT, currentDialog.getDialogId(), true, true);
        }
    }

	private boolean isNewAssignedAgent(ConversationData data, Dialog currentDialog) {
		return TextUtils.isEmpty(data.getAssignedAgentId()) && !TextUtils.isEmpty(currentDialog.getAssignedAgentId());
	}

	private boolean isNewEmptyAssignedAgent(ConversationData data, Dialog currentDialog) {
		return !TextUtils.isEmpty(data.getAssignedAgentId()) && TextUtils.isEmpty(currentDialog.getAssignedAgentId());
	}

	private boolean isReplaceAssignedAgent(ConversationData data, Dialog currentDialog) {
		return (currentDialog.getAssignedAgentId() != null) && (data.getAssignedAgentId() != null) &&
				!currentDialog.getAssignedAgentId().equals(data.getAssignedAgentId());
	}

}
