package com.liveperson.messaging.commands.pusher

import com.liveperson.infra.Infra
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.controller.connection.ConnectionParamsCache

/**
 * Helper class to provide pusher request a necessary data.
 * Should be initialized from each pusher request class as calls are independent from each other
 */
class PusherHelper(val brandId: String, val messagingController: Messaging) {

    companion object {
        /**
         * Perform cleanup for pusher API commands (Mostly when moving to background).
         */
        fun cleanup() {
            SendReadAcknowledgementCommand.clearAcknowledgedConversations()
            GetUnreadMessagesCountCommand.clearMappedUnreadCount()
            ClearUnreadMessagesCountCommand.clearRequestState()
        }
    }

    fun getPusherDomain(): String? {
        var pusherDomain = messagingController.mAccountsController.getServiceUrl(brandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY)
        if (pusherDomain.isNullOrEmpty()) {
            pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, brandId, null)
        }
        return pusherDomain
    }

    fun getConsumerId(): String? {
        var consumerId = messagingController.amsUsers.getConsumerId(brandId)
        if (consumerId.isNullOrEmpty()) {
            if (Infra.instance.consumerManager.hasActiveConsumer()) {
                consumerId = Infra.instance.consumerManager.getActiveConsumer()?.consumerId
            }
        }
        return consumerId
    }

    fun getAuthToken(): String? {
        var token = messagingController.mAccountsController.getToken(brandId)
        if (token.isNullOrEmpty()) {
            if (Infra.instance.consumerManager.hasActiveConsumer()) {
                token = Infra.instance.consumerManager.getActiveConsumer()?.lpToken
            }
        }
        return token
    }

    fun getCertificatePinningKeys(): MutableList<String>? {
        var certificates = messagingController.mAccountsController.getCertificatePinningKeys(brandId)
        if (certificates.isNullOrEmpty()) {
            if (Infra.instance.consumerManager.hasActiveConsumer()) {
                val lpAuthenticationParams = Infra.instance.consumerManager.getActiveConsumer()?.lpAuthenticationParams
                certificates = lpAuthenticationParams?.certificatePinningKeys
            }
        }
        return certificates
    }

    fun getAuthType(): String? {
        val consumerManager = Infra.instance.consumerManager
        val params = messagingController.mAccountsController
            ?.getAccount(brandId)
            ?.lpAuthenticationParams
            ?: consumerManager.takeIf { it.hasActiveConsumer() }?.getActiveConsumer()?.lpAuthenticationParams
        return params?.authType?.name
    }
}