package com.liveperson.messaging.model;

import android.os.Bundle;
import android.os.Handler;

import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.MessagingFactory;

/**
 * Created by shiranr on 23/11/2015.
 */
public class TTRManager implements ShutDown, Clearable {

    private static final String TAG = "TTRManager";
    private static final int MINIMUM_TIME = 1000 * 60;
    private static final String TILL_WHEN_OFF_HOURS = "TILL_WHEN_OFF_HOURS";
    private static final String TTR_VALUE = "TTR_VALUE";
    private static final String MANUAL_TTR = "MANUAL_TTR";
    private static final String EFFECTIVE_TTR = "EFFECTIVE_TTR";
    private static final String DELAY_TILL_WHEN = "DELAY_TILL_WHEN";

	private boolean firstTTRShown = false;
	private int mTtrFistTimeDelay;
    private long mNextTTRUpdate;
    private UpdateTTRRunnable updateTTRRunnable;
    private long mTillWhenOffHours;
	private int mTTRFrequency;
//    private long mEffectiveTTR;
	private long mTTRValue;
	private long mManualTTR;
	private long mDelayTillWhen;
	private long mClockDiff;
	private String mTargetId;
	private long mEffectiveTTR;

    public TTRManager(String targetId) {
    	mTargetId = targetId;
        mTillWhenOffHours = PreferenceManager.getInstance().getLongValue(TILL_WHEN_OFF_HOURS, targetId, -1);
        mTTRValue = PreferenceManager.getInstance().getLongValue(TTR_VALUE, targetId, -1);
        mDelayTillWhen = PreferenceManager.getInstance().getLongValue(DELAY_TILL_WHEN, targetId, -1);
        mEffectiveTTR = PreferenceManager.getInstance().getLongValue(EFFECTIVE_TTR, targetId, -1);
        mManualTTR = PreferenceManager.getInstance().getLongValue(MANUAL_TTR, targetId, -1);
		LPLog.INSTANCE.d(TAG, "TTRManager: loaded data from preferences. ttrValue=" + mTTRValue + ", tillWhenOffHours=" + mTillWhenOffHours +
				", delayTillWhen=" + mDelayTillWhen + ", effectiveTTR=" + mEffectiveTTR + ", manualTTR=" + mManualTTR);

		mTtrFistTimeDelay = Configuration.getInteger(R.integer.ttrFirstTimeDelaySeconds) * 1000;

		int ttrFrequencySeconds = Configuration.getInteger(R.integer.ttrShowFrequencyInSeconds);
		mTTRFrequency = ttrFrequencySeconds * 1000;
		LPLog.INSTANCE.d(TAG, "TTRManager: TTR frequency is: " + mTTRFrequency);
	}

    public long calculateEffectiveTTR(final String targetId, final long ttrValue, final long manualTTR, final long delayTillWhen, final long clockDiff){

		LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: targetId="+ targetId +", ttrValue=" + ttrValue + ", manualTTR=" + manualTTR +" ,delayTillWhen=" + delayTillWhen +", clockDiff=" + clockDiff);

		// We DON'T calculate effectiveTTR if it's greater than 0 and there is no change in other values
		if ((mEffectiveTTR > 0) && (mManualTTR == manualTTR) && (mTTRValue == ttrValue) && (mDelayTillWhen == delayTillWhen)) {
			return mEffectiveTTR;
		}

		mTTRValue = ttrValue;
		mManualTTR = manualTTR;
		mDelayTillWhen = delayTillWhen;
		mClockDiff = clockDiff;
		PreferenceManager.getInstance().setLongValue(TTR_VALUE, targetId, ttrValue);
		PreferenceManager.getInstance().setLongValue(MANUAL_TTR, targetId, manualTTR);
		PreferenceManager.getInstance().setLongValue(DELAY_TILL_WHEN, targetId, delayTillWhen);

		return calculateEffectiveTTR();
	}

/*
	public void updateEffectiveTTR() {
		mEffectiveTTR = calculateEffectiveTTR();
	}
*/

	private long calculateEffectiveTTR() {
		LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: TTR updated, set to show TTR");

		long currentTime = System.currentTimeMillis();

		LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: currentTime: " + currentTime + ", clockDiff: " + mClockDiff);

		if (mManualTTR != -1) {
			long calculatedManualTTR = mManualTTR + mClockDiff;
			LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: manualETTR is on (" + mManualTTR + "), return it + clockDiff: " + calculatedManualTTR);
			return calculatedManualTTR;
		}

		// If TTRValue was not updated yet then return
		if (mTTRValue == -1) {
			LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: ttrValue is -1");
			return -1;
		}

		long delay = currentTime;

		// If delay exist and is greater than current time
		if(mDelayTillWhen > 0 && mDelayTillWhen > currentTime) {
			delay = mDelayTillWhen + mClockDiff;
		}

		LPLog.INSTANCE.d(TAG, "calculateEffectiveTTR: ttrValue: " + mTTRValue + ", delay: " + delay + ". Total effectiveTTR: " + mTTRValue + delay);
		return mTTRValue + delay;
	}

	public void resetEffectiveTTR(){
		LPLog.INSTANCE.d(TAG, "resetEffectiveTTR: reset the effective TTR");
		mEffectiveTTR = -1;
		PreferenceManager.getInstance().setLongValue(EFFECTIVE_TTR, mTargetId, -1);
	}

    public void updateTTR(final String targetId, final long effectiveTTR) {

        // If AutoMessages is enabled we shouldn't display not TTR nor OfflineHours banner
        if (isAutoMessagesEnabled(targetId)) {
            return;
        }

        // If the currently active dialog is of type POST_SURVEY we shouldn't display not TTR nor OfflineHours banner
        if (isDuringPostSurveyDialog()) {
            return;
        }

        //check if this brand is currently under off hours.
        if (handleOffHours(targetId, mDelayTillWhen)) {
            LPLog.INSTANCE.d(TAG, "showTTR: we're in off hours");

        } else if (effectiveTTR > 0 && mEffectiveTTR != effectiveTTR) {
            mEffectiveTTR = effectiveTTR;
            PreferenceManager.getInstance().setLongValue(EFFECTIVE_TTR, targetId, effectiveTTR);

            long ttrDisplayDelay = getTTRDisplayDelayTime();
            //Show only in case we got delay greater or equal to 0
            LPLog.INSTANCE.d(TAG, "ttrDisplayDelay " + ttrDisplayDelay);

            if (ttrDisplayDelay >= 0) {
                removeLastTTRCallback();
                updateTTRRunnable = new UpdateTTRRunnable(effectiveTTR, mClockDiff, targetId);
                Infra.instance.getApplicationHandler().postDelayed(updateTTRRunnable, ttrDisplayDelay);
            }
        }
    }


    public void showTTR(String targetId) {

		LPLog.INSTANCE.d(TAG, "showTTR: start");
		// If AutoMessages is enabled we shouldn't display not TTR nor OfflineHours banner
        if (isAutoMessagesEnabled(targetId)){
            return;
        }

        if (isDuringPostSurveyDialog()) {
            return;
        }

        //check if this brand is currently under off hours.
		if (handleOffHours(targetId, mDelayTillWhen)) {
			LPLog.INSTANCE.d(TAG, "showTTR: we're in off hours");
		} else {
			LPLog.INSTANCE.d(TAG, "showTTR: showing TTR");

			// Recalculate TTR if needed
			if (mEffectiveTTR <= 0) {
				LPLog.INSTANCE.d(TAG, "showTTR: effectiveTTR is < 0. Recalculate");
				mEffectiveTTR = calculateEffectiveTTR();
			}

			LPLog.INSTANCE.d(TAG, "showTTR: effectiveTTR = " + mEffectiveTTR);

			if(mEffectiveTTR > 0) {
				long ttrDisplayDelay = getTTRDisplayDelayTime();
				//Show only in case we got delay greater or equal to 0
				LPLog.INSTANCE.d(TAG, "ttrDisplayDelay " + ttrDisplayDelay);

				if (ttrDisplayDelay >= 0) {
					removeLastTTRCallback();
					updateTTRRunnable = new UpdateTTRRunnable(mEffectiveTTR, mClockDiff, targetId);
					Infra.instance.getApplicationHandler().postDelayed(updateTTRRunnable, ttrDisplayDelay);
				}
			}
		}
	}

    private boolean isAutoMessagesEnabled(String targetId) {
        if (MessagingFactory.getInstance().getController().mAccountsController.isAutoMessagesEnabled(targetId)) {
            LPLog.INSTANCE.d(TAG, "updateTTR: AutoMessages is enabled on this account (" + targetId + "). Disabling TTR and OfflineHours banner");
            return true;
        }
        return false;
    }

    private boolean isDuringPostSurveyDialog() {
        if (MessagingFactory.getInstance().getController().amsDialogs.getActiveDialog() != null
                && MessagingFactory.getInstance().getController().amsDialogs.getActiveDialog().getDialogType() == DialogType.POST_SURVEY){
            LPLog.INSTANCE.d(TAG, "This is a post survey dialog. Disabling TTR and OfflineHours banner");
            return true;
        }
        return false;
    }

    private void removeLastTTRCallback() {
        Handler applicationHandler = Infra.instance.getApplicationHandler();
        if (applicationHandler != null) {
            //TODO: Perry suggests to never nullify the "application handler"
		    applicationHandler.removeCallbacks(updateTTRRunnable);
        }
	}

	/**
     * Send an update that Off Hours is on/off - check if the specified time is in the future.
     * @return
     */
    private boolean handleOffHours(String targetId, long delayTillWhen) {

        boolean isInOffHours = false;
        long now = System.currentTimeMillis();

        // If there is no delay TTR of it is in the past it means we're not in off hours
        if (delayTillWhen <= 0 || (delayTillWhen < now)) {
            if (inOffHours()) {
                LPLog.INSTANCE.d(TAG, "handleOffHours: not in off hours, delayTillWhen = " + delayTillWhen);
                mTillWhenOffHours = -1;
                PreferenceManager.getInstance().setLongValue(TILL_WHEN_OFF_HOURS, targetId, -1);
                cancelOffHours(targetId);
            }
        } else {
            LPLog.INSTANCE.d(TAG, "handleOffHours: in off hours, delayTillWhen = " + delayTillWhen + ", mTillWhenOffHours = " + mTillWhenOffHours);
            isInOffHours = true;

            // If we're already in off hours we don't need to send broadcast
            boolean dateChanged = (mTillWhenOffHours != delayTillWhen);
            mTillWhenOffHours = delayTillWhen;
            PreferenceManager.getInstance().setLongValue(TILL_WHEN_OFF_HOURS, targetId, delayTillWhen);
            //send intent to update ui
            sendUpdateIntent(targetId, delayTillWhen, dateChanged);

            if (!inOffHours()) {
                //callback
                MessagingFactory.getInstance().getController().mEventsProxy.onOfflineHoursChanges(true);

            }
        }

        return isInOffHours;
    }

    private boolean inOffHours() {
        return mTillWhenOffHours > 0;
    }

    /**
     * Send an update of the current off hours state
     * @param targetId
     */
    public void updateIfOffHours(String targetId) {
        //get the time when off hours will be over.
        long tillWhenOffHours = PreferenceManager.getInstance().getLongValue(TILL_WHEN_OFF_HOURS, targetId, -1);

        //if time exists and time didn't pass - send an update
        handleOffHours(targetId, tillWhenOffHours);
    }

    private void cancelOffHours(String targetId) {
        sendUpdateIntent(targetId, -1, false);
        MessagingFactory.getInstance().getController().mEventsProxy.onOfflineHoursChanges(false);

    }

    private void sendUpdateIntent(String targetId, long realDelayTTR, boolean dateChanged) {
        Bundle bundle = new Bundle();
        bundle.putString(AmsConversations.KEY_CONVERSATION_TARGET_ID, targetId);
        bundle.putLong(AmsConversations.DELAY_TILL_WHEN, realDelayTTR);
        bundle.putBoolean(AmsConversations.DELAY_TILL_WHEN_CHANGED, dateChanged);
        LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_UPDATE_CONVERSATION_OFF_HOURS, bundle);
    }


    /**
     * Return TimeBundle only in case the TTR in grater than
     *
     * @param effectiveTTR
     * @param clockDiff
     * @return
     */
    private TimeBundle getTimeToRespondMilliseconds(long effectiveTTR, long clockDiff) {
        effectiveTTR += clockDiff - System.currentTimeMillis();
        if (effectiveTTR < MINIMUM_TIME) {
            return null;
        }
        return new TimeBundle(effectiveTTR);
    }
    /**
     * Get the next TTR delay
     * If this is the first time the TTR is shown, show it after 10 sec
     * otherwise keep a 5 sec interval between each time
     *
     * @return - next delay in millisecond
     */
    private long getTTRDisplayDelayTime() {
        long nextDelay;

        if (!firstTTRShown) {
            LPLog.INSTANCE.d(TAG, "TTR First message in the session");
            firstTTRShown = true;
            long startTime = System.currentTimeMillis();
            mNextTTRUpdate = startTime + mTtrFistTimeDelay;
            nextDelay = mTtrFistTimeDelay;

        } else {
            nextDelay = -1;
            long endTime = System.currentTimeMillis();
            long elapsedTime = endTime - mNextTTRUpdate;

            if (elapsedTime > 0) {
                nextDelay = 0;

                if (elapsedTime < mTTRFrequency) {
                    return -1;
                }

                mNextTTRUpdate = endTime;
            }
        }
        LPLog.INSTANCE.d(TAG, "TTR delay " + nextDelay);
        return nextDelay;
    }

	public void cancelAll() {
		removeLastTTRCallback();
		resetEffectiveTTR();
	}

	@Override
	public void clear() {
		PreferenceManager.getInstance().setLongValue(TILL_WHEN_OFF_HOURS, mTargetId, -1);
		PreferenceManager.getInstance().setLongValue(TTR_VALUE, mTargetId, -1);
		PreferenceManager.getInstance().setLongValue(DELAY_TILL_WHEN, mTargetId, -1);
		PreferenceManager.getInstance().setLongValue(EFFECTIVE_TTR, mTargetId, -1);
		PreferenceManager.getInstance().setLongValue(MANUAL_TTR, mTargetId, -1);

	}

	@Override
    public void shutDown() {
        cancelAll();
    }

    private class UpdateTTRRunnable implements Runnable {
        private final long effectiveTTR;
        private final long clockDiff;
        private final String targetId;

        public UpdateTTRRunnable(long effectiveTTR, long clockDiff, String targetId) {
            this.effectiveTTR = effectiveTTR;
            this.clockDiff = clockDiff;
            this.targetId = targetId;
        }

        @Override
		public void run() {
			TimeBundle ttrBundle = getTimeToRespondMilliseconds(effectiveTTR, clockDiff);
            if (ttrBundle != null) {
                LPLog.INSTANCE.d(TAG, "TTR Days " + ttrBundle.diffDays);
                LPLog.INSTANCE.d(TAG, "TTR Hours " + ttrBundle.diffHours);
                LPLog.INSTANCE.d(TAG, "TTR Minutes " + ttrBundle.diffMinutes);
                Bundle bundle = new Bundle();
                bundle.putParcelable(AmsConversations.KEY_CONVERSATION_TTR_TIME, ttrBundle);
                bundle.putString(AmsConversations.KEY_CONVERSATION_TARGET_ID, targetId);
                LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_UPDATE_CONVERSATION_TTR, bundle);
            } else {
                LPLog.INSTANCE.d(TAG, "Don't show TTR time is less than a minute");
            }
		}
    }
}
