package com.liveperson.api.request;

import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.CampaignInfo;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.PushMessage;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.infra.utils.LocaleUtils;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by Shayl on 8/9/2015.
 */
public class ConsumerRequestConversation extends RequestConversation {

    private static final String TAG = "ConsumerRequestConversation";
    private static final String MOBILE_APP_CONTEXT = "MobileAppContext";
    private static final String SHARK_CONTEXT = "SharkContext";
    private static final String CONTEXT = "context";
    private static final String TTR_DEF_NAME = "ttrDefName";
    private static final String APP_ID = "APP_ID";
    private static final String BRAND_ID = "brandId";
    private static final String SKILL_ID = "skillId";
    private static final String LANG = "lang";
    private static final String CAMPAIGN_INFO = "campaignInfo";
    private static final String CAMPAIGN_ID = "campaignId";
    private static final String ENGAGEMENT_ID = "engagementId";
    private static final String VISITOR_ID = "visitorId";
    private static final String SESSION_ID = "sessionId";

    private static final String CONVERSATION_CONTEXT_KEY = "conversationContext";
    public static final String INTERACTION_CONTEXT_ID = "interactionContextId";

    public String appId;
    public TTRType ttrType;
    public String brandId;
    public String skillId;
    public CampaignInfo mCampaignInfo;
    public String transactionId;
    public boolean isOutBoundCampaignConversation = false;
    public boolean isProactiveRichContent = false;

    public ConsumerRequestConversation(String app_id, TTRType ttrType, String brandId, String skillId, @Nullable CampaignInfo campaignInfo) {
        this.ttrType = ttrType;
        this.appId = app_id;
        this.brandId = brandId;
        this.skillId = skillId;
        this.mCampaignInfo = campaignInfo;
    }


    @Override
    public String getMessageType() {
        return "cm.ConsumerRequestConversation";
    }

    @Override
    protected void toJson(JSONObject json) throws JSONException {
        JSONObject context = new JSONObject();
        context.put(APP_ID, appId);

        JSONObject convContext = new JSONObject();

        useOutboundCampaignIfAvailable();

        //LE-82861 [Android][C4M] Change conversationContext format
		// Add CampaignInfo if all required info available
		if (mCampaignInfo != null && mCampaignInfo.getCampaignId() != null && mCampaignInfo.getEngagementId() != null) {

            JSONObject campaignInfoObject = new JSONObject();
			campaignInfoObject.put(CAMPAIGN_ID, mCampaignInfo.getCampaignId());
			campaignInfoObject.put(ENGAGEMENT_ID, mCampaignInfo.getEngagementId());
			body.put(CAMPAIGN_INFO, campaignInfoObject);

			if (mCampaignInfo.isOutboundCampaign()) {
                //If there's a valid CampaignInfo from proactive push message, use "type": "MobileAppContext"
                convContext.put(JSON_KEY_TYPE, MOBILE_APP_CONTEXT);
            } else {
                //If there's a valid default CampaignInfo, use "type": "SharkContext"
                convContext.put(JSON_KEY_TYPE, SHARK_CONTEXT);
            }
            if (!TextUtils.isEmpty(mCampaignInfo.getContextId())){
                convContext.put(INTERACTION_CONTEXT_ID, mCampaignInfo.getContextId());
            }

            // Add sessionId/visitorId if exist
			if (mCampaignInfo.getSessionId() != null) {
				convContext.put(SESSION_ID, mCampaignInfo.getSessionId());
			}

			if (mCampaignInfo.getVisitorId() != null) {
				convContext.put(VISITOR_ID, mCampaignInfo.getVisitorId());
			}

		}else{
            //If there's no CampaignInfo - use "type": "MobileAppContext"
            convContext.put(JSON_KEY_TYPE, MOBILE_APP_CONTEXT);
        }
        convContext.put(LANG, LocaleUtils.getInstance().getLocaleCode());

		body.put(CONVERSATION_CONTEXT_KEY, convContext);
        body.put(CONTEXT, context);
        body.put(TTR_DEF_NAME, ttrType.name());
        body.put(BRAND_ID, brandId);

        if (TextUtils.isEmpty(skillId)) {
            body.put(SKILL_ID, -1);
        } else {
            body.put(SKILL_ID, skillId);
        }

        json.put(AbstractRequest.JSON_KEY_BODY, body);
    }

    /**
     * Override the default campaign info with outbound campaign details If available in cache (received via push)
     */
    private void useOutboundCampaignIfAvailable() {
        String pushMessageId = PushMessagePreferences.INSTANCE.getClickedNotificationId();

        // If push notification is not clicked but host wants to use stored outbound campaign
        if (TextUtils.isEmpty(pushMessageId) &&
                Configuration.getBoolean(R.bool.show_outbound_in_app_message)) {
            pushMessageId = PushMessagePreferences.INSTANCE.getLatestNotificationIdForBrand(brandId);
        }

        if (!TextUtils.isEmpty(pushMessageId)) {
            PushMessage pushMessage = PushMessagePreferences.INSTANCE.getCachedPushMessage(pushMessageId, brandId);
            if (pushMessage != null && pushMessage.getCampaignInfo().isOutboundCampaign() && !pushMessage.isExpired()) {
                LPLog.INSTANCE.d(TAG, "useOutboundCampaignIfAvailable: We have a valid outbound Campaign Info. Use it for consumer's conversation request.");
                isOutBoundCampaignConversation = true;
                mCampaignInfo = pushMessage.getCampaignInfo();
                transactionId = pushMessage.getTransactionId();
                if (pushMessage.getMessageEvent() != null) {
                    isProactiveRichContent = true;
                }
                PushMessagePreferences.INSTANCE.clearPushMessage(brandId, pushMessageId);
            } else if (pushMessage != null && pushMessage.isExpired()) {
                LPLog.INSTANCE.d(TAG, "useOutboundCampaignIfAvailable: Outbound campaign has expired...");
            }
        }
    }
}

