package com.liveperson.messaging.background;

import android.os.Bundle;
import android.text.TextUtils;

import androidx.annotation.Nullable;

import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.Result;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPostRequest;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.model.AmsConversations;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.CoBrowseMetadata;
import com.liveperson.messaging.model.FullMessageRow;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.model.MessagingUserProfile;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.liveperson.infra.errors.ErrorCode.*;

public class CoBrowseManager {

	private static final String TAG = "CoBrowseManager";

	public static String MODE_VIDEO_CALL = "VIDEO_CALL";
	public static String MODE_VOICE_CALL = "VOICE_CALL";

	public static String STATUS_INVITED = "INVITED";
	public static String STATUS_CLOSED = "CLOSED";
	public static String STATUS_ACCEPTED = "ACCEPTED";

	public static String NOTIFICATION_KEY_JOINED = "joined";
	public static String NOTIFICATION_KEY_ACCEPTED = "accepted";
	public static String NOTIFICATION_KEY_CONSUMER_LEFT = "consumer_left";

	public static String KEY_COBROWSE_METADATA = "cobrowse_metadata";

	public static final CoBrowseManager instance = new CoBrowseManager();

	public void createMessage(Messaging controller, String dialogId, long timestamp, String mainDialogId, String originatorId) {
		MessagingChatMessage inviteMessage = new MessagingChatMessage(originatorId,
				dialogId,
				timestamp, mainDialogId, UniqueID.createUniqueMessageEventId(),
				MessagingChatMessage.MessageType.COBROWSE, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);

		controller.amsMessages.addMessage(inviteMessage, true).execute();
	}

	public void createTranscendentMessage(Messaging controller, String mainDialogId, long timestamp, String originatorId, String text, String brandId) {
		controller.amsMessages.transcendentMessagesExists(mainDialogId, timestamp).setPostQueryOnBackground(exists -> {
			if (!exists) {
				MessagingChatMessage inviteMessage = new MessagingChatMessage(originatorId,
						text,
						timestamp, mainDialogId, UniqueID.createUniqueMessageEventId(),
						MessagingChatMessage.MessageType.CONTROLLER_SYSTEM, MessagingChatMessage.MessageState.RECEIVED, AmsMessages.TRANSCENDENT_MESSAGE_SEQUENCE_NUMBER, ContentType.text_plain.getText(), EncryptionVersion.NONE);
				long lastForegroundMillis = ForegroundService.getInstance().getLatestForegroundTime(brandId);
				controller.amsMessages.addMessage(inviteMessage, lastForegroundMillis < timestamp).execute();
			}
		}).execute();
	}

	public ArrayList<FullMessageRow> getMessages(Messaging controller, String dialogId, String filterDialogId, Long filterTimestamp, String filterString) {
		ArrayList<FullMessageRow> searchedMessageList = controller.amsMessages.loadMessagesOnDbThread(AmsMessages.MessagesSortedBy.DialogId, dialogId, -1, -1, -1);
		ArrayList<FullMessageRow> filtersMessageList = new ArrayList<>();

        if (searchedMessageList != null) {
            LPLog.INSTANCE.d(TAG, "CoBrowse messages retrieved: " + searchedMessageList.size() + "   for mainDialogId: " + dialogId);
            if (filterDialogId != null) {
                for (FullMessageRow msg : searchedMessageList) {
                    if (msg.getMessagingChatMessage().getMessage().equals(filterDialogId)) {
                        filtersMessageList.add(msg);
                    }
                }
            } else if (filterTimestamp != null && filterString != null) {
                for (FullMessageRow msg : searchedMessageList) {
                    if (msg.getMessagingChatMessage().getTimeStamp() == filterTimestamp && msg.getMessagingChatMessage().getMessage().equals(filterString)) {
                        filtersMessageList.add(msg);
                    }
                }
            } else {
                filtersMessageList.addAll(searchedMessageList);
            }
        }

		return filtersMessageList;
	}

	public CoBrowseMetadata saveMetadata(String brandId, String dialogId, CoBrowseMetadata metadata) {
		PreferenceManager.getInstance().setStringValue(KEY_COBROWSE_METADATA + dialogId, brandId, metadata.getJsonObject().toString());

		return metadata;
	}

	@Nullable
	public CoBrowseMetadata getMetadata(String brandId, String dialogId) {
		try {
			String metadataJson = PreferenceManager.getInstance().getStringValue(KEY_COBROWSE_METADATA + dialogId, brandId, null);

			if(metadataJson != null) {
				LPLog.INSTANCE.d(TAG, "CoBrowse manager metadataJson "+metadataJson);

				JSONObject metadata = new JSONObject(metadataJson);
				return new CoBrowseMetadata(metadata);
			}

			return null;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	public void cobrowseMessageReceived(String brandId, Messaging controller, DialogData dialog, Result result, String assignedAgentId) {
		DialogData mainDialog = null;

		for(DialogData dialogData : result.conversationDetails.dialogs) {
			if(dialogData.dialogType == DialogType.MAIN) {
				mainDialog = dialogData;
			}
		}
		if (mainDialog == null || !mainDialog.isOpen) {
			LPLog.INSTANCE.e(TAG, ERR_00000159,"main dialog missing or closed");
			return;
		}

		CoBrowseMetadata currentMetadata = CoBrowseManager.instance.getMetadata(result.conversationDetails.brandId, dialog.dialogId);
		if(currentMetadata != null && dialog.metaData != null && dialog.metaData.callLink == null) {
			//because the call link is lost with the ACCEPTED cobrowse update
			dialog.metaData.callLink = currentMetadata.callLink;
		}

		CoBrowseMetadata metadata = dialog.metaData;
		if (metadata == null) {
			LPLog.INSTANCE.e(TAG, ERR_00000159,"metadata is missing");
			return;
		}

		saveMetadata(result.conversationDetails.brandId, dialog.dialogId, metadata);

		AtomicBoolean newMessages = new AtomicBoolean(false);

		if(dialog.metaData.isInvite()) {
			ArrayList<FullMessageRow> messages = CoBrowseManager.instance.getMessages(controller, mainDialog.dialogId, metadata.dialogId, null, null);
			LPLog.INSTANCE.d(TAG, "CoBrowse INVITATION intercepted!! callLink: "+ metadata.callLink);

			LPLog.INSTANCE.d(TAG, "CoBrowse found "+messages.size()+" messages on mainDialogId "+mainDialog.dialogId+" with coBrowseDialog "+metadata.dialogId);

			if(messages.size() == 0) {
				LPLog.INSTANCE.d(TAG, "CoBrowse INVITATION intercepted!! creating a message for dialogId: " + dialog.dialogId);
				CoBrowseManager.instance.createMessage(controller, dialog.dialogId, dialog.metaDataLastUpdateTs, mainDialog.dialogId, assignedAgentId);
				newMessages.set(true);
				controller.amsDialogs.setActiveCoBrowseDialogId(dialog.dialogId);
			} else {
				LPLog.INSTANCE.d(TAG, "CoBrowse has already a message. Updated only the CoBrowse metadata");
			}
		}

		boolean transcendentMessagesEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.ENABLE_TRANSCENDENT_MESSAGES_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
		if (metadata.notificationKey != null &&
				(MessagingFactory.getInstance().getController().mAccountsController.isAutoMessagesEnabled(brandId) || transcendentMessagesEnabled)) {
			String packageName = Infra.instance.getApplicationContext().getPackageName();
			int resId = Infra.instance.getApplicationContext().getResources().getIdentifier("cobrowse_" + metadata.getAutomaticMessageKey(), "string", packageName);
			final String str = Infra.instance.getApplicationContext().getString(resId);
			String agentId = result.conversationDetails.participants.ASSIGNED_AGENT[0];
			DialogData finalMainDialog = mainDialog;
			MessagingFactory.getInstance().getController().amsUsers.getUserById(agentId)
					.setPostQueryOnUI((MessagingUserProfile data) -> {
						if (data != null) {
							String msgFormatted = String.format(str, data.getNickname());
							CoBrowseManager.instance.createTranscendentMessage(MessagingFactory.getInstance().getController(), finalMainDialog.dialogId, dialog.metaDataLastUpdateTs, agentId, msgFormatted, brandId);
							newMessages.set(true);
						} else {
							LPLog.INSTANCE.d(TAG, "CoBrowseLogic Agent NOT available, showing system message");
						}
					}).execute();
		}

		LPLog.INSTANCE.d(TAG, "CoBrowse intercepted!! " + dialog.metaData.sessionState+"  dialog.metaDataLastUpdateTs: "+dialog.metaDataLastUpdateTs+"   dialogId: "+metadata.dialogId+"  getAutomaticMessageKey: "+metadata.getAutomaticMessageKey());

		Bundle bundle = new Bundle();
		bundle.putString("dialogId", metadata.dialogId);
		bundle.putString("brandId", brandId);
		bundle.putString("agentId", result.conversationDetails.participants.ASSIGNED_AGENT[0]);
		bundle.putString("sessionState", metadata.sessionState);
		bundle.putString("automaticMessageKey", metadata.getAutomaticMessageKey());
		bundle.putBoolean("newMessages", newMessages.get());
		bundle.putString("mainDialogId", mainDialog.dialogId);
		LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_COBROWSE_RECEIVED, bundle);
	}

	public void cancelCobrowse(String brandId, String url, String action) {
		try {
			String token = MessagingFactory.getInstance().getController().mAccountsController.getToken(brandId);

			JSONObject requestJson = new JSONObject();
			requestJson.put("action", action);

			HttpPostRequest httpPostRequest = new HttpPostRequest(url);
			HttpRequestBody jsonBody = new LPJSONObjectBody(requestJson);
			httpPostRequest.setBody(jsonBody);
			httpPostRequest.addHeader("authorization", "bearer " + token);

			httpPostRequest.setCallback(new ICallback<String, Exception>() {

				@Override
				public void onSuccess(String value) {
                    if (TextUtils.isEmpty(value)) {
                        LPLog.INSTANCE.e(TAG, ERR_00000158, "Received empty response from server for cancel cobrowse");
                    }
                }

				@Override
				public void onError(Exception exception) {
					if (exception != null) {
						LPLog.INSTANCE.e(TAG, ERR_00000158, "Failed to send cancel CoBrowse request.", exception);
					}
				}
			});
			HttpHandler.execute(httpPostRequest);
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

}
