package com.liveperson.messaging.commands.tasks;

import android.text.TextUtils;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.ConfigurationRequest;
import com.liveperson.infra.utils.VersionUtils;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;
import com.liveperson.messaging.model.AmsConnectionAnalytics;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 12/29/15.
 */
public class ConfigurationFetcherTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "ConfigurationFetcherTask";

    // The version element name in the response json
    private static final String VERSION_ELEMENT = "android_SDK_min_version";
    private static final String PHOTO_SHARING_ENABLED_ELEMENT = "enablePhotoSharing";

    private final AccountsController mAccountsController;

    public ConfigurationFetcherTask(AccountsController accountsController) {
        mAccountsController = accountsController;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.d(TAG, "Running configuration check task...");

        AmsConnectionAnalytics.configurationFetcherTaskStart();

        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        final String configurationServiceDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_LE_CDN_DOMAIN_KEY);

        new ConfigurationRequest(configurationServiceDomain, certificates, new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String configurationString) {
                try {

                    if (!TextUtils.isEmpty(configurationString)) {

                        JSONObject confJson = new JSONObject(configurationString);

                        // Get enable photo sharing
                        getPhotoSharingEnabledConfiguration(confJson);

                        // Get version
                        String serverVersion = confJson.getString(VERSION_ELEMENT);

                        LPLog.INSTANCE.d(TAG, "onSuccess: Checking SDK version against version from lp-cdn");

                        // Write server version to persistent memory
                        IConnectionParamsCache cache = mAccountsController.getConnectionParamsCache(mBrandId);
                        if (cache != null) {
                            cache.updateLeCdnVersion(serverVersion);
                        }

                        // If the sdk version is smaller than the version in the server we abort
                        if (!VersionUtils.isValidSdkVersion(serverVersion)) {

                            // Send an error
                            String detailMessage = "Current SDK version is smaller than the one from the server (" + serverVersion + "). SDK will not connect. Please upgrade SDK. ";
                            mCallback.onTaskError(TaskType.VERSION, LpError.INVALID_SDK_VERSION, new Exception(detailMessage));
                            return;
                        }

                        LPLog.INSTANCE.d(TAG, "Build version is OK");

                        // Version is OK
                        AmsConnectionAnalytics.configurationFetcherTaskEnd();
                        mCallback.onTaskSuccess();
                    }

                } catch (NullPointerException | JSONException ne) {
                    // In case of an error or we cannot get the configuration string
                    // we continue as usual and not fail the process
                    mCallback.onTaskSuccess();
                }
            }

            @Override
            public void onError(Exception exception) {
                if(exception instanceof SSLPeerUnverifiedException) {
                    mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, LpError.INVALID_CERTIFICATE, exception);
                }else {
                    // Currently, if we cannot get to the configuration server we don't fail
                    mCallback.onTaskSuccess();
                }
            }
        }).execute();

    }

    private void getPhotoSharingEnabledConfiguration(JSONObject confJson) {

        boolean photoSharingEnabled = true; // Default is true (in case we don't get the value from remote configuration)

        try {
            photoSharingEnabled = confJson.getBoolean(PHOTO_SHARING_ENABLED_ELEMENT);

            LPLog.INSTANCE.d(TAG, "onSuccess: configuration enablePhotoSharing is: " + photoSharingEnabled);

        } catch (JSONException e) {
            LPLog.INSTANCE.w(TAG, "getPhotoSharingEnabledConfiguration: photo sharing enabled parameter does not exist in configuration json", e);
        }

        PreferenceManager.getInstance().setBooleanValue(PreferenceManager.KILL_SWITCH_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, photoSharingEnabled);

    }

    @Override
    public String getName() {
        return TAG;
    }
}
