package com.liveperson.messaging.commands.tasks;

import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.monitoring.MonitoringFactory;
import com.liveperson.monitoring.cache.MonitoringParamsCache;
import com.liveperson.monitoring.model.EngagementDetails;
import com.liveperson.monitoring.model.LPMonitoringIdentity;
import com.liveperson.monitoring.sdk.callbacks.EngagementCallback;
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType;
import com.liveperson.monitoring.sdk.responses.LPEngagementResponse;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000D8;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A task to get the token from the IDP service
 */
public class UnAuthGetEngagementTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "UnAuthGetEngagementTask";
    private AccountsController mAccountsController;


    public UnAuthGetEngagementTask(AccountsController accountsController){
        mAccountsController = accountsController;
    }

    /**
     *
     */
    @Override
    public void execute() {

        LPAuthenticationParams auth = mAccountsController.getLPAuthenticationParams(mBrandId);
        if (auth != null) {
            LPLog.INSTANCE.d(TAG, "Running GetEngagement task, relevant only if auth type is UnAuth. current type: " + auth.getAuthType());
            if (auth.getAuthType() != LPAuthenticationType.UN_AUTH){
                //nothing to do here..
                mCallback.onTaskSuccess();
                return;
            }
        } else {
            LPLog.INSTANCE.d(TAG, "getLPAuthenticationParams: getLPAuthenticationParams returns null for brand with id: " + mBrandId);
            mCallback.onTaskError(TaskType.IDP, LpError.IDP, new NullPointerException());
            return;
        }

        if (!MonitoringFactory.INSTANCE.isInitialized()) {
            //THIS SHOULD NOT HAPPEN! the only case this can be is if the host app didn't insert APP install ID - than we won't initialized monitoring.
            Exception exception = new Exception(mBrandId + ": UnAuth flow, monitoring isn't initialized. Are you missing 'app install id' in LivePerson.init? ");
            LPLog.INSTANCE.e(TAG, ERR_000000D8, "MonitoringFactory is not initialized.", exception);
            mCallback.onTaskError(TaskType.IDP, LpError.IDP, exception);
            return;
        }


        if (!isConnectorIdEmpty()) {
            //Already have connector id, nothing to do here..
            LPLog.INSTANCE.d(TAG, "Type is UnAuth and we already have connector id");
            mCallback.onTaskSuccess();
            return;
        }


        MonitoringFactory.INSTANCE.getEngagement(Infra.instance.getApplicationContext(), Collections.singletonList(new LPMonitoringIdentity()), null, new EngagementCallback() {
            @Override
            public void onSuccess(@NotNull LPEngagementResponse lpEngagementResponse) {
                try {
                    List<EngagementDetails> detailsList = lpEngagementResponse.getEngagementDetailsList();
                    if (!detailsList.isEmpty()) {
                        String connectorId = detailsList.get(0).getConnectorId();
                        AmsAccount account = mAccountsController.getAccount(mBrandId);
                        if (account != null ) {
                            account.setConnectorId(connectorId);
                        }
                    }
                     mCallback.onTaskSuccess();
                } catch (NullPointerException e) {
                    mCallback.onTaskError(TaskType.IDP, LpError.IDP, e);
                }
            }

            @Override
            public void onError(@NotNull MonitoringErrorType errorType, @Nullable Exception exception) {
                mCallback.onTaskError(TaskType.IDP, LpError.IDP, exception);
            }
        });
    }


    private boolean isConnectorIdEmpty() {
        String connectorId = null;
        MonitoringParamsCache mMonitoringParamsCache = MonitoringFactory.INSTANCE.getMonitoring().getParamsCache();
        if (mMonitoringParamsCache != null) {
            connectorId = mMonitoringParamsCache.getConnectorId();
        }
        AmsAccount account = mAccountsController.getAccount(mBrandId);
        if (account == null) {
            return true;
        }
        if (connectorId == null) {
            connectorId = account.getConnectorId();
            return TextUtils.isEmpty(connectorId);
        }
        account.setConnectorId(connectorId);
        return false;
    }

    @Override
    public String getName() {
        return TAG;
    }


}
