package com.liveperson.messaging.model;

import android.os.Bundle;
import androidx.annotation.NonNull;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.socket.SocketHandler;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.network.socket.SocketState;
import com.liveperson.infra.network.socket.state.SocketStateListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.StateMachineExecutor;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.IMessaging;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.commands.InitAmsSessionCommand;
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionCallback;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.ConnectionsController;
import com.liveperson.messaging.controller.connection.ConnectionStateMachine;
import com.liveperson.messaging.controller.connection.IConnectionListener;
import com.liveperson.messaging.controller.connection.InternetInformationProvider;
import com.liveperson.messaging.controller.connection.MessagingStateMachineInterface;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000145;

/**
 * Connection model
 */
public class AmsConnection implements ShutDownAsync {

    private static final String TAG = "AmsConnection";

    private static final int IDP_TOKEN_EXPIRED_UNAUTHENTICATED_FLOW = 4407; //Unauthenticated
    private static final int IDP_TOKEN_EXPIRED_AUTHENTICATED_FLOW = 4401; //Authenticated
    public static final String BROADCAST_CONNECTING_TO_SERVER_ERROR = "BROADCAST_CONNECTING_TO_SERVER_ERROR";
    public static final String BROADCAST_START_CONNECTING = "BROADCAST_START_CONNECTING";
    public static final String BROADCAST_SOCKET_OPEN_ACTION = "BROADCAST_SOCKET_OPEN_ACTION";
    public static final String BROADCAST_KEY_SOCKET_READY_ACTION = "BROADCAST_KEY_SOCKET_READY_ACTION";
    public static final String BROADCAST_KEY_SOCKET_READY_EXTRA = "BROADCAST_KEY_SOCKET_READY_EXTRA";
    public static final String BROADCAST_KEY_BRAND_ID = "BROADCAST_KEY_BRAND_ID";
    public static final String BROADCAST_AMS_CONNECTION_UPDATE_ACTION = "BROADCAST_AMS_CONNECTION_UPDATE_ACTION";
    public static final String BROADCAST_AMS_CONNECTION_UPDATE_EXTRA = "BROADCAST_AMS_CONNECTION_UPDATE_EXTRA";
    public static final String BROADCAST_AMS_TOKEN_EXPIRED = "BROADCAST_AMS_TOKEN_EXPIRED";

    protected String mBrandId;

    private long mClockDiff;
    private boolean firstNotificationAfterSubscribe = true;
    private PreferenceManager mPreferenceManager;
    private AmsSocketState mSocketState;
    private final Messaging mController;

    private ConnectionStateMachine mStateMachine;


    /**
     * Keep alive callback - responsible to keep the socket alive by sending GetClock requests.
     */
    private boolean mIsUpdated = false;
    private boolean mIsAgentDetailsUpdated = false;
    private boolean socketReady = false;

    public AmsConnection(Messaging messagingController, String brandId) {
        mController = messagingController;
        mPreferenceManager = PreferenceManager.getInstance();
        mBrandId = brandId;
        mSocketState = new AmsSocketState();

        initConnectionStateMachine();
    }

    private void initConnectionStateMachine() {
        IConnectionListener connectionListener = createConnectionListener();
        InternetInformationProvider internetInformationProvider = createInternetInformationProvider();
        MessagingStateMachineInterface controller = createMessagingStateMachineInterface();

        mStateMachine = new ConnectionStateMachine(controller, internetInformationProvider, ForegroundService.getInstance(), mBrandId, connectionListener);
        mStateMachine.setStateMachineExecutor(new StateMachineExecutor(mStateMachine.getTag(), mStateMachine));
    }

    @NonNull
    private MessagingStateMachineInterface createMessagingStateMachineInterface() {
        return new MessagingStateMachineInterface() {

            @Override
            public AccountsController getAccountsController() {
                return mController.mAccountsController;
            }

            @Override
            public ConnectionsController getConnectionController() {
                return mController.mConnectionController;
            }

            @Override
            public AmsMessages getAmsMessages() {
                return mController.amsMessages;
            }

            @Override
            public AmsConversations getAmsConversations() {
                return mController.amsConversations;
            }

            @Override
            public AmsDialogs getAmsDialogs() {
                return mController.amsDialogs;
            }

            @Override
            public AmsUsers getAmsUsers() {
                return mController.amsUsers;
            }

            @Override
            public IMessaging getMessagingController() {
                return mController;
            }

        };
    }

    @NonNull
    private InternetInformationProvider createInternetInformationProvider() {
        return new InternetInformationProvider() {
            @Override
            public boolean isNetworkAvailable() {
                return InternetConnectionService.isNetworkAvailable();
            }

            @Override
            public void unregisterToNetworkChanges() {
                Infra.instance.unregisterToNetworkChanges();

            }

            @Override
            public void registerToNetworkChanges() {
                Infra.instance.registerToNetworkChanges();
            }
        };
    }

    @NonNull
    private IConnectionListener createConnectionListener() {
        return new IConnectionListener() {
            @Override
            public void notifyError(final TaskType type, final LpError lpError, final String message) {
                mController.mEventsProxy.onError(type, message);
                mController.mEventsProxy.onError(lpError, message);
                LocalBroadcast.sendBroadcast(BROADCAST_CONNECTING_TO_SERVER_ERROR);
            }

            @Override
            public void notifyError(LpError lpError, String message) {
                mController.mEventsProxy.onError(lpError, message);
                LocalBroadcast.sendBroadcast(BROADCAST_CONNECTING_TO_SERVER_ERROR);
            }

            @Override
            public void notifyStartConnecting() {
                LocalBroadcast.sendBroadcast(BROADCAST_START_CONNECTING);
                AmsConnectionAnalytics.startConnecting();
                LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "Start connecting for brand: " + mBrandId);
            }

            @Override
            public void notifyStartDisconnecting() {
                onDisconnected();
                AmsConnectionAnalytics.startDisconnecting();
            }

            @Override
            public void notifyDisconnected() {
                onDisconnected();
                //only when completely disconnected we can clear all the waiting messages
                mController.amsMessages.mMessageTimeoutQueue.removeAll();
            }

            @Override
            public void notifyConnected() {
                //state machine connected - means socket open.
                LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "On connected for brand: " + mBrandId);
                AmsConnectionAnalytics.connected();
            }

            @Override
            public void notifyTokenExpired() {
                notifyHostAppTokenExpired();
            }

            @Override
            public void notifyUserExpired() {
                notifyHostAppUserExpired();
            }
        };
    }

    public AmsSocketState registerSocket() {
        String connectionUrl = mController.mAccountsController.getConnectionUrl(mBrandId);
        String connectionUrlForLogs = mController.mAccountsController.getConnectionUrlForLogs(mBrandId);
	    LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "Register socket for brand " + mBrandId + ", connectionUrl = " + connectionUrlForLogs);
        SocketManager.getInstance().registerToSocketState(connectionUrl, mSocketState);
        return mSocketState;
    }


    private void unregisterSocketListener() {
        String connectionUrl = mController.mAccountsController.getConnectionUrl(mBrandId);
        String connectionUrlForLogs = mController.mAccountsController.getConnectionUrlForLogs(mBrandId);
	    LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "Unregister socket for brand " + mBrandId + ", connectionUrl = " + connectionUrlForLogs);
        SocketManager.getInstance().unregisterFromSocketState(connectionUrl, mSocketState);
    }


    public void networkLost() {
        LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "networkLost: brand " + mBrandId);
        mStateMachine.networkLost();
    }

    public void networkAvailable() {
        LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, "networkAvailable: brand " + mBrandId);
        mStateMachine.networkAvailable();
    }

    public void moveToForeground() {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "moveToForeground for brand " + mBrandId);
        mStateMachine.moveToForeground();
    }

    public void moveToBackground(long timeoutTimerMs) {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "moveToBackground for brand " + mBrandId);
        mStateMachine.moveToBackground(timeoutTimerMs);
    }

    public void serviceStarted() {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "serviceStarted for brand " + mBrandId);
        mStateMachine.serviceStarted();
    }

    public void serviceStopped() {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "serviceStopped for brand " + mBrandId);
        mStateMachine.serviceStopped();
    }

    public void startConnecting(boolean connectInBackground) {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "startConnecting for brand, connectInBackground = " + connectInBackground);
        mStateMachine.startConnecting(connectInBackground);
    }

    private void startDisconnecting() {
	    LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "startDisconnecting for brand " + mBrandId);
        mStateMachine.startDisconnecting();
    }

    private void onSocketProblem() {
	    LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "onSocketProblem for brand " + mBrandId);
        mStateMachine.onSocketProblem();
    }

    /**
     * Indicates if the connection process been finished
     *
     * @return TRUE in case there is a connection, return FALSE otherwise
     */
    public boolean isSocketOpen() {
        synchronized (this) {
            return mStateMachine.isConnected();
        }
    }

    private void setSocketReady(boolean connectedAndUpdating) {
        this.socketReady = connectedAndUpdating;
        sendSocketReadyStatus();
    }

    public boolean isSocketReady() {
        return socketReady;
    }

    private void sendSocketReadyStatus() {
        Bundle connectionBundle = new Bundle();
        connectionBundle.putBoolean(BROADCAST_KEY_SOCKET_READY_EXTRA, isSocketReady());
        connectionBundle.putString(BROADCAST_KEY_BRAND_ID, mBrandId);
        LocalBroadcast.sendBroadcast(BROADCAST_KEY_SOCKET_READY_ACTION, connectionBundle);
    }

    public boolean isConnecting() {
        return mStateMachine.isConnecting();
    }

    public long getClockDiff() {
        return mClockDiff;
    }

    public void setClock(long clock) {
        mClockDiff = clock;
    }

    public long getLastUpdateTime() {
        return mPreferenceManager.getLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, mBrandId, 0);
    }

    private void getUpdates() {
	    LPLog.INSTANCE.i(TAG, "Socket open - starting updating data...");
        initAmsSession();
    }

    private void initAmsSession(){
        new InitAmsSessionCommand(mController, mBrandId, new ICallback<Object, Throwable>() {
            @Override
            public void onSuccess(Object value) {
                LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "getUpdates - Socket connection updates Success");
                AmsConnectionAnalytics.socketReady();
                setSocketReady(true);
            }

            @Override
            public void onError(Throwable exception) {
                LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "getUpdates - Error. ", exception);
                onSocketProblem();
            }
        }).execute();
    }

    public void notifySocketTaskFailure(LpError error, Throwable exception) {
        mStateMachine.notifyLPError(error, exception);
    }

    public void setLastUpdateTime(long lastUpdateTime) {
        mPreferenceManager.setLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, mBrandId, lastUpdateTime);
    }

    public boolean isLastUpdateTimeExists() {
        return (mPreferenceManager.getLongValue(ConnectionsController.KEY_PREF_LAST_UPDATE_TIME, mBrandId, 0) == 0);
    }

    @Override
    public void shutDown(ShutDownCompletionListener listener) {
        //shutting down the connection state machine.
        mStateMachine.shutDown(listener);
    }

    public void init() {
        if (!mStateMachine.isInitialized()) {
            mStateMachine.setStateMachineExecutor(new StateMachineExecutor(mStateMachine.getTag(), mStateMachine));
        }
    }

    public class AmsSocketState implements SocketStateListener {

        private BaseAmsAccountConnectionCallback callback;

        @Override
        public void onStateChanged(SocketState state) {
            LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "onStateChanged with state " + state.name());
            switch (state) {
                case ERROR:
                case CLOSED:
                    if (callback != null) {
                        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "Notify error to task callback");
                        callback.onTaskError(TaskType.OPEN_SOCKET, LpError.SOCKET, new Exception("Open Socket - " + state.name()));
                        callback = null;
                    } else {
                        LPLog.INSTANCE.i(TAG, "Notify socket closed to state machine");
                        onSocketProblem();
                    }
                    break;
                case OPEN:
                    if (callback != null) {
                        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "Notify socket open successfully to task callback");
                        AmsConnectionAnalytics.openSocketTaskEnd();
                        callback.onTaskSuccess();
                        callback = null;
                    } else {
                        LPLog.INSTANCE.e(TAG, ERR_00000145, "Notify socket open successfully but callback is null - CHECK THIS OUT!");
                    }

                    //notify socket open
                    LocalBroadcast.sendBroadcast(BROADCAST_SOCKET_OPEN_ACTION);
                    //update data from socket.
                    getUpdates();
                    break;
            }
        }

        @Override
        public void onDisconnected(String reason, int code) {
            LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "onDisconnected, reason " + reason + " code " + code);

            if (code == 0) {
                //that's a normal closing. will be handled by onStateChanged
                return;
            }

            //Notify the host app only when the token is expired
            if (code == IDP_TOKEN_EXPIRED_UNAUTHENTICATED_FLOW || code == IDP_TOKEN_EXPIRED_AUTHENTICATED_FLOW) {
                startDisconnecting();
                notifyHostAppTokenExpired();
            } else if(code == SocketHandler.CERTIFICATE_ERROR){
                startDisconnecting();
                notifyHostAppCertificateError(reason);
            } else {
                LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, "on disconnect: " + " code " + code + ", Notify socket closed to state machine");
                onSocketProblem();
            }


        }

        public void setCallback(BaseAmsAccountConnectionCallback callback) {
            this.callback = callback;
        }
    }

    private void notifyHostAppTokenExpired() {
		LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, mBrandId + ": notifying host app token expired!");
            LocalBroadcast.sendBroadcast(BROADCAST_AMS_TOKEN_EXPIRED);
            mController.mEventsProxy.onTokenExpired();
    }

    private void notifyHostAppUserExpired() {
            LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, mBrandId + ": notifying host app user expired!");
            mController.mEventsProxy.onUnauthenticatedUserExpired();
    }

    private void notifyHostAppCertificateError(String errorMessage){
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, mBrandId + ": notifying host app invalid certificate");
        mController.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, errorMessage);
        LocalBroadcast.sendBroadcast(BROADCAST_CONNECTING_TO_SERVER_ERROR);
    }

    private void onDisconnected() {
        LPLog.INSTANCE.i(TAG, FlowTags.LOGIN, mBrandId + ": disconnected!");
        setIsUpdated(false);
        setSocketReady(false);
        unregisterSocketListener();
        mController.getMessagingEventSubscriptionManager().clearAllSubscriptions();
    }

       public boolean isUpdated() {
        return mIsUpdated;
    }

    public void setIsUpdated(boolean isUpdated) {
        if (isUpdated != mIsUpdated) {
            LPLog.INSTANCE.d(TAG, FlowTags.LOGIN, mBrandId + ": setIsUpdated = " + isUpdated);
            mIsUpdated = isUpdated;
            notifyOnConnectionCompleted(mIsUpdated);
            if (mIsUpdated) {
                mController.amsReadController.registerForegroundConnectionReceiver(mBrandId);
            }
        }
    }

    private void notifyOnConnectionCompleted(final boolean isUpdated) {
		Bundle connectionBundle = new Bundle();
		connectionBundle.putString(BROADCAST_KEY_BRAND_ID, mBrandId);
		connectionBundle.putBoolean(BROADCAST_AMS_CONNECTION_UPDATE_EXTRA, isUpdated);
		LocalBroadcast.sendBroadcast(BROADCAST_AMS_CONNECTION_UPDATE_ACTION, connectionBundle);

        mController.mEventsProxy.onConnectionChanged(isUpdated);
    }

    public boolean isAgentDetailsUpdated() {
        return mIsAgentDetailsUpdated;
    }

    public void setAgentDetailsUpdated(boolean updated) {
        this.mIsAgentDetailsUpdated = updated;
    }

    public boolean isFirstNotificationAfterSubscribe() {
        return firstNotificationAfterSubscribe;
    }

    public void setFirstNotificationAfterSubscribe(boolean firstNotificationAfterSubscribe) {
        this.firstNotificationAfterSubscribe = firstNotificationAfterSubscribe;
    }

}
