package com.liveperson.messaging.commands;

import androidx.annotation.NonNull;

import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.database.DBUtilities;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.BaseSocketRequest;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.FileMessage;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.http.RestRequestParams;
import com.liveperson.messaging.network.http.SendFileRequestRest;
import com.liveperson.messaging.network.socket.requests.NewConversationRequest;
import com.liveperson.messaging.network.socket.requests.SendFileRequest;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000AA;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000AF;

/**
 * A command for sending message
 */
public abstract class SendFileCommand extends SendMessageCommand {

	private static final String TAG = "SendFileCommand";
	private final String mFileContentType;
	private final String mFileTypeExtension;
	private String mThumbnailPath;
	private String mLocalFilePath;
	protected String mCaption;
	private String mRelativePath;
	private String mFileType;
	protected String mPreview;

	protected SendFileCommandListener mCallback;
	protected long mMessageRowId = -1;
	private boolean sendViaRest;
	private RestRequestParams restDomain;

	protected abstract MessagingChatMessage.MessageType getMessageType(MaskedMessage message);

	/**
	 * Creates a command for sending messages
	 *
	 * @param controller
	 * @param message
	 */
	public SendFileCommand(Messaging controller, String targetId, String brandId, String contentType, String thumbnailPath, String filePath, String fileTypeExtension, MaskedMessage message) {
		super(controller, targetId, brandId, message);
		this.mFileContentType = contentType;
		mThumbnailPath = thumbnailPath;
		mLocalFilePath = filePath;
		mMessage = message;
		this.mCaption = mMessage.getServerMessage();
		mFileTypeExtension = fileTypeExtension;
	}

	public void setCallback(SendFileCommandListener mCallback) {
		this.mCallback = mCallback;
	}

	public void setFileDetails(String relativePath, String fileType, String previewBase64) {

		this.mRelativePath = relativePath;
		this.mFileType = fileType;
		this.mPreview = "data:" + mFileContentType + ";base64," + previewBase64;
	}

	/**
	 * Adding file message to db under the current available conversation and dialog.
	 * we'll update it again after connecting to the server
	 */
	public void addMessageToDB() {
		try {
			LPLog.INSTANCE.d(TAG, "addMessageToDB");
			String newDialogId = updateMessageWithCurrentDialog();

			LPLog.INSTANCE.d(TAG, "addMessageToDB File! - dialogId = " + newDialogId);
			addMessageToDBAndSend(newDialogId, mMessage);

		} catch (Exception e) {
			LPLog.INSTANCE.e(TAG, ERR_000000AA, "Exception while adding message to database or sending it.", e);
		}
	}

	/**
	 * Create conversation and dialog if they are not open yet.
	 *
	 * @return new Temp dialog id in case of opening new dialog or the current dialog id otherwise
	 */
	private String updateMessageWithCurrentDialog() {
		//Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		Dialog dialog = mController.amsDialogs.getActiveDialog();

		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		String newDialogId;

		if (dialog == null) {
			LPLog.INSTANCE.i(TAG, "updateMessageWithCurrentDialog: creating new temp dialog and conversation, there is no open one.");
			newDialogId = Dialog.TEMP_DIALOG_ID;

			createQueuedConversationAndDialog();

		} else {
			switch (dialog.getState()) {
				case CLOSE:
					LPLog.INSTANCE.i(TAG, "updateMessageWithCurrentDialog: creating new temp dialog and conversation, they are closed.");
					newDialogId = Dialog.TEMP_DIALOG_ID;

					BaseAMSSocketRequest newConversationRequest = createPendingConversationAndDialog();
					sendCreateConversationRequest(newConversationRequest);
					break;
				case OPEN:
				case PENDING:
				case QUEUED:
				default:
					newDialogId = dialog.getDialogId();
					LPLog.INSTANCE.i(TAG, "updateMessageWithCurrentDialog: getting current dialog ID: " + newDialogId + ", state: " + dialog.getState());
					break;
			}
		}
		return newDialogId;
	}

	/**
	 * This method is used to add a message to DB. If regexPattern and maskCharacter are provided, the message is checked
	 * and masked if the regex pattern exist in the message. Otherwise it is sent as is.
	 *
	 * @param dialogId
	 * @param message
	 */
	protected void addMessageToDBAndSend(String dialogId, MaskedMessage message) {
		mEventId = UniqueID.createUniqueMessageEventId();
		LPLog.INSTANCE.d(TAG, "addMessageToDBAndSend, createNewChatMessage, ContentType: " + mFileContentType);
		MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message);

		chatMessage.setContentType(mFileContentType);

		// Add the message to DB
		mController.amsMessages.addMessage(chatMessage, true)
				.setPostQueryOnBackground(rowId -> {

					if (rowId == DBUtilities.ROW_UPDATED) {
						LPLog.INSTANCE.d(TAG, "onResult: message was updated on DB (and not inserted). No need to add the file to DB");
					} else {
						LPLog.INSTANCE.d(TAG, "addMessageToDBAndSend,  MessagingChatMessage was added. row id: " + rowId + ". Adding fileMessage to db.");

						// Add the file the DB
						FileMessage fileMessage = new FileMessage(mThumbnailPath, mFileTypeExtension, mLocalFilePath, null, rowId);
						long fileRowId = MessagingFactory.getInstance().getController().amsFiles.addFile(rowId, fileMessage).executeSynchronously();

						mMessageRowId = rowId;
						LPLog.INSTANCE.d(TAG, "addMessageToDBAndSend, fileMessage was added to db. Thumbnail path: " + mThumbnailPath + ", local file path: " + mLocalFilePath);

						mController.amsMessages.updateFileMessageByRowId(rowId, fileRowId).executeSynchronously();
						if (mCallback != null) {
							mCallback.onFileAddedToDB(mMessageRowId, fileRowId);
						}
					}
				}).execute();


		// If messages are different it means the message was masked. We add a system message for the user
		if (mMessage.isMasked()) {
			MessagingChatMessage warning = new MessagingChatMessage(chatMessage.getOriginatorId(),
					message.getMaskedSystemMessage(),
					chatMessage.getTimeStamp() + 1, chatMessage.getDialogId(), UniqueID.createUniqueMessageEventId(),
					MessagingChatMessage.MessageType.SYSTEM_MASKED, MessagingChatMessage.MessageState.RECEIVED,
					AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER, ContentType.text_plain.getText(), EncryptionVersion.NONE);

			mController.amsMessages.addMessage(warning, true).execute();
		}

	}


	/**
	 * After connected to UMS we updating message's dialog id according to the new one (if there is)
	 */
	public void updateMessageConversationId() {
		LPLog.INSTANCE.i(TAG, "update Message dialog ID");
		String newDialogId = updateMessageWithCurrentDialog();

		LPLog.INSTANCE.i(TAG, "update Message dialog ID - updating file message: " + mMessageRowId + " with new dialog id = " + newDialogId);

		//updating the message with new dialog id (if needed)
		mController.amsMessages.updateMessageDialogServerIdAndTime(mMessageRowId, newDialogId)
				.setPostQueryOnBackground((Void data) -> {
					LPLog.INSTANCE.i(TAG, "Finished updating file message ( dialog ID and new time ) ");
					if (mCallback != null) {
						mCallback.onMessageUpdatedInDB();
					}
				}).execute();
	}

	/**
	 * Sending the message or creating conversation and adding the message to current conversation.
	 */
	@Override
	public void execute() {
		LPLog.INSTANCE.i(TAG, "Sending file message. creating new conversation if there is no open one.");
		//Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		Dialog dialog = mController.amsDialogs.getActiveDialog();

		mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

		boolean needMetadata = false;

		if (dialog == null) {
			LPLog.INSTANCE.e(TAG, ERR_000000AF, "SHOULD NEVER HAPPEN!!");
			addMessageToDBAndSend(Dialog.TEMP_DIALOG_ID, mMessage);
			BaseAMSSocketRequest newConversationRequest = createPendingConversationAndDialog();
			sendCreateConversationRequest(newConversationRequest);
		} else {
			LPLog.INSTANCE.i(TAG, "checking current dialog, state - " + dialog.getState());
			switch (dialog.getState()) {
				case CLOSE:
					// If the conversation is closed during file upload, we fail the message
					LPLog.INSTANCE.i(TAG, "Dialog is closed. Fail the file message");
					markMessageAsError();
					break;
				case QUEUED:
					LPLog.INSTANCE.i(TAG, "Dialog is queued and waiting to be created...");
					markMessageAsPending();
					sendNewConversationRequest(dialog.getConversationId(), dialog.getDialogId(), dialog.getRequestId());
					needMetadata = true;
					break;
				case OPEN:
				case PENDING:
					LPLog.INSTANCE.i(TAG, "Dialog is open/pending. Sending message");
					markMessageAsPending();
				default:
					break;
			}
		}
		// When dialog is queued and waiting to be created, this will be the first message of the conversation
		// then need to form the metadata of welcome message if it's being stored
		if (needMetadata) {
			formMetadataOnNewConversation(mBrandId);
		}
		sendMessageIfDialogIsOpen();
	}

	private void markMessageAsPending() {
		LPLog.INSTANCE.i(TAG, "Changing message state to pending.. waiting to be send...");
		mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.PENDING);
	}

	private void markMessageAsError() {
		LPLog.INSTANCE.i(TAG, "Changing message state to Error.");
		mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.ERROR);
	}

	private void sendNewConversationRequest(String newConversationId, String newDialogId, long requestId) {
		LPLog.INSTANCE.i(TAG, "Sending request to create new conversation. conversation id = " + newConversationId);
		NewConversationRequest newConversationRequest = createNewConversationRequest(newConversationId, newDialogId, requestId);
		mController.amsConversations.updateConversationState(mTargetId, newConversationId, ConversationState.PENDING);
		mController.amsDialogs.updateDialogState(newDialogId, DialogState.PENDING);
		SocketManager.getInstance().send(newConversationRequest);
	}

	private void createQueuedConversationAndDialog(){
		long requestId = BaseSocketRequest.createRequestId();
		createQueuedConversation(requestId);
		createQueuedDialog(requestId);
	}

	private void createQueuedConversation(long requestId) {
		mController.amsConversations.createQueuedConversation(mTargetId, mBrandId, requestId);
	}

	private void createQueuedDialog(long requestId){
		mController.amsDialogs.createQueuedDialog(mTargetId, mBrandId, requestId);
	}

	@NonNull
	protected MessagingChatMessage createNewChatMessage(String dialogId, MaskedMessage message) { // TODO: 5/7/18 make abstract and implement on image and voice
		return new MessagingChatMessage(mController.getOriginatorId(mTargetId),
				message.getDbMessage(), System.currentTimeMillis(),
				dialogId,
				mEventId,
				getMessageType(message),
				MessagingChatMessage.MessageState.QUEUED,
				EncryptionVersion.NONE);
	}

    @NonNull
    protected SendMessageRequest createMessageRequest(Messaging mController, String eventId, String mTargetId, String mBrandId, String dialogId, String conversationId) {
		SendFileRequest sendFileRequest = new SendFileRequest(mController, eventId, mTargetId, mBrandId, dialogId, conversationId);
		sendFileRequest.setFileContent(mCaption, mRelativePath, mFileType, mPreview);
		sendFileRequest.setInfo(mInfo);
		return sendFileRequest;
	}

	@Override
	protected void sendMessage(String conversationServerId, SendMessageRequest sendMessageRequest) {
		if (sendViaRest) {
			new SendFileRequestRest(restDomain, sendMessageRequest).execute();
		} else {
			super.sendMessage(conversationServerId, sendMessageRequest);
		}
	}

	public void failMessage() {

		LPLog.INSTANCE.w(TAG, "failMessage: upload file failed");

		if (mMessageRowId != -1) {
			LPLog.INSTANCE.d(TAG, "failMessage: setting message (rowId: " + mMessageRowId + ") to error");
			mController.amsMessages.updateMessageState(mMessageRowId, MessagingChatMessage.MessageState.ERROR);
		}

		// If conversation is in state QUEUED close it
		final Conversation conversation = mController.amsConversations.getConversation(mTargetId);
		if (conversation != null && conversation.getState() == ConversationState.QUEUED) {
			LPLog.INSTANCE.d(TAG, "failMessage: conversation " + conversation.getConversationId() + " is queued. Close it");
			ConversationData conversationData = new ConversationData();
			conversationData.conversationId = Conversation.TEMP_CONVERSATION_ID;
			conversationData.brandId = mBrandId;
			conversationData.targetId = mTargetId;

			mController.amsConversations.updateClosedConversation(conversationData, false).execute();

		}

        Dialog dialog = mController.amsDialogs.getActiveDialog();

        // If dialog is in state QUEUED close it
        if (dialog != null && dialog.getState() == DialogState.QUEUED) {
            ConversationData conversationData = new ConversationData();
            conversationData.conversationId = Conversation.TEMP_CONVERSATION_ID;
            conversationData.brandId = mBrandId;
            conversationData.targetId = mTargetId;
            mController.amsDialogs.updateClosedDialog(conversationData, dialog, false).execute();
        }
	}

	public void setSendViaRest(boolean sendViaRest, RestRequestParams restDomain) {
		this.sendViaRest = sendViaRest;
		this.restDomain = restDomain;
	}

	public interface SendFileCommandListener {
		void onFileAddedToDB(long messageRowId, long fileRowId);

		void onMessageUpdatedInDB();
	}
}
