package com.liveperson.messaging.controller;

import android.content.Context;
import android.os.Build;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.BuildConfig;
import com.liveperson.infra.messaging.R;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.TimeZone;

import static com.liveperson.infra.utils.VersionUtils.getAppVersion;
import static com.liveperson.infra.utils.DeviceUtils.INSTANCE;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000008E;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000008F;

/**
 * Created by shiranr on 30/06/2016.
 */
public class ClientProperties implements Clearable {

	private static final String TAG = "ClientProperties";

	private static final String APP_ID_PREFERENCE_KEY = "APP_ID_PREFERENCE_KEY";

	private static final String APP_ID = "appId";
	private static final String DEVICE_FAMILY = "deviceFamily";
	private static final String OS = "os";
	private static final String IP_ADDRESS = "ipAddress";
	private static final String OS_NAME = "osName";
	private static final String OS_VERSION = "osVersion";
	private static final String INTEGRATION = "integration";
	private static final String INTEGRATION_VERSION = "integrationVersion";
	//	private static final String BROWSER = "browser";
    //	private static final String BROWSER_VERSION = "browserVersion";
	private static final String TIME_ZONE = "timeZone";
	private static final String DEVICE_MANUFACTURER = "deviceManufacture"; // Incorrect spelling required
	private static final String DEVICE_MODEL = "deviceModel";
	private static final String APP_VERSION = "appVersion";
	private static final String FEATURES = "features";
	private static final String TYPE = "type";
	private static final String TYPE_VALUE = ".ClientProperties";


	private String mAppId;
	private String mSdkVersion;
	private final String ipAddress;

	private JSONObject mJson;

	public ClientProperties(String appId, String appVersion) {
		String _ipAddress = "unknown";
		try {
			_ipAddress = InetAddress.getLocalHost().getHostAddress();
		} catch (UnknownHostException e) {
			LPLog.INSTANCE.e(TAG, ERR_0000008E, "Failed to get ip address, unknown host exception.", e);
		}

		ipAddress = _ipAddress;
		saveAppId(appId);
		saveSDKVersion(appVersion);
		createJson();

		LPLog.INSTANCE.v(TAG, getLoggablePropertyDump(
				Infra.instance.getApplicationContext(), "Created ClientProperties:", appId));
	}

	public ClientProperties() {
		this(null, null);
	}

	private void createJson() {
		mJson = new JSONObject();
		try {
			mJson.put(APP_ID, mAppId);
			mJson.put(APP_VERSION, getAppVersion(Infra.instance.getApplicationContext()));
			mJson.put(DEVICE_FAMILY, DeviceFamily.MOBILE.name());
			mJson.put(OS, OperatingSystem.ANDROID.name());
			mJson.put(OS_NAME, INSTANCE.getOsName());
			mJson.put(OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
			mJson.put(DEVICE_MODEL, Build.MODEL);
			mJson.put(DEVICE_MANUFACTURER, Build.MANUFACTURER);
			mJson.put(INTEGRATION, Integration.MOBILE_SDK.name());
			mJson.put(INTEGRATION_VERSION, mSdkVersion);
			mJson.put(TYPE, TYPE_VALUE);

			// Build the features array
			JSONArray featuresArray = new JSONArray();
			featuresArray.put(Features.PHOTO_SHARING.name())
					.put(Features.CO_APP.name())
					.put(Features.RICH_CONTENT.name())
					.put(Features.SECURE_FORMS.name())
					.put(Features.AUTO_MESSAGES.name())
					.put(Features.QUICK_REPLIES.name())
					.put(Features.MULTI_DIALOG.name());

			if (Configuration.getBoolean(R.bool.enable_file_sharing)) {
				featuresArray.put(Features.FILE_SHARING.name());
			}

			if (Configuration.getBoolean(R.bool.enable_markdown_hyperlink)) {
				featuresArray.put(Features.MARKDOWN_HYPERLINKS.name());
			}

			mJson.put(FEATURES, featuresArray);

			mJson.put(TIME_ZONE, TimeZone.getDefault().getID());

		} catch (JSONException e) {
			LPLog.INSTANCE.e(TAG, ERR_0000008F, "JSONException while adding properties to JSON Object.", e);
		}
	}

	/**
	 * Move the cobrowse to above method. This is a temporary fix to disable cobrowse for unauth flow.
	 */
	public String getJsonString(boolean isUnauth) {
		if (Configuration.getBoolean(R.bool.lp_enable_voice_video_call) && !isUnauth) {
			try {
				if (!hasCobrowse(mJson.getJSONArray(FEATURES))) {
					mJson.getJSONArray(FEATURES).put(Features.CO_BROWSE.name());
				}
			} catch (JSONException e) {
				LPLog.INSTANCE.e(TAG, ERR_0000008F, "JSONException while adding properties to JSON Object.", e);
			}
		}
		return mJson.toString();
	}

	private boolean hasCobrowse(JSONArray jsonArray) throws JSONException {
		for (int i = 0; i < jsonArray.length(); i++) {
			if (jsonArray.getString(i).equals(Features.CO_BROWSE.name())) {
				return true;
			}
		}
		return false;
	}

	private void saveAppId(String newAppId) {
		if (TextUtils.isEmpty(newAppId)) {
			mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
		} else {
			mAppId = newAppId;
			PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mAppId);
		}
	}

	private void saveSDKVersion(String newSdkVersion) {
		if (TextUtils.isEmpty(newSdkVersion)) {
			mSdkVersion = PreferenceManager.getInstance().getStringValue(PreferenceManager.SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");
		} else {

			final String oldSdkVersion = PreferenceManager.getInstance().getStringValue(PreferenceManager.SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, "");

			if (!TextUtils.isEmpty(oldSdkVersion) && (!oldSdkVersion.equals(newSdkVersion))) {
				PreferenceManager.getInstance().setBooleanValue(PreferenceManager.SDK_VERSION_CHANGED_CLEAR_DATABASE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
			}

			mSdkVersion = newSdkVersion;
			PreferenceManager.getInstance().setStringValue(PreferenceManager.SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, mSdkVersion);
		}
	}

	@Override
	public void clear() {
		PreferenceManager.getInstance().remove(APP_ID_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
		PreferenceManager.getInstance().remove(PreferenceManager.SDK_VERSION_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
		PreferenceManager.getInstance().remove(PreferenceManager.SDK_VERSION_CHANGED_CLEAR_DATABASE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES);
	}

	@NonNull
	public static String getLoggablePropertyDump(@NonNull Context applicationContext, @NonNull String dumpTitle, @Nullable String appId) {
		return "\n========================================\n" +
				dumpTitle + "\n" +
				"----------------------------------------\n" +
				"Device Family:      " + DeviceFamily.MOBILE.name() + "\n" +
				"OS:                 " + OperatingSystem.ANDROID.name() + "\n" +
				"OS Name:            " + INSTANCE.getOsName() + "\n" +
				"OS Version:         " + Build.VERSION.RELEASE+ "\n" +
				"OS Codename:        " + Build.VERSION.CODENAME + "\n" +
				"API Version:        " + Build.VERSION.SDK_INT + "\n" +
				"Device Model:       " + Build.MODEL + "\n" +
				"Device Maker:       " + Build.MANUFACTURER + "\n" +
				(appId != null ? "Host App ID:        " + appId + "\n" : "") +
				"Host App Version:   " + getAppVersion(applicationContext) + "\n" +
				"LP Client:          " + Integration.MOBILE_SDK.name() + "\n" +
				"LP Client Version:  " + BuildConfig.VERSION_NAME + "\n" +
				"========================================";
	}

	/* https://lpgithub.dev.lprnd.net/Architecture/api-ums/blob/314b3b72eaf0c276a35118414c79ad16b4ad7c6a/3.0.0.9/scheme.json
	  "appId" : "appIdExample",
	  "ipAddress" : "127.321.22.1",
	  "deviceFamily" : "MOBILE_PHONE",
	  "os" : "ANDROID",
	  "osVersion" : "18",
	  "integration" : "MOBILE_SDK",
	  "integrationVersion" : "1.2",
	  "browser" : "OTHER",
	  "browserVersion" : "",
	  "timeZone" : "Europe/Athens"
	  */
	private enum DeviceFamily {
		MOBILE,
		TABLET,
		PERSONAL_COMPUTER,
		PDA,
		OTHER,
		UNKNOWN
	}

	private enum OperatingSystem {
		WINDOWS,
		LINUX,
		OSX,
		ANDROID,
		IOS,
		OTHER
	}

	private enum Integration {
		/** In case of connecting via regular browser */
		WEB_SDK,

		/** In case of connecting via native application */
		MOBILE_SDK,

		/** In case of connecting via singe brand connection (GW) */
		BRAND_SDK
	}

	private enum Browser {
		CHROME,
		FIREFOX,
		IE,
		OPERA,
		SAFARI,
		OTHER
	}

	public enum Features {
		CO_BROWSE,
		CO_APP,
		PHOTO_SHARING,
		SECURE_FORMS,
		RICH_CONTENT,
		AUTO_MESSAGES,
		QUICK_REPLIES,
		MULTI_DIALOG,
		FILE_SHARING,
		MARKDOWN_HYPERLINKS
	}
}
