package com.liveperson.messaging.commands.pusher;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LPConversationsHistoryStateToDisplay;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPutRequest;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;

/**
 * SendReadAcknowledgementCommand sends read acknowledgement request to pusher for the list of conversationIds provided in request body.
 */
public class SendReadAcknowledgementCommand implements Command {
    private static final String TAG = "SendReadAcknowledgementCommand";

    static final String CONSUMER_ID = "consumerId";
    static final String CONVERSATION_IDS = "conversationIds";
    private static final String PUSHER_URL = "https://%s/api/account/%s/device/read-ack";
    private final Messaging messagingController;
    private String brandId;
    private static HashSet<String> acknowledgedConversationIds;
    private String dialogId;
    private String conversationId;
    private ICallback<String, Exception> callback;

    public SendReadAcknowledgementCommand(Messaging messagingController, String brandId, String dialogId, String conversationId, ICallback<String, Exception> callback) {
        this.messagingController = messagingController;
        this.brandId = brandId;
        this.dialogId = dialogId;
        this.conversationId = conversationId;
        this.callback = callback;
    }

    @Override
    public void execute() {

        if (acknowledgedConversationIds == null) {
            acknowledgedConversationIds = new HashSet<>();
        }

        LPConversationsHistoryStateToDisplay historyStateToDisplay = messagingController.getConversationViewParams().getHistoryConversationsStateToDisplay();

        boolean isDialogClosed = messagingController.isDialogClosed(dialogId);

        LPLog.INSTANCE.d(TAG, "isDialog " + dialogId + " Closed: " + isDialogClosed + " historyStateToDisplay: " + historyStateToDisplay);

        // Do not send ack. for closed conversation if customer has chosen to display only open conversations and vice-versa
        if ((isDialogClosed && historyStateToDisplay == LPConversationsHistoryStateToDisplay.OPEN)
                || (!isDialogClosed && historyStateToDisplay == LPConversationsHistoryStateToDisplay.CLOSE)
                || !ForegroundService.getInstance().isBrandForeground(brandId)) {
            return;
        } else if (acknowledgedConversationIds.contains(conversationId)) {
            LPLog.INSTANCE.d(TAG, "Already acknowledged conversation: " + conversationId);
            callback.onSuccess(conversationId);
            return;
        }
        // We cache conversationId to prevent sending read status repeatedly for same the conversationId
        acknowledgedConversationIds.add(conversationId);

        String pusherDomain = messagingController.mAccountsController.getServiceUrl(brandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
        String requestURL = String.format(PUSHER_URL, pusherDomain, brandId);

        validateStateAndSendRequest(requestURL);
    }

    /**
     * Checking if we have an active internet connection and send the request to Pusher
     *
     * @param requestURL
     */
    private void validateStateAndSendRequest(final String requestURL) {

        if (InternetConnectionService.isNetworkAvailable()) {
            sendRequest(requestURL);
        } else {
            new SynchronizedInternetConnectionCallback(() -> sendRequest(requestURL)).execute();
        }
    }

    private void sendRequest(String requestURL) {
        String token = messagingController.mAccountsController.getToken(brandId);

        JSONObject requestJson = getBody();
        HttpPutRequest httpPutRequest = new HttpPutRequest(requestURL);
        HttpRequestBody jsonBody = new LPJSONObjectBody(requestJson);
        httpPutRequest.setBody(jsonBody);
        httpPutRequest.addHeader("authorization", "bearer " + token);

        httpPutRequest.setCallback(new ICallback<String, Exception>() {

            @Override
            public void onSuccess(String value) {
                try {
                    if (!TextUtils.isEmpty(value)) {
                        LPLog.INSTANCE.d(TAG, "onSuccess: " + value);
                        JSONArray jsonArray = new JSONArray(value);
                        for (int i = 0; i < jsonArray.length(); i++) {
                            JSONObject obj = jsonArray.getJSONObject(i);
                            String id = obj.getString("conversationId");
                            callback.onSuccess(id);
                        }
                    } else {
                        LPLog.INSTANCE.e(TAG, "Received empty response from pusher for read ack request");
                    }
                } catch (JSONException error) {
                    LPLog.INSTANCE.e(TAG, "Failed to parse read-ack response: ", error);
                }
            }

            @Override
            public void onError(Exception exception) {
                if (exception != null) {
                    try {
                        String mess = parseException(exception.getMessage());
                        JSONObject error = new JSONObject(mess);
                        String errorString = error.getString("error");
                        if (!TextUtils.isEmpty(errorString)) {
                            JSONObject errorBody = new JSONObject(errorString);
                            String statusCode = errorBody.getString("statusCode");
                            String internalCode = errorBody.getString("internalCode");

                            // Ignore the error If tried to send read-ack for conversation id that is not present in pusher.
                            if (!statusCode.equals("404") && !internalCode.equals("23")) {
                                LPLog.INSTANCE.e(TAG, "Failed to send read acknowledgement to pusher pusher.", exception);
                            } else {
                                callback.onSuccess("");
                            }
                        }

                    } catch (Exception e) {
                        LPLog.INSTANCE.e(TAG, "Failed to parse unread message count exception from pusher.", e);
                    }
                }
            }
        });
        LPLog.INSTANCE.d(TAG, "Sending read ack to pusher for: " + conversationId);
        HttpHandler.execute(httpPutRequest);
    }

    @NonNull
    protected JSONObject getBody() {
        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put(CONSUMER_ID, messagingController.amsUsers.getConsumerId(brandId));
            JSONArray ids = new JSONArray();
            ids.put(0, conversationId);
            requestBody.put(CONVERSATION_IDS, ids);
        } catch (Exception error) {
            LPLog.INSTANCE.e(TAG, "getBody: ", error);
        }
        return requestBody;
    }

    /**
     * Extract actual error object from Exception.
     * @param exception
     * @return
     */
    private String parseException(String exception) {
        return exception.substring(exception.indexOf('{'));
    }

    /**
     * Clear list of all acknowledged conversation ids when user moves to background.
     */
    public static void clearAcknowledgedConversations() {
        if (acknowledgedConversationIds != null) {
            LPLog.INSTANCE.d(TAG, "Removing all cached conversation ids");
            acknowledgedConversationIds.clear();
        }
    }
}
