package com.liveperson.messaging.model;

import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.network.http.AgentProfileRequest;
import com.liveperson.messaging.utils.ClockUtils;

public class DialogUtils {

    private static final String TAG = "DialogUtils";
    protected final Messaging mController;

    public DialogUtils(Messaging controller) {
        mController = controller;
    }

    /**
     * Add resolve message and notify to the UI that the conversation has resolved
     *
     * @param resolvedBy
     */
    public void addClosedDialogDivider(final String brandId, final Dialog dialog, String assignedAgentServerId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        final String originatorId;

        if (resolvedBy == CloseReason.CONSUMER) {
            originatorId = mController.getOriginatorId(brandId);
        } else if (resolvedBy == CloseReason.TIMEOUT || resolvedBy == CloseReason.SYSTEM) {
            originatorId = null;
        } else {
            originatorId = assignedAgentServerId;
        }

        addResolvedDivider(dialog, originatorId, resolvedBy, updateUI, callback);
    }

    /**
     * Adds a divider between the messages, this methods usually helps to add a simple divider without a timestamp between dialogs.
     *
     * @param dialog
     * @param originatorId
     * @param resolvedBy
     * @param updateUI
     * @param callback
     */
    private void addResolvedDivider(final Dialog dialog, final String originatorId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        final long timestamp = ClockUtils.addDiffToTimestamp(dialog.getEndTimestamp());
        if (dialog.getDialogType() == DialogType.MAIN) {
            if (resolvedBy == CloseReason.TIMEOUT || resolvedBy == CloseReason.SYSTEM) {
                LPLog.INSTANCE.d(TAG, "This conversation was Auto closed - Don't add Resolved message");
                if (callback != null){
                    callback.onError(new Exception("This conversation timed out or Auto closed"));
                }
            } else {
                addResolveMessage(dialog.getDialogId(), timestamp, "", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_RESOLVED, callback);
            }
        } else {
            addResolveMessage(dialog.getDialogId(), timestamp, "dialog closed", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_DIALOG_RESOLVED, callback);
        }
    }

    private void addResolveMessage(String dialogId, long timestamp, String message, String originatorId, boolean updateUI, MessagingChatMessage.MessageType messageType, final ICallback<Void, Exception> callback) {
	    LPLog.INSTANCE.d(TAG, "creating message resolve at time: " + timestamp
                + " timestamp: " + timestamp);
        String eventId = timestamp + "_" + AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER;

        MessagingChatMessage systemResolved = new MessagingChatMessage(originatorId, message, timestamp, dialogId, eventId,
                messageType, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
        systemResolved.setServerSequence(AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER);
        mController.amsMessages.addMessage(systemResolved, updateUI).setPostQueryOnBackground(data -> {
            if (callback != null) {
                callback.onSuccess(null);
            }
        }).execute();
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, participants, userType, dialogId, updateUI, forceUpdate, null);
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(userProfile -> {
                    if (userProfile == null) {
                        userProfile = new MessagingUserProfile("", "", userType);
                        userProfile.setOriginatorID(userId);
                        mController.amsUsers.updateUserProfile(userProfile);

	                    LPLog.INSTANCE.i(TAG, "First time bringing information for another participant that joined dialog " + dialogId);
                        sendUpdateUserRequest(targetId, userId, dialogId, updateUI, callback);
                    } else if (userProfile.isEmptyUser() || forceUpdate) {
                        sendUpdateUserRequest(targetId, userId, dialogId, updateUI, callback);
                    } else if (callback != null) {
                        callback.onSuccess(userProfile);
                    }
                }).execute();
            } else {
                LPLog.INSTANCE.e(TAG, "Missing agent ID!");
            }
        }
    }

    /**
     * @param targetId
     * @param userId
     * @param dialogId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String userId, String dialogId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, dialogId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(dialogId)){
                LPLog.INSTANCE.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsDialogs.queryDialogById(dialogId)
                    .setPostQueryOnBackground(dialog -> {
	                    LPLog.INSTANCE.d(TAG, "onResult: Calling agent details callback with null agent");
                        mController.onAgentDetailsChanged(null, dialog.isOpen());
                    }).execute();
        }
    }

}
