package com.liveperson.messaging.model;

import android.text.TextUtils;

import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.EncryptionVersion;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by nirni on 4/9/18.
 *
 * This holds a QuickReplies JSON string including the related message timestamp, originatorId and a boolean whether to show it or not
 */
public class QuickRepliesMessageHolder {

	private static final String TAG = "QuickRepliesMessageHolder";
	private static String QUICK_REPLIES_SHARED_PREFERENCES_KEY = "quick_replies_shared_preferences_key";

	private static String QUICK_REPLIES_KEY = "quick_replies_key";
	private static String TIMESTAMP_KEY = "timestamp_key";
	private static String ORIGINATOR_ID_KEY = "originator_id_key";
	private static String SEQUENCE_KEY = "sequence_key";
	private static String SHOW_KEY = "show_key";

	private String mQuickRepliesString;
	private String mOriginatorId;
	private int mSequence;
	private long mTimestamp;
	private boolean mShow;
	private String mBrandId;

	/**
	 * Constructor from params
	 * @param brandId
	 * @param quickRepliesString
	 * @param timestamp
	 * @param originatorId
	 * @param sequence
	 * @param show
	 */
	public QuickRepliesMessageHolder(String brandId, String quickRepliesString, long timestamp, String originatorId, int sequence, boolean show) {
		mQuickRepliesString = quickRepliesString;
		mTimestamp = timestamp;
		mOriginatorId = originatorId;
		mSequence = sequence;
		mShow = show;
		mBrandId = brandId;
	}

	/**
	 * Create a QuickRepliesMessageHolder object from the given ContentEventNotification
	 * @param notification
	 * @return a new created QuickRepliesMessageHolder object or null if quick reply string does not exist
	 */
	public static QuickRepliesMessageHolder fromContentEventNotification(String brandId, ContentEventNotification notification) {

		if (TextUtils.isEmpty(notification.event.quickRepliesJsonString)) {
			return null;
		}

		return new QuickRepliesMessageHolder(brandId, notification.event.quickRepliesJsonString, notification.serverTimestamp, notification.originatorId, notification.sequence, true);
	}

	/**
	 * Create a QuickRepliesMessageHolder object from the given JSON string
	 * @param jsonString
	 * @return a new created QuickRepliesMessageHolder object or null if quick has a JSON error
	 */
	public static QuickRepliesMessageHolder fromJsonString(String brandId, String jsonString) {

		try {
			JSONObject obj = new JSONObject(jsonString);
			return new QuickRepliesMessageHolder(brandId, obj.getString(QUICK_REPLIES_KEY), obj.getLong(TIMESTAMP_KEY), obj.getString(ORIGINATOR_ID_KEY), obj.getInt(SEQUENCE_KEY), obj.getBoolean(SHOW_KEY));
		} catch (JSONException e) {
			return null;
		}
	}


	/**
	 * Convert this quick replies object to a JSON string
	 * @return
	 */
	public String getJsonString(){

		if (mQuickRepliesString == null) {
			return "";
		}

		try {
			JSONObject obj = new JSONObject();
			obj.put(QUICK_REPLIES_KEY, mQuickRepliesString);
			obj.put(TIMESTAMP_KEY, mTimestamp);
			obj.put(ORIGINATOR_ID_KEY, mOriginatorId);
			obj.put(SEQUENCE_KEY, mSequence);
			obj.put(SHOW_KEY, mShow);

			return obj.toString();

		} catch (JSONException e) {
			LPLog.INSTANCE.w(TAG, "getJsonString: error parsing quick reply json", e);
			return "";
		}
	}

	/**
	 * This checks if the contained quick replies string is not null.
	 * @return true - quick replies string is not null, false if quick replies string is null
	 */
	public boolean isValid(){
		return (mQuickRepliesString != null) && (mShow);
	}

	/**
	 * Write this object (converted to JSON string) to shared preferences with the given brandId
	 */
	public void writeToSharedPreferences() {

		// Save in SharedPrefs only if it's newer than the one already saved
		QuickRepliesMessageHolder currentStoredQuickRepliesMessageHolder = QuickRepliesMessageHolder.loadFromSharedPreferences(mBrandId);

		if (this.newerOrEqualThan(currentStoredQuickRepliesMessageHolder)) {

			// Encrypt message before saving
			String encryptedJson = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, getJsonString());

			PreferenceManager.getInstance().setStringValue(QUICK_REPLIES_SHARED_PREFERENCES_KEY, mBrandId, encryptedJson);
		}
	}

	public void deleteFromSharedPreferences(){
		PreferenceManager.getInstance().remove(QUICK_REPLIES_SHARED_PREFERENCES_KEY, mBrandId);
	}

	public static QuickRepliesMessageHolder loadFromSharedPreferences(String brandId) {

		String encryptedJsonString = PreferenceManager.getInstance().getStringValue(QUICK_REPLIES_SHARED_PREFERENCES_KEY, brandId, null);
		String jsonString = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, encryptedJsonString);

		if (jsonString == null) {
			return null;
		}

		return QuickRepliesMessageHolder.fromJsonString(brandId, jsonString);

	}

	public String getQuickRepliesString() {
		return mQuickRepliesString;
	}

	public String getOriginatorId() {
		return mOriginatorId;
	}

	public long getTimestamp() {
		return mTimestamp;
	}

	public boolean isShow() {
		return mShow;
	}

	public void setShow(boolean show) {
		mShow = show;

		// Update this in SharedPrefs
		writeToSharedPreferences();
	}

	public int getSequence() {
		return mSequence;
	}

	/**
	 * Checks if this QuickRepliesMessageHolder is newer than the given one
	 * @param quickRepliesMessageHolder
	 * @return true if this is newer than the given, false if older or equal
	 */
	public boolean newerThan(QuickRepliesMessageHolder quickRepliesMessageHolder) {

		if (quickRepliesMessageHolder == null) {
			return true;
		}

		return (this.mSequence > quickRepliesMessageHolder.mSequence);
	}

	/**
	 * Checks if this QuickRepliesMessageHolder is newer or equal than the given one
	 * @param quickRepliesMessageHolder
	 * @return true if this is newer than the given, false if older or equal
	 */
	public boolean newerOrEqualThan(QuickRepliesMessageHolder quickRepliesMessageHolder) {

		if (quickRepliesMessageHolder == null) {
			return true;
		}

		return (this.mSequence >= quickRepliesMessageHolder.mSequence);
	}

	public static void updateQuickReplies(String brandId, String quickReplies) {
		PreferenceManager.getInstance().setStringValue(QuickRepliesMessageHolder.QUICK_REPLIES_SHARED_PREFERENCES_KEY, brandId, quickReplies);
	}

	@Override
	public String toString() {
		return "QuickRepliesMessageHolder{" +
				"mQuickRepliesString='" + mQuickRepliesString + '\'' +
				", mOriginatorId='" + mOriginatorId + '\'' +
				", mSequence=" + mSequence +
				", mTimestamp=" + mTimestamp +
				", mShow=" + mShow +
				", mBrandId='" + mBrandId + '\'' +
				'}';
	}
}
