package com.liveperson.messaging.model;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.utils.TokenUtils;

/**
 * Created by Ilya Gazman on 11/26/2015.
 * <p/>
 * A command to perform a synchronized connection and provide a callback
 */
public class SynchronizedAuthenticationCompletedCallback implements Command {
    private static final String TAG = "SynchronizedAuthenticationCompletedCallback";
    private static final String ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to make request.";
    private final AccountsController mAccountController;

    private String mBrandId;
    private boolean handled = false;
    private LocalBroadcastReceiver mLocalBroadcastReceiver;
    private PreferenceManager mPreferenceManager;
    private ICallback<Void, Exception> mCallback;

    /**
     * Synchronously listening to when connection is accomplished
     *
     * @param accountsController
     * @param callback
     */
    public SynchronizedAuthenticationCompletedCallback(AccountsController accountsController, String brandId, ICallback<Void, Exception> callback) {
        mAccountController = accountsController;
        mBrandId = brandId;
        mCallback = callback;
        mPreferenceManager = PreferenceManager.getInstance();
    }

    @Override
    public void execute() {
        executeWithReturnValue();
    }

    public boolean executeWithReturnValue() {
        if (isValidAuthentication()) {
            handleConnection();
            return true;
        } else {
            registerToConnectionStateChanges();
            validateStatusDidNotChangedDuringRegistration();
            return false;
        }
    }

    private boolean isValidAuthentication() {
        if (mAccountController.getAccount(mBrandId) == null) {
           String localToken = getTokenFromSharedPreferences();
            if (TokenUtils.isJwtExpired(localToken)) {
                return false;
            } else if (localToken != null) {
                return true;
            }
        }
        return mAccountController.getAccount(mBrandId) != null && mAccountController.getAccount(mBrandId).isAuthenticatedCompleted() && !mAccountController.getAccount(mBrandId).isTokenExpired();
    }

    /**
     * Get the token that is stored in the shared preferences
     *
     * @return
     */
    private String getTokenFromSharedPreferences() {

        String token;
        String decryptedToken = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN_ENC, mBrandId, null);
        if (TextUtils.isEmpty(decryptedToken)) {
            token = mPreferenceManager.getStringValue(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId, null);
            mPreferenceManager.remove(AmsAccount.KEY_ACCOUNT_TOKEN, mBrandId);
        } else {
            token = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedToken);
        }

        return token;
    }

    private void registerToConnectionStateChanges() {
        try {
            if (mLocalBroadcastReceiver == null) {
                mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
                        .addAction(AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION)
                        .addAction(AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR)
                        .addAction(AmsConnection.BROADCAST_AMS_TOKEN_EXPIRED)
                        .build((context, intent) -> {
                            if (AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION.equals(intent.getAction())) {
                                handleConnection();
                            } else if (AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR.equals(intent.getAction())) {
                                handleError("Error: Failed to connect to the server");
                            } else if (AmsConnection.BROADCAST_AMS_TOKEN_EXPIRED.equals(intent.getAction())) {
                                handleError("Error: Token expired, refresh the token and try again");
                            }
                        });
            }
            mLocalBroadcastReceiver.register();
        } catch (Exception error) {
            LPLog.INSTANCE.e(TAG, "registerToConnectionStateChanges: Failed to register", error);
        }
    }

    private synchronized void validateStatusDidNotChangedDuringRegistration() {
        if (!handled) {
            if (isValidAuthentication()) {
                handleConnection();
            }
        }
    }

    private synchronized void handleConnection() {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onSuccess(null);
    }
    private synchronized void handleError(String error) {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " " + error));
    }
}