package com.liveperson.messaging.model;

import com.liveperson.infra.Command;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LocalBroadcastReceiver;

/**
 * This callback is executing only after an internet connection is available. It first checks
 * if a connection available and is so execute immediately. If not it registers a broadcast receiver
 * for internet connection and when it's back it executes.
 *
 * Created by nirni on 1/28/16.
 */
public class SynchronizedInternetConnectionCallback implements Command{

	protected boolean handled = false;
	protected LocalBroadcastReceiver mLocalBroadcastReceiver;
	protected Runnable mCallback;

	/**
	 * Synchronously listening to when connection is accomplished
	 *
	 * @param callback
	 */
	public SynchronizedInternetConnectionCallback(Runnable callback) {
		mCallback = callback;

	}

	@Override
	public void execute() {

		if (InternetConnectionService.isNetworkAvailable()) { // If available execute immediately
			handleConnection();

		} else { // Not available. Register a broadcast receiver to wait for connection

			registerToConnectionStateChanges();
			validateStatusDidNotChangedDuringRegistration();
		}
	}

	private void registerToConnectionStateChanges() {
		mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
				.addAction(InternetConnectionService.BROADCAST_INTERNET_CONNECTION_CONNECTED)
				.build((context, intent) -> handleConnection());
	}

	private synchronized void validateStatusDidNotChangedDuringRegistration() {
		if (!handled) {
			if (InternetConnectionService.isNetworkAvailable()) {
				handleConnection();
			}
		}
	}

	private synchronized void handleConnection() {
		if (handled) {
			return;
		}
		if (mLocalBroadcastReceiver != null) {
			mLocalBroadcastReceiver.unregister();
		}
		handled = true;
		mCallback.run();
	}

}
