package com.liveperson.messaging.model;

import com.liveperson.infra.Command;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.messaging.controller.ConnectionsController;

/**
 * Created by Ilya Gazman on 11/26/2015.
 * <p/>
 * A command to perform a synchronized connection and provide a callback
 */
public class SynchronizedAmsConnectionCallback implements Command {
    private final ConnectionsController mConnectionController;

    private String mBrandId;
    private boolean handled = false;
    private LocalBroadcastReceiver mLocalBroadcastReceiver;
    private Runnable mCallback;

    /**
     * Synchronously listening to when connection is accomplished
     *
     * @param connectionController
     * @param callback
     */
    public SynchronizedAmsConnectionCallback(ConnectionsController connectionController, String brandId, Runnable callback) {
        mConnectionController = connectionController;
        mBrandId = brandId;
        mCallback = callback;

    }

    @Override
    public void execute() {
        if (mConnectionController.isSocketReady(mBrandId)) {
            handleConnection();
        } else {
            registerToConnectionStateChanges();
            validateStatusDidNotChangedDuringRegistration();
        }
    }

    private void registerToConnectionStateChanges() {
        mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
                .addAction(AmsConnection.BROADCAST_KEY_SOCKET_READY_ACTION)
                .build((context, intent) -> {
                    boolean isSocketReady = intent.getBooleanExtra(AmsConnection.BROADCAST_KEY_SOCKET_READY_EXTRA, false);
                    if (isSocketReady){
                        handleConnection();
                    }
                });
    }

    private synchronized void validateStatusDidNotChangedDuringRegistration() {
        if (!handled) {
            if (mConnectionController.isSocketReady(mBrandId)) {
                handleConnection();
            }
        }
    }

    private synchronized void handleConnection() {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.run();
    }
}