package com.liveperson.messaging;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.api.LivePersonCallback;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.PushType;
import com.liveperson.infra.PushUnregisterType;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.model.types.ChatState;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.messaging.background.filesharing.FileSharingType;
import com.liveperson.messaging.model.AgentData;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.model.UserProfile;

/**
 * Created by ofira on 03/02/2016.
 */

public interface IMessaging {

    void init(Context context, MessagingInitData initData, InitLivePersonCallBack callBack);
    void init(Context context,  MessagingInitData initData, InitProcess entryPoint);

    void clearAllConversationData(String targetId);

    void shutDown(ShutDownProcess process);

	/**
	 * Lite logout when a different consumer id detected in response of {@link com.liveperson.messaging.commands.tasks.IdpTask}
	 *
	 * @param brandId       The brandId
	 * @param oldConsumerId The old consumerId
	 * @param newConsumerId The new consumerId
	 */
	void liteLogout(String brandId, String oldConsumerId, String newConsumerId);
    void logout(Context context, MessagingInitData initData, boolean forceLogOut, PushUnregisterType type, LogoutProcess logoutProcess);
    boolean isInitialized();

    void sendMessage(String targetId, String brandId, String message, @Nullable DeliveryStatusUpdateInfo info);
    void sendMessageWithURL(String targetId, String brandId, String message, String urlToParse, String title, String description, String imageURL, String siteName);
	void sendFormSubmissionMessageCommand(final String invitationId);

    int resendMessage(String eventId, String conversationId, MessagingChatMessage.MessageType messageType, boolean offlineFlow);
    int resendMessage(String eventId, String conversationId, long fileRowId, MessagingChatMessage.MessageType messageType, boolean offlineFlow);

	void updateMessage(final String invitationId, final String conversationId, final MessagingChatMessage.MessageType messageType, final MessagingChatMessage.MessageState messageState);

	void registerPusher(String brandId, String appId, String token, PushType pushType, LPAuthenticationParams authenticationParams, final ICallback<Void, Exception> registrationCompletedCallback);

	/**
	 * Unregister pusher for the old consumer.
	 *
	 * @param brandId    The brandId
	 * @param consumerId The old consumerId
	 */
	void unregisterPusherOnLiteLogout(String brandId, String consumerId);
    void unregisterPusher(String brandId, String appId, PushUnregisterType type, ICallback<Void, Exception> callback, boolean isFullLogout);
	void updateTokenInBackground(String brandId, LPAuthenticationParams authenticationParams);
    ActionFailureReason changeChatState(String targetId, String brandId, ChatState state);
    ActionFailureReason closeDialog(String brandId);
    ActionFailureReason markConversationAsUrgent(String targetId, String brandId);
    ActionFailureReason markConversationAsNormal(String targetId, String brandId);
    void sendCSAT(String brandId, String conversationID, int mStarsNumber, int yesNoValue);
    void sendUserProfile(String brandId, UserProfile userProfile);
    void setCallback(LivePersonCallback listener);
    void removeCallback();
    void checkActiveConversation(String brandId, ICallback<Boolean, Exception> callback);
    void checkConversationIsMarkedAsUrgent(String brandId, ICallback<Boolean, Exception> callback);
    void checkAgentID(String brandId, ICallback<AgentData, Exception> callback);
	boolean clearHistory(String targetId);
    MaskedMessage getMaskedMessage(String brandId, String message);
	void generateUploadToken(String formId, String brandId, String invitationId);

    void sendFileMessage(FileSharingType fileSharingType, String brandId, String targetId,String imageUriString, String message, boolean imageFromCamera);
    void downloadFile(FileSharingType fileSharingType, String brandId, String targetId, String imageSwiftPath, long messageRowId, long fileRowId, String conversationId) ;

    void removeMultipleOlderImages(String brandId);

    boolean isSocketOpen(String brandId);
    void connect(String brandId, LPAuthenticationParams lpAuthenticationParams, ConversationViewParams conversationViewParams, boolean connectInBackground);
    void connect(String brandId,  LPAuthenticationParams lpAuthenticationParams, ConversationViewParams conversationViewParams);
	void reconnect(String brandId,  LPAuthenticationParams lpAuthenticationParams);
	void moveToBackground(String brandId, long timeout);
	void moveToForeground(String brandId,  LPAuthenticationParams lpAuthenticationParams, ConversationViewParams conversationViewParams);
    void onConversationDestroyed(String brandId);
	void serviceStarted(String brandId);
	void serviceStopped(String brandId);
    void getUnreadMessagesCount(String brandId, String appId, LPAuthenticationParams authenticationParams, ICallback<Integer, Exception> callback);
    void getNumUnreadMessages(String brandId, String appId, ICallback<Integer, Exception> callback);
    void isPusherRegistered(String brandId, String deviceToken, String appId, LPAuthenticationParams authenticationParams, @NonNull ICallback<Boolean, Exception> callback);
}
