package com.liveperson.messaging.model;

import android.os.Bundle;
import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.request.message.FilePublishMessage;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.Event;
import com.liveperson.api.response.model.Participants;
import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.Message;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.background.filesharing.FileExtensionTypes;

import org.json.JSONException;
import org.json.JSONObject;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000009F;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A0;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A1;

/**
 * Created by ofira on 09/07/2015.
 *
 * Modified by nirni
 */
public class MessagingChatMessage extends Message {

	private static final String TAG = "MessagingChatMessage";


    public static final String EXTRA_MESSAGE_STATE = "EXTRA_MESSAGE_STATE";
    public static final String EXTRA_SERVER_SEQUENCE = "EXTRA_SERVER_SEQUENCE";
    public static final String EXTRA_DIALOG_ID = "EXTRA_DIALOG_ID";

    public enum MessageState {
        //MessageState order: ERROR, PENDING, QUEUED, SENT, READ, VIEWED+SUBMITTED+EXPIRED(only for pci invitation)
        PENDING,
        SENT,
        RECEIVED,
        READ,
        ERROR,
        QUEUED,
        VIEWED,
        SUBMITTED,
        EXPIRED,
        OFFLINE;

        //The problem is with QUEUED state that break the ordinal
        public static boolean validChange(MessageState oldVal, MessageState newVal){
            return oldVal == OFFLINE || oldVal == PENDING || oldVal == ERROR ||
                (oldVal == QUEUED && newVal != PENDING) ||
                (newVal.ordinal() > oldVal.ordinal() && newVal != QUEUED);
        }

        /**
         * @return if the message is in status received and not read
         * (READ and ERROR status are "read" status)
         */
        public boolean isReceivedMessageNotRead() {
            return this == MessagingChatMessage.MessageState.RECEIVED;
        }
    }

    // ****** PLEASE CHECK IF NEW TYPES MATCHES ANY OF THE FUNCTIONS BELOW!! ******** //
    public enum MessageType {
        SYSTEM_RESOLVED, SYSTEM_DIALOG_RESOLVED, SYSTEM_MASKED, LOADING, UNREAD_INDICATOR, CONTROLLER_SYSTEM, BRAND,
        CONSUMER, CONSUMER_URL, CONSUMER_FORM, CONSUMER_MASKED, CONSUMER_IMAGE, CONSUMER_DOCUMENT, CONSUMER_IMAGE_MASKED, CONSUMER_URL_MASKED, CONSUMER_VOICE,
        AGENT, AGENT_URL, AGENT_IMAGE, AGENT_DOCUMENT, AGENT_FORM, AGENT_STRUCTURED_CONTENT, AGENT_IS_TYPING_INDICATOR, AGENT_QUICK_REPLIES, AGENT_MARKDOWN_HYPERLINK, COBROWSE, DATE_HEADER;
    // ****** PLEASE CHECK IF NEW TYPES MATCHES ANY OF THE FUNCTIONS BELOW!! ******** //
        //PLEASE ADD NEW AGENT TYPE ALSO TO isAgent()
        //PLEASE ADD NEW MASKED CONSUMER MESSAGE TYPE ALSO TO isConsumerMaskedMessage()

        public static MessageType getMessageContentTypeForConsumer(ContentEventNotification event, ContentType messageContentType) {
            switch (messageContentType) {
            case hosted_file:
                // Get the message type according to the file extension
                MessageType fileMessageType = getFileMessageType(event);
                if (fileMessageType != null) {
                    return fileMessageType;
                }

                LPLog.INSTANCE.e(TAG, ERR_0000009F, "getMessageContentTypeForConsumer: file extension not recognized. Return CONSUMER type");
                return CONSUMER;
            case forms_secure_submission:
                return CONSUMER_FORM;
            case text_plain:
            case text_html:
            default:
                return CONSUMER;

            }
        }

		/**
		 * Get the message type of a "hosted_file" contentType according to the file extension.
		 * Currently the supported types are Image and Voice
		 *
		 * @param eventNotification
		 * @return
		 */
        @Nullable
        private static MessageType getFileMessageType(ContentEventNotification eventNotification) {
            Event event = eventNotification.event;
            if (event.message instanceof FilePublishMessage) {
                String relativePath = ((FilePublishMessage)event.message).getRelativePath();
                String extension = "";

                int i = relativePath.lastIndexOf('.');
                if (i >= 0) {
                    extension = relativePath.substring(i+1).toLowerCase();
                }

                LPLog.INSTANCE.d(TAG, "getMessageContentTypeForConsumer: file extension: " + extension);

                if (FileExtensionTypes.getImageExtensions().contains(extension)) {
                    return eventNotification.originatorMetadata.mRole == Participants.ParticipantRole.CONSUMER ? CONSUMER_IMAGE : AGENT_IMAGE;
                }

                if(FileExtensionTypes.getDocumentExtensions().contains(extension)){
                    return eventNotification.originatorMetadata.mRole == Participants.ParticipantRole.CONSUMER ? CONSUMER_DOCUMENT: AGENT_DOCUMENT;
                }

                if (FileExtensionTypes.getVoiceExtensions().contains(extension)) {
                    return CONSUMER_VOICE;
                }
            }
            return null;
        }

        public static MessageType getMessageContentTypeForAgent(ContentEventNotification notification, ContentType messageContentType) {
            switch (messageContentType) {
            case forms_secure_invitation:
                return AGENT_FORM;
            case text_plain:
            case text_html:
            case hosted_file:
                if (messageContentType.isFile()) {
                    // Get the message type according to the file extension
                    MessageType fileMessageType = getFileMessageType(notification);
                    if (fileMessageType != null) {
                        return fileMessageType;
                    }
                    LPLog.INSTANCE.e(TAG, ERR_000000A0, "getMessageContentTypeForAgent: file extension not recognized. Return Agent type");
                }
                return AGENT;
            default:
                return AGENT;

            }
        }

        public static boolean isAgent(MessageType messageType) {
            //maybe in the future there will be more types of agent.
            return messageType == AGENT
                    || messageType == AGENT_URL
                    || messageType == AGENT_FORM
                    || messageType == AGENT_STRUCTURED_CONTENT
                    || messageType == AGENT_IS_TYPING_INDICATOR
                    || messageType == AGENT_DOCUMENT
                    || messageType == AGENT_IMAGE
                    || messageType == AGENT_MARKDOWN_HYPERLINK;
        }

        /**
         * Method used to determine whether consumer message is not masked
         * @param messageType type to check
         * @return true if type is consumer, but not masked, false otherwise.
         */
        public static boolean isConsumerUnmasked(MessageType messageType) {
            return MessageType.CONSUMER == messageType
                    || MessageType.CONSUMER_IMAGE == messageType
                    || MessageType.CONSUMER_URL == messageType
                    || MessageType.CONSUMER_DOCUMENT == messageType;
        }

        public static boolean isConsumer(MessageType messageType) {
            return MessageType.CONSUMER == messageType
                    || MessageType.CONSUMER_MASKED == messageType
                    || MessageType.LOADING == messageType
                    || MessageType.CONSUMER_IMAGE == messageType
                    || MessageType.CONSUMER_IMAGE_MASKED == messageType
                    || MessageType.CONSUMER_URL == messageType
                    || MessageType.CONSUMER_URL_MASKED == messageType
                    || MessageType.CONSUMER_FORM == messageType
                    || MessageType.CONSUMER_VOICE == messageType
                    || MessageType.CONSUMER_DOCUMENT == messageType;
        }

        public static boolean isConsumerMaskedMessage(MessageType messageType) {
            return messageType == MessagingChatMessage.MessageType.CONSUMER_MASKED ||
                    messageType == MessagingChatMessage.MessageType.CONSUMER_IMAGE_MASKED||
                    messageType == MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
        }

		public static boolean isSystem(MessageType messageType) {
			return messageType == SYSTEM_RESOLVED || messageType == SYSTEM_DIALOG_RESOLVED || messageType == SYSTEM_MASKED || messageType == BRAND || messageType == CONTROLLER_SYSTEM;
		}
        public static boolean isSystemResolved(MessageType messageType) {
            return messageType == SYSTEM_RESOLVED || messageType == SYSTEM_DIALOG_RESOLVED;
        }
        public static boolean isImage(MessageType messageType) {
            return messageType == CONSUMER_IMAGE || messageType == CONSUMER_IMAGE_MASKED || messageType == AGENT_IMAGE;
        }

        public static boolean isConsumerFile(MessageType messageType) {
            return messageType == CONSUMER_IMAGE || messageType == CONSUMER_IMAGE_MASKED
                    || messageType == CONSUMER_DOCUMENT
                    || messageType == CONSUMER_VOICE;
        }
	}

    private MessageType mMessageType;
    private MessageState mMessageState;

    public MessagingChatMessage(String originatorId, String message, long timeStamp, String dialogId, String eventId, MessageType messageType, MessageState messageState, EncryptionVersion encryptionVersion) {
        super(originatorId, message, timeStamp, dialogId, eventId, encryptionVersion);
        mMessageType = messageType;
        mMessageState = messageState;
    }

    public MessagingChatMessage(String originatorId, String message, long timeStamp, String dialogId, String eventId, MessageType mMessageType, MessageState mMessageState, int serverSequence, String contentType, EncryptionVersion encryptionVersion) {
        this(originatorId, message, timeStamp, dialogId, eventId, mMessageType, mMessageState, encryptionVersion);
        setContentType(contentType);
        setServerSequence(serverSequence);
    }

    public MessagingChatMessage(MessagingChatMessage message) {
        this(message.getOriginatorId(), message.getMessage(), message.getTimeStamp(), message.getDialogId(), message.getEventId(), message.getMessageType(), message.mMessageState, message.getServerSequence(), message.getContentType(), EncryptionVersion.NONE);
        setLocalId(message.getLocalId());
    }

    ///////////////// Getters + Setters ////////////////////////////////

    public void setLocalId(long localId) {
        this.mId = localId;
    }

    public MessageType getMessageType() {
        return mMessageType;
    }

    public void setMessageType(MessageType messageType) {
        mMessageType = messageType;
    }

    public MessageState getMessageState() {
        return mMessageState;
    }

    public void setMessageState(MessageState messageState) {
        mMessageState = messageState;
    }

    public boolean isSystemMessageFromAgent(String myUserId) {
        return ((getMessageType() == MessageType.SYSTEM_RESOLVED || getMessageType() == MessageType.SYSTEM_DIALOG_RESOLVED) && !TextUtils.equals(getOriginatorId(), myUserId));
    }

    public boolean isFileType() {
        return mMessageType == MessageType.CONSUMER_IMAGE
                || mMessageType == MessageType.CONSUMER_DOCUMENT
                || mMessageType == MessageType.CONSUMER_IMAGE_MASKED
                || mMessageType == MessageType.CONSUMER_VOICE
                || mMessageType == MessageType.AGENT_IMAGE
                || mMessageType == MessageType.AGENT_DOCUMENT;
    }

    public Bundle update(MessagingChatMessage newChatMessage) {
        Bundle extras = new Bundle();
        if (mMessageState != newChatMessage.getMessageState()) {
            mMessageState = newChatMessage.getMessageState();
            extras.putInt(EXTRA_MESSAGE_STATE, mMessageState.ordinal());
        }
        if (getServerSequence() != newChatMessage.getServerSequence()) {
            setServerSequence(newChatMessage.getServerSequence());
            extras.putLong(EXTRA_SERVER_SEQUENCE, getServerSequence());
        }
        if (!TextUtils.equals(getDialogId(), newChatMessage.getDialogId())) {
            setDialogId(newChatMessage.getDialogId());
            extras.putString(EXTRA_DIALOG_ID, getDialogId());
        }

        if (!TextUtils.equals(getEventId(), newChatMessage.getEventId())) {
            setEventId(newChatMessage.getEventId());
        }

        setMessageId(newChatMessage.getLocalId());
        setMessageType(newChatMessage.getMessageType());
        setContentType(newChatMessage.getContentType());
        return extras;
    }


    @Override
    public String toString() {
        return "["
                + "MessageRowId " + getLocalId() + "\n"
                + super.toString() + "\n"
                + "MessageType " + mMessageType + "\n"
                + "MessageState " + mMessageState + "]";
    }

    public String getFormalMessage() {
        try {
            switch (mMessageType){
                case AGENT_FORM:
                    JSONObject jsonObject = new JSONObject(getMessage());
                    return jsonObject.getString(FormsManager.TITLE);
				case AGENT_STRUCTURED_CONTENT:
					return Infra.instance.getApplicationContext().getString(R.string.lp_new_message);
                case AGENT_DOCUMENT:
                case AGENT_IMAGE:
                    return getMessage().isEmpty()?Infra.instance.getApplicationContext().getString(R.string.lp_new_message):getMessage();
				default:
                    return getMessage();
            }

        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_000000A1, "JSONException while unpacking Agent Form.", e);
        }
        return getMessage();
    }
}
