package com.liveperson.messaging.network.http

import android.content.Context
import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.callbacks.PKCEParamsCallBack
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.messaging.R
import com.liveperson.infra.model.JWKInfo
import com.liveperson.infra.model.errors.PkceGenerateError
import com.liveperson.infra.model.types.PkceGenerateFailureReason
import com.liveperson.infra.network.http.requests.CSDSRequest
import com.liveperson.infra.network.http.requests.JWKSRequest
import com.liveperson.infra.utils.PKCEUtils
import java.util.HashMap

class GeneratePKCEParamsRequest(val context: Context, val brandId: String,
                                val lpAuthenticationParams: LPAuthenticationParams?,
                                val callback: PKCEParamsCallBack) : Command {
    companion object {
        private const val TAG = "GeneratePKCEParamsRequest"
        private const val LP_IDP_SERVICE_NAME = "idp"
    }

    private var csdsDomain: String? = null

    init {
        csdsDomain = if (brandId.startsWith("qa") || brandId.startsWith("le")) {
            context.resources.getString(R.string.csds_qa_url)
        } else {
            context.resources.getString(R.string.csds_url)
        }
    }

    override fun execute() {
        // set pinning keys here
        CSDSRequest(csdsDomain, brandId, LP_IDP_SERVICE_NAME,
            lpAuthenticationParams?.certificatePinningKeys,
            object : ICallback<HashMap<String, String>, Exception> {
            override fun onSuccess(csdsUrls: HashMap<String, String>?) {
                if (!csdsUrls.isNullOrEmpty()) {
                    LPLog.i(TAG, csdsUrls.toString())

                    val lpIdpDomain = csdsUrls[LP_IDP_SERVICE_NAME]
                    if (lpIdpDomain != null) {
                        JWKSRequest(lpIdpDomain, object : ICallback<JWKInfo, java.lang.Exception> {
                            override fun onSuccess(jwkInfo: JWKInfo?) {
                                if (jwkInfo != null) {
                                    try {
                                        val params = PKCEUtils.generatePKCEParams(jwkInfo.kId, jwkInfo.jwk)
                                        callback.onPKCEGenerateSuccess(params)
                                        return
                                    } catch (e: Exception) {
                                        callback.onPKCEGenerateFailed(PkceGenerateError(
                                            PkceGenerateFailureReason.GENERATE_JWE,
                                            e))
                                    }
                                } else {
                                    callback.onPKCEGenerateFailed(PkceGenerateError(
                                        PkceGenerateFailureReason.LP_JWKS,
                                        Exception("Failed to get JWK - null value")
                                    ))
                                }
                            }

                            override fun onError(exception: Exception?) {
                                exception.let {
                                    LPLog.e(TAG, ErrorCode.ERR_0000016C, "JWKSRequest onError: ", it)
                                }
                                callback.onPKCEGenerateFailed(PkceGenerateError(
                                    PkceGenerateFailureReason.LP_JWKS,
                                    exception))
                            }

                        }).execute()
                    }
                } else {
                    LPLog.e(TAG, ErrorCode.ERR_0000016C, "CSDSRequest: Received empty CSDS domain map")
                    callback.onPKCEGenerateFailed(PkceGenerateError(
                        PkceGenerateFailureReason.CSDS_DOMAIN,
                        Exception("Failed to get CSDS domain")
                    ))
                }
            }

            override fun onError(exception: Exception?) {
                exception.let {
                    LPLog.e(TAG, ErrorCode.ERR_0000016C, "CSDSRequest onError: ", it)
                }
                callback.onPKCEGenerateFailed(PkceGenerateError(
                    PkceGenerateFailureReason.CSDS_DOMAIN,
                    exception))
            }
        }).execute()
    }
}