package com.liveperson.messaging.network.socket.requests;

import androidx.annotation.Nullable;

import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.api.request.PublishEvent;
import com.liveperson.api.request.message.PublishMessage;
import com.liveperson.api.request.message.TextPublishMessage;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.BaseResponseHandler;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.ConversationUtils;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * A request to send message
 */
public class SendMessageRequest extends BaseAMSSocketRequest<PublishEvent.Response, SendMessageRequest> {

    private static final String TAG = "SendMessageRequest";

    private final String mTargetId;
    private final String mEventId;
    private String mDialogId;
    private String mConversationId;
	protected PublishMessage mMessage;
    private PublishEvent.Type mType = PublishEvent.Type.ContentEvent;
	private ContentType mContentType = ContentType.text_plain;
    private Messaging mController;
	private DeliveryStatusUpdateInfo mInfo = null;


	/**
	 * This constructor gets the requestId as one of its parameters and override with it the request's one
	 * @param brandId
	 * @param dialogId
	 * @param msg
	 */
	public SendMessageRequest(Messaging controller, String eventId, String targetId, String brandId, String conversationId, String dialogId,String msg, ContentType contentType) {
		super(controller.mAccountsController.getConnectionUrl(brandId));
		mController = controller;
        mDialogId = dialogId;
        mConversationId = conversationId;
		mMessage = new TextPublishMessage(msg);
		mTargetId = targetId;
		mEventId = eventId;
		mContentType = contentType;
	}

    /**
	 * This constructor gets the requestId as one of its parameters and override with it the request's one
	 * @param brandId
	 * @param dialogId
	 * @param msg
	 */
    public SendMessageRequest(Messaging controller, String eventId, String targetId, String brandId, String dialogId, String msg, String conversationId) {
        super(controller.mAccountsController.getConnectionUrl(brandId));
        mController = controller;
        mDialogId = dialogId;
        mConversationId = conversationId;
        mMessage = new TextPublishMessage(msg);
        mTargetId = targetId;
        mEventId = eventId;
    }

	public SendMessageRequest(Messaging controller, String eventId, String targetId, String brandId, String dialogId, String conversationId) {
		super(controller.mAccountsController.getConnectionUrl(brandId));
		mController = controller;
        mDialogId = dialogId;
        mConversationId = conversationId;
		mTargetId = targetId;
		mEventId = eventId;
	}
    public void setMessageContent(String msg){
        mMessage = new TextPublishMessage(msg);
    }

	public void setInfo(@Nullable DeliveryStatusUpdateInfo info){
		LPLog.INSTANCE.d(TAG, "setInfo: Adding DeliveryStatusUpdateInfo with metadata: " + ((info == null) ? "null" : LPLog.INSTANCE.mask(info.getMetadata())));
		mInfo = info;
	}

    public SendMessageRequest setConversationId(String conversationId) {
        mConversationId = conversationId;
        return this;
    }

	public SendMessageRequest setDialogId(String dialogId) {
        mDialogId = dialogId;
        return this;
    }

    public SendMessageRequest setContentType(ContentType contentType) {
        mContentType = contentType;
        return this;
    }

    public String getEventId() {
        return mEventId;
    }

    @Override
    protected String getData() {
        return new PublishEvent(mDialogId, mConversationId, mMessage, mType, mContentType, mEventId, mInfo).toJsonString(getRequestId());
    }

    @Override
    protected String getRequestName() {
        return TAG;
    }

    @Override
    protected BaseResponseHandler<PublishEvent.Response, SendMessageRequest> getResponseHandler() {
        return new BaseResponseHandler<PublishEvent.Response, SendMessageRequest>() {

            @Override
            public String getAPIResponseType() {
                return PublishEvent.Response.PUBLISH_EVENT_RESPONSE_TYPE;
            }

            @Override
            protected PublishEvent.Response parse(JSONObject jsonObject) throws JSONException {
                return new PublishEvent.Response(jsonObject);
            }

            @Override
            protected boolean handle(PublishEvent.Response data) {
                final int serverSequence = data.getBody().sequence;
	            LPLog.INSTANCE.i(TAG, "Got send message response eventId = " + mEventId + ", with sequence = "+ serverSequence);
                mController.amsMessages.updateOnMessageAck(mEventId, serverSequence);
                //mController.amsConversations.updateLastServerSequenceByConvId(mDialogId, serverSequence);
                mController.amsDialogs.updateLastServerSequenceByDialogId(mDialogId, serverSequence);

				// Remove the message from the messageTimeoutQueue (we received an ack on the message)
				mController.amsMessages.mMessageTimeoutQueue.remove((int) getRequestId());

				// Show TTR if needed
				ConversationUtils.showTTR(mController, mTargetId);

                return true;
            }

            @Override
            protected void giveUp() {
                super.giveUp();
                LPLog.INSTANCE.i(TAG, getRequestId() + ": Request lost (socket closed) for send message request.");
                // Change message status to failed from the messageTimeoutQueue
                DataBaseExecutor.execute(() -> mController.getOfflineManager().removePendingOfflineMessage(mController.getActiveBrandId(), getEventId()));
                mController.amsMessages.mMessageTimeoutQueue.remove((int) getRequestId());
            }
        };
    }
}
