package com.liveperson.messaging.offline

import com.liveperson.infra.ICallback
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.messaging.R
import com.liveperson.infra.model.LPWelcomeMessage
import com.liveperson.infra.network.socket.SocketManager
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.Conversation
import com.liveperson.messaging.model.ConversationUtils
import com.liveperson.messaging.model.Dialog
import com.liveperson.messaging.model.FullMessageRow
import com.liveperson.messaging.model.MessagingChatMessage
import com.liveperson.messaging.network.socket.requests.NewConversationRequest
import com.liveperson.messaging.offline.api.OfflineMessagesController

/**
 * Class responsible for determining the configuration of the offline messaging feature,
 * for sending offline messages and for showing the offline welcome message if required
 */
internal class OfflineMessagesControllerImpl internal constructor(
    private val mMessaging: Messaging
) : OfflineMessagesController {

    @Volatile
    override var isSubscribedToActiveDialog = false

    @Volatile
    override var isFullySynced: Boolean = false

    override val currentBrandId: String
        get() = mMessaging.activeBrandId

    override val isOfflineModeEnabled: Boolean
        get() = Configuration.getBoolean(R.bool.lp_is_offline_messaging_enabled)

    override val isOfflineWelcomeMessageEnabled: Boolean
        get() = Configuration.getBoolean(R.bool.lp_is_offline_welcome_message_enabled)

    override val offlineModePCSBehavior: Int
        get() = Configuration.getInteger(R.integer.lp_offline_messages_for_pcs_behavior)

    private val mConversationUtils = ConversationUtils(mMessaging)

    override fun resolveActiveDialog(dialogId: String, callback: ICallback<Int, Exception>?) {
        mMessaging.closeDialog(currentBrandId, dialogId, callback)
    }

    override fun sendNewConversationRequest(requestId: Long) {
        val brandId: String = currentBrandId
        val newConversationRequest = NewConversationRequest(
            mMessaging,
            brandId,
            brandId,
            Conversation.TEMP_CONVERSATION_ID,
            Dialog.TEMP_DIALOG_ID
        )
        newConversationRequest.setRequestId(requestId)
        mMessaging.amsConversations.enqueuePendingConversationRequestId(requestId)
        SocketManager.getInstance().send(newConversationRequest)
    }

    override fun showOfflineWelcomeMessage(
        brandId: String,
        dialogId: String,
        welcomeMessage: LPWelcomeMessage
    ) {
        mConversationUtils.showWelcomeMessage(
            brandId,
            welcomeMessage,
            dialogId,
            MessagingChatMessage.MessageState.OFFLINE
        )
    }

    override fun sendOfflineMessage(message: FullMessageRow) {
        val chatMessage = message.messagingChatMessage
        val fileMessage = message.fileMessage
        if (fileMessage == null) {
            mMessaging.resendMessage(
                chatMessage.eventId,
                chatMessage.dialogId,
                chatMessage.messageType,
                true
            )
        } else {
            mMessaging.resendMessage(
                chatMessage.eventId,
                chatMessage.dialogId,
                fileMessage.fileRowId,
                chatMessage.messageType,
                true
            )
        }
    }
}
