package com.liveperson.messaging.utils

import android.database.Cursor

/**
 * Extension function over Cursor class to read value
 * by required type and close cursor.
 *
 * @param block lambda expression to read a value from cursor.
 *
 * @see readBoolean to get a boolean value from cursor.
 * @see readFloat to get a float value from cursor.
 * @see readInt to get a integer value from cursor.
 * @see readLong to get a long value from cursor.
 * @see readString to get a string value from cursor.
 */
internal inline fun<reified T: Any> Cursor.readValue(valueProvider: (Cursor) -> T): T? = use {
    it.takeIf { it.moveToFirst() }?.let(valueProvider)
}

internal fun Cursor.readBoolean(index: Int = 0, defaultValue: Boolean): Boolean = readValue { it.getInt(index) > 0} ?: defaultValue

internal fun Cursor.readFloat(index: Int = 0, defaultValue: Float): Float = readValue { it.getFloat(index) } ?: defaultValue

internal fun Cursor.readInt(index: Int = 0, defaultValue: Int): Int = readValue { it.getInt(index) } ?: defaultValue

internal fun Cursor.readLong(index: Int = 0, defaultValue: Long): Long = readValue { it.getLong(index) } ?: defaultValue

internal fun Cursor.readString(index: Int = 0, defaultValue: String): String = readValue { it.getString(index) } ?: defaultValue

/**
 * Method used to receive value from cursor in typesafe way
 * or return a default value.
 */
internal inline fun<T> Cursor?.getOrDefault(defaultValue: T, block: (Cursor) -> T): T {
    return this?.let(block) ?: defaultValue
}

