package com.liveperson.messaging.utils

import android.content.ContentValues
import android.content.SharedPreferences.Editor

/**
 * A typesafe method to write a value to shared preferences.
 * Note: if value type is not supported UnsupportedOperationException would be thrown.
 * Supported types:
 * - Boolean
 * - Integer
 * - Long
 * - Float
 * - String
 */
internal fun Editor.write(key: String, value: Any?) = apply {
    when (value) {
        null -> remove(key)
        is Boolean -> putBoolean(key, value)
        is Int -> putInt(key, value)
        is Long -> putLong(key, value)
        is Float -> putFloat(key, value)
        is String -> putString(key, value)
        is Set<*> -> putStringSet(key, value.map { it.toString() }.toSet())
        else -> throw UnsupportedOperationException()
    }
}

/**
 *  A typesafe method to write content values to shared preferences.
 *  This method iterates over content values key pairs and writes
 *  each value by its key to preferences.
 *  @see write(String, Any?)
 */
internal fun Editor.write(contentValues: ContentValues) = apply  {
    contentValues.valueSet().forEach { (key, value) ->
        write(key, value)
    }
}