package com.liveperson.messaging.wm.impl

import android.content.ContentResolver
import android.content.Context
import android.content.SharedPreferences
import android.content.UriMatcher
import android.database.MatrixCursor
import android.net.Uri
import com.liveperson.messaging.wm.core.exceptions.throwTypeError
import com.liveperson.messaging.wm.core.exceptions.throwTypeErrorOf

const val AUTHORITY_PREFIX = "com.liveperson.wm."

const val COMMON_TYPE = "type"
const val COMMON_KEY = "key"

private const val MATCH_DATA = 0x010000

private const val INT_TYPE = "integer"
private const val LONG_TYPE = "long"
private const val FLOAT_TYPE = "float"
private const val BOOLEAN_TYPE = "boolean"
private const val STRING_TYPE = "string"

/**
 * Extension property over context to get
 * an appropriate authority to access welcome message
 * data from welcome message provider
 */
private val Context.welcomeMessageProviderAuthority
    get() = "$AUTHORITY_PREFIX$packageName"

/**
 * Extension property over context to create to get
 * an appropriate mime to access welcome message
 * data from welcome message provider
 */
internal val Context.welcomeMessageMimeType
    get() = "${ContentResolver.CURSOR_ITEM_BASE_TYPE}/vnd.$welcomeMessageProviderAuthority.item"

/**
 * Extension function over context to create uri
 * for data retrieval from welcome message provider.
 */
private fun Context.createWelcomeMessageProviderUri(): Uri {
    return Uri.parse("content://$welcomeMessageProviderAuthority")
}

/**
 * Extension function over to create a uri
 * matcher to interact with stored data using
 * welcome message content provider.
 */
internal val Context.welcomeMessageUriMatcher
    get() = UriMatcher(UriMatcher.NO_MATCH).apply {
        addURI(
            welcomeMessageProviderAuthority,
            "*/*",
            MATCH_DATA
        )
    }

/**
 * Method used to create a uri that's associated with WelcomeMessageContentProvider to
 * store, edit or delete dynamic welcome message.
 *
 * @param key key of stored share preference
 * @param type type of stored shared preference
 *
 * Note: There are string value of of supported types:
 * - "integer" for int values
 * - "long" for long values
 * - "float" for float values
 * - "boolean" for bool values
 * - "string" for string values
 * If provided string that represents a type is not one of listed about
 * the UnsupportedTypeException would be thrown.
 */
internal fun Context.contentUriOf(key: String, type: String): Uri =
    createWelcomeMessageProviderUri().buildUpon().appendPath(key).appendPath(type).build()

/**
 * Extension function used to validate request to ContentProvider
 * whether executed request is associated
 * with dynamic welcome message flow.
 *
 * @param block - block that will be executed if uri passes validation
 */
internal inline fun <T> Uri.onMatched(matcher: UriMatcher, block: (Uri) -> T): T? =
    takeIf { matcher.match(it) == MATCH_DATA }?.let(block)

/**
 * Extension function used to receive key and type for required preference
 * from URI.
 *
 * @param block - block that will be executed if uri passes validation
 * and path segments could be read successfully.
 */
internal inline fun <T> Uri.onPathMatched(
    matcher: UriMatcher,
    block: (key: String, type: String) -> T
): T? = onMatched(matcher) {
    val pathSegments = it.pathSegments
    block(pathSegments.first(), pathSegments.last())
}

/**
 * Extension function used to map a requested by key value to matrix cursor
 * required by ContentProvider API.
 *
 * @param key a key associated with preference
 * @param type type of requested preference.
 *
 * Note: There are string value of of supported types:
 * - "integer" for int values
 * - "long" for long values
 * - "float" for float values
 * - "boolean" for bool values
 * - "string" for string values
 * If provided string that represents a type is not one of listed about
 * the UnsupportedTypeException would be thrown.
 */
internal fun SharedPreferences.getValueAsCursor(key: String, type: String): MatrixCursor {
    val cursor = MatrixCursor(arrayOf(key))
    val row = cursor.newRow()

    val value = when (type) {
        INT_TYPE -> getInt(key, -1)
        LONG_TYPE -> getLong(key, -1L)
        FLOAT_TYPE -> getFloat(key, -1f)
        BOOLEAN_TYPE -> getBoolean(key, false)
        STRING_TYPE -> getString(key, "") ?: ""
        COMMON_TYPE -> @Suppress("IMPLICIT_CAST_TO_ANY") this.all
        else -> throwTypeError("Unsupported type: $type")
    }
    row.add(key, value)
    return cursor
}

/**
 * Extension property to determine an internal type of object
 * by its type.
 *
 * Note: There are string value of of supported types:
 * - "integer" for int values
 * - "long" for long values
 * - "float" for float values
 * - "boolean" for bool values
 * - "string" for string values
 * If provided string that represents a type is not one of listed about
 * the UnsupportedTypeException would be thrown.
 */
internal fun <T : Any?> T.receiveInternalType(): String {
    return when (this) {
        is String -> STRING_TYPE
        is Int -> INT_TYPE
        is Long -> LONG_TYPE
        is Boolean -> BOOLEAN_TYPE
        is Float -> FLOAT_TYPE
        else -> throwTypeErrorOf(this)
    }
}