package com.liveperson.api;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;

import androidx.annotation.NonNull;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.liveperson.api.sdk.LPConversationData;
import com.liveperson.api.sdk.PermissionType;
import com.liveperson.infra.utils.LPDeprecated;
import com.liveperson.messaging.LpError;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.model.AgentData;


/**
 * Created by shiranr on 6/27/17.
 */
public class LivePersonIntents {

	public interface ILivePersonIntentAction {

		// Conversation Events

		/**
		 * Called whenever either the consumer or the agent starts a new conversation.
		 * <br/><br/>
		 * Use LivePersonIntents.getLPConversationData(intent) to acquire the
		 * LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA data from the Intent.
		 */
		String LP_ON_CONVERSATION_STARTED_INTENT_ACTION = "LP_ON_CONVERSATION_STARTED_INTENT_ACTION";

		/**
		 * Called when the current conversation is marked as resolved by either the consumer
		 * or the agent.
		 * <br/><br/>
		 * Use LivePersonIntents.getLPConversationData(intent) to acquire the
		 * LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA data from the Intent.
		 */
		String LP_ON_CONVERSATION_RESOLVED_INTENT_ACTION = "LP_ON_CONVERSATION_RESOLVED_INTENT_ACTION";

		/**
		 * Called when the Conversation Fragment is closed (only called when using Fragment
		 * Mode).
		 */
		String LP_ON_CONVERSATION_FRAGMENT_CLOSED_INTENT_ACTION = "LP_ON_CONVERSATION_FRAGMENT_CLOSED_INTENT_ACTION";

		/**
		 * Called when the current conversation gets marked as urgent by either the consumer
		 * or the agent.
		 */
		String LP_ON_CONVERSATION_MARKED_AS_URGENT_INTENT_ACTION = "LP_ON_CONVERSATION_MARKED_AS_URGENT_INTENT_ACTION";

		/**
		 * Called when the current conversation gets marked as normal by either the consumer
		 * or the agent.
		 */
		String LP_ON_CONVERSATION_MARKED_AS_NORMAL_INTENT_ACTION = "LP_ON_CONVERSATION_MARKED_AS_NORMAL_INTENT_ACTION";


		// Agent Action Events

		/**
		 * Called when the assigned agent is typing a message. When there are 2 seconds of
		 * idle time, the callback returns false to indicate that the agent stopped typing.
		 * <br/><br/>
		 * Use LivePersonIntents.getAgentTypingValue(intent) to acquire the
		 * LP_AGENT_IS_TYPING_INTENT_BOOLEAN_EXTRA data from the Intent.
		 */
		String LP_ON_AGENT_TYPING_INTENT_ACTION = "LP_ON_AGENT_TYPING_INTENT_ACTION";

		/**
		 * Called when the assigned agent of the current conversation has changed, or their
		 * details are updated. When no agent is associated with the conversation, the
		 * callback passes a null value. For example, this happens when an agent returns the
		 * consumer to the request queue.
		 * <br/><br/>
		 * Use LivePersonIntents.getAgentData(intent) to acquire the
		 * LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA data from the Intent.
		 */
		String LP_ON_AGENT_DETAILS_CHANGED_INTENT_ACTION = "LP_ON_AGENT_DETAILS_CHANGED_INTENT_ACTION";

		/**
		 * Called when there is a change in agent availability.
		 * <br/><br/>
		 * Use LivePersonIntents.getOfflineHoursOn(intent) to acquire the
		 * LP_IS_OFFLINE_HOURS_ON_INTENT_BOOLEAN_EXTRA data from the Intent.
		 */
		String LP_ON_OFFLINE_HOURS_CHANGES_INTENT_ACTION = "LP_ON_OFFLINE_HOURS_CHANGES_INTENT_ACTION";


		// User Action Events

		/**
		 * Called if the user denied a necessary system permission for the action they
		 * tried to perform. For example, when the user clicks on the camera/gallery
		 * button to add an image, the permission system dialog was displayed, and the
		 * user denied permission.
		 * <br/><br/>
		 * Use LivePersonIntents.getPermissionType(intent) to acquire the
		 * LP_PERMISSION_TYPE_EXTRA data from the Intent.
		 * <br/><br/>
		 * Use LivePersonIntents.getPermissionDoNotShowAgainMarked(intent) to acquire
		 * the LP_PERMISSION_DO_NOT_SHOW_AGAIN_EXTRA data from the Intent.
		 */
		String LP_ON_USER_DENIED_PERMISSION = "LP_ON_USER_DENIED_PERMISSION";

		/**
		 * Called before requiring a permission that the user has not yet accepted. For
		 * example, when the user clicks the camera or gallery buttons to add an image,
		 * this callback gets called just before the permission dialog is displayed. If
		 * the user already allowed permission, this callback does not get called.
		 * <br/><br/>
		 * Use LivePersonIntents.getPermissionType(intent) to acquire the
		 * LP_PERMISSION_TYPE_EXTRA data from the Intent.
		 */
		String LP_ON_USER_ACTION_ON_PREVENTED_PERMISSION = "LP_ON_USER_ACTION_ON_PREVENTED_PERMISSION";

		/**
		 * Called when the user taps on the agent avatar.
		 * The icon is available next to the agent message bubble or on the top of the
		 * toolbar (if using activity mode).
		 * <br/><br/>
		 * Use LivePersonIntents.getAgentData(intent) to acquire the
		 * LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA data from the Intent.
		 */
		String LP_ON_AGENT_AVATAR_TAPPED_INTENT_ACTION = "LP_ON_AGENT_AVATAR_TAPPED_INTENT_ACTION";

		/**
		 * Called when a structured content control with Link action gets clicked.
		 * <br/><br/>
		 * This callback only gets called if the `structured_content_link_as_callback`
		 * parameter in the branding.xml is set to <b>true</b>.
		 * <br/><br/>
		 * Use LivePersonIntents.getLinkUri(intent) to acquire the LP_LINK_URI_EXTRA
		 * data from the Intent.
		 */
		String LP_ON_STRUCTURED_CONTENT_LINK_CLICKED = "LP_ON_STRUCTURED_CONTENT_LINK_CLICKED";


		// Survey Events

		/**
		 * Called when the feedback screen launches.
		 */
		String LP_ON_CSAT_LAUNCHED_INTENT_ACTION = "LP_ON_CSAT_LAUNCHED_INTENT_ACTION";

		/**
		 * Called when the feedback screen gets dismissed with any result, positive or
		 * negative. (The user taps the <b>Submit</b>, <b>Skip</b>, or <b>Back</b>
		 * buttons.)
		 */
		String LP_ON_CSAT_DISMISSED_INTENT_ACTION = "LP_ON_CSAT_DISMISSED_INTENT_ACTION";

		/**
		 * Called when the user taps the <b>Submit</b> button on the feedback screen.
		 * The onCsatDismissed callback is called as well.
		 * <br/><br/>
		 * Use LivePersonIntents.getConversationID(intent) to acquire the
		 * LP_CONVERSATION_ID_INTENT_STRING_EXTRA data from the Intent.
		 * <br/><br/>
		 * Use LivePersonIntents.getCsatStarRating(intent) to acquire the
		 * LP_CSAT_STAR_RATING_INTENT_INT_EXTRA data from the Intent.
		 */
		String LP_ON_CSAT_SUBMITTED_INTENT_ACTION = "LP_ON_CSAT_SUBMITTED_INTENT_ACTION";

		/**
		 * Called when the user taps the <b>Skip</b> or <b>Back</b> buttons.
		 * The onCsatDismissed callback is called as well.
		 */
		String LP_ON_CSAT_SKIPPED_INTENT_ACTION = "LP_ON_CSAT_SKIPPED_INTENT_ACTION";


		// Authentication Events

		/**
		 * Called if the token used in the session has expired and is no longer valid.
		 * The host app needs to reconnect with a new authentication key.
		 */
		String LP_ON_TOKEN_EXPIRED_INTENT_ACTION = "LP_ON_TOKEN_EXPIRED_INTENT_ACTION";

		/**
		 * Called if the temporary user authentication used specifically in an
		 * Unauthenticated type flow expires.
		 */
		String LP_ON_UNAUTHENTICATED_USER_EXPIRED_INTENT_ACTION = "LP_ON_UNAUTHENTICATED_USER_EXPIRED_INTENT_ACTION";


		// Connection Events

		/**
		 * Called when the connection to the conversation server has been established or
		 * disconnected.
		 * <br/><br/>
		 * Use LivePersonIntents.getConnectedValue(intent) to acquire the
		 * LP_IS_CONNECTED_INTENT_BOOLEAN_EXTRA data from the Intent.
		 */
		String LP_ON_CONNECTION_CHANGED_INTENT_ACTION = "LP_ON_CONNECTION_CHANGED_INTENT_ACTION";


		// Error Events

		/**
		 * Called to indicate that an internal SDK error occurred.
		 * <br/><br/>
		 * Use LivePersonIntents.getOnErrorTaskType(intent) to acquire the
		 * LP_ON_ERROR_TASK_TYPE_INTENT_INT_EXTRA data from the Intent.
		 * <br/><br/>
		 * Use LivePersonIntents.getOnErrorMessage(intent) to acquire the
		 * LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA data from the Intent.
		 *
		 * @deprecated Since September 28, 2020
		 */
		@Deprecated
		@LPDeprecated(since = 1601280000)
		String LP_ON_ERROR_INTENT_ACTION = "LP_ON_ERROR_INTENT_ACTION"; //TODO: remove it when upgrade to v6.0.0, we should separate TaskType with Errors.

		/**
		 * Called to indicate that an internal SDK error occurred.
		 * <br/><br/>
		 * Use LivePersonIntents.getOnErrorType(intent) to acquire the
		 * LP_ON_ERROR_INTENT_INT_EXTRA data from the Intent.
		 * <br/><br/>
		 * Use LivePersonIntents.getOnErrorMessage(intent) to acquire the
		 * LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA data from the Intent.
		 */
		String LP_ON_ERROR_TYPE_INTENT_ACTION = "LP_ON_ERROR_TYPE_INTENT_ACTION";
	}

	public interface ILivePersonIntentExtras {

		// Conversation Extras

		/**
		 * Contains {@link LPConversationData}, containing the ID of the resolved
		 * Conversation, and a {@link com.liveperson.api.response.types.CloseReason}
		 * describing why the conversation ended.
		 * <br/><br/>
		 * Used for LP_ON_CONVERSATION_STARTED_INTENT_ACTION and
		 * LP_ON_CONVERSATION_RESOLVED_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getLPConversationData(intent).
		 */
		String LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA = "LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA";


		// Agent Extras

		/**
		 * Contains <b>true</b> if the Agent is currently typing; <b>false</b> if the Agent's
		 * keyboard becomes idle.
		 * <br/><br/>
		 * Used for LP_ON_AGENT_TYPING_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getAgentTypingValue(intent).
		 */
		String LP_AGENT_IS_TYPING_INTENT_BOOLEAN_EXTRA = "LP_AGENT_IS_TYPING_INTENT_BOOLEAN_EXTRA";

		/**
		 * Contains {@link AgentData}, containing the first and last names, avatar url, and
		 * employee ID of the new (or updated) currently-assigned Agent.
		 * <br/><br/>
		 * Used for LP_ON_AGENT_DETAILS_CHANGED_INTENT_ACTION and
		 * LP_ON_AGENT_AVATAR_TAPPED_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getAgentData(intent).
		 */
		String LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA = "LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA";

		/**
		 * Contains <b>true</b> if the Agent is in Offline Hours mode; <b>false</b> if the
		 * Agent returns to an online state.
		 * <br/><br/>
		 * Used for LP_ON_OFFLINE_HOURS_CHANGES_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getOfflineHoursOn(intent).
		 */
		String LP_IS_OFFLINE_HOURS_ON_INTENT_BOOLEAN_EXTRA = "LP_IS_OFFLINE_HOURS_ON_INTENT_BOOLEAN_EXTRA";


		// User Extras

		/**
		 * The type of permission required for the action the user was attempting to
		 * perform.
		 * <br/><br/>
		 * Used for LP_ON_USER_DENIED_PERMISSION and
		 * LP_ON_USER_ACTION_ON_PREVENTED_PERMISSION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getPermissionType(intent).
		 */
		String LP_PERMISSION_TYPE_EXTRA = "LP_PERMISSION_TYPE_EXTRA";

		/**
		 * <b>true</b> if the user directed the Permissions system to never ask again
		 * for this permissionType; <b>false</b> otherwise.
		 * <br/><br/>
		 * Used for LP_ON_USER_DENIED_PERMISSION.
		 * <br/><br/>
		 * Acquire from an Intent with
		 * LivePersonIntents.getPermissionDoNotShowAgainMarked(intent).
		 */
		String LP_PERMISSION_DO_NOT_SHOW_AGAIN_EXTRA = "LP_PERMISSION_DO_NOT_SHOW_AGAIN_EXTRA";

		/**
		 * A link to the content the user was sent to when they tapped on a piece of
		 * Structured Content.
		 * <br/><br/>
		 * Used for LP_ON_STRUCTURED_CONTENT_LINK_CLICKED.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getLinkUri(intent).
		 */
		String LP_LINK_URI_EXTRA = "LP_LINK_URI_EXTRA";


		// Survey Extras

		/**
		 * ID of the conversation the survey is rating.
		 * <br/><br/>
		 * Used for LP_ON_CSAT_SUBMITTED_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getConversationID(intent).
		 */
		String LP_CONVERSATION_ID_INTENT_STRING_EXTRA = "LP_CONVERSATION_ID_INTENT_STRING_EXTRA";

		/**
		 * How many stars (out of 5) the user rated this conversation.
		 * <br/><br/>
		 * Used for LP_ON_CSAT_SUBMITTED_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getCsatStarRating(intent).
		 */
		String LP_CSAT_STAR_RATING_INTENT_INT_EXTRA = "LP_CSAT_STAR_RATING_INTENT_INT_EXTRA";


		// Connection Extras

		/**
		 * Contains <b>true</b> for connected, <b>false</b> for disconnected.
		 * <br/><br/>
		 * Used for LP_ON_CONNECTION_CHANGED_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getConnectedValue(intent).
		 */
		String LP_IS_CONNECTED_INTENT_BOOLEAN_EXTRA = "LP_IS_CONNECTED_INTENT_BOOLEAN_EXTRA";


		// Error Extras

		/**
		 * The category or type of error.
		 * <br/><br/>
		 * Used for LP_ON_ERROR_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getOnErrorTaskType(intent).
		 *
		 * @deprecated Since September 28, 2020
		 */
		@Deprecated
		@LPDeprecated(since = 1601280000)
		String LP_ON_ERROR_TASK_TYPE_INTENT_INT_EXTRA = "LP_ON_ERROR_TASK_TYPE_INTENT_INT_EXTRA";

		/**
		 * The error type.
		 * <br/><br/>
		 * Used for LP_ON_ERROR_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getErrorType(intent).
		 */
		String LP_ON_ERROR_INTENT_INT_EXTRA = "LP_ON_ERROR_INTENT_INT_EXTRA";

		/**
		 * A detailed message on the error.
		 * <br/><br/>
		 * Used for LP_ON_ERROR_INTENT_ACTION.
		 * <br/><br/>
		 * Acquire from an Intent with LivePersonIntents.getOnErrorMessage(intent).
		 */
		String LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA = "LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA";
	}

//region == INTENT SENDERS ==

	// Conversation Events
	public static void sendOnConversationStarted(Context context, LPConversationData convData) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONVERSATION_STARTED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA, convData);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnConversationResolved(Context context, LPConversationData convData) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONVERSATION_RESOLVED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA, convData);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnConversationFragmentClosed(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONVERSATION_FRAGMENT_CLOSED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnConversationMarkedAsUrgent(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONVERSATION_MARKED_AS_URGENT_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnConversationMarkedAsNormal(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONVERSATION_MARKED_AS_NORMAL_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	// Agent Action Events
	public static void sendOnAgentTyping(Context context, boolean isTyping) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_AGENT_TYPING_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_AGENT_IS_TYPING_INTENT_BOOLEAN_EXTRA, isTyping);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnAgentDetailsChanged(Context context, AgentData agentData) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_AGENT_DETAILS_CHANGED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA, agentData);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnOfflineHoursChanges(Context context, boolean isOffHoursOn) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_OFFLINE_HOURS_CHANGES_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_IS_OFFLINE_HOURS_ON_INTENT_BOOLEAN_EXTRA, isOffHoursOn);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	// User Action Events
	public static void sendOnUserDeniedPermission(Context context, PermissionType permissionType, boolean doNotShowAgainMarked) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_USER_DENIED_PERMISSION);
		intent.putExtra(ILivePersonIntentExtras.LP_PERMISSION_TYPE_EXTRA, permissionType.ordinal());
		intent.putExtra(ILivePersonIntentExtras.LP_PERMISSION_DO_NOT_SHOW_AGAIN_EXTRA, doNotShowAgainMarked);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnUserActionOnPreventedPermission(Context context, PermissionType permissionType) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_USER_ACTION_ON_PREVENTED_PERMISSION);
		intent.putExtra(ILivePersonIntentExtras.LP_PERMISSION_TYPE_EXTRA, permissionType.ordinal());
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnAgentAvatarTapped(Context context, AgentData agentData) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_AGENT_AVATAR_TAPPED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA, agentData);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnStructuredContentLinkClicked(Context context, String uri) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_STRUCTURED_CONTENT_LINK_CLICKED);
		intent.putExtra(ILivePersonIntentExtras.LP_LINK_URI_EXTRA, uri);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	// Survey Events
	public static void sendOnCsatLaunched(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CSAT_LAUNCHED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnCsatDismissed(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CSAT_DISMISSED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnCsatSubmitted(Context context, String conversationId, Integer starRating) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CSAT_SUBMITTED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_CONVERSATION_ID_INTENT_STRING_EXTRA, conversationId);
		intent.putExtra(ILivePersonIntentExtras.LP_CSAT_STAR_RATING_INTENT_INT_EXTRA, starRating);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnCsatSkipped(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CSAT_SKIPPED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	// Authentication Events
	public static void sendOnTokenExpired(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_TOKEN_EXPIRED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	public static void sendOnUserExpired(Context context) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_UNAUTHENTICATED_USER_EXPIRED_INTENT_ACTION);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	// Connection Events
	public static void sendOnConnectionChanged(Context context, boolean isUpdated) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_CONNECTION_CHANGED_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_IS_CONNECTED_INTENT_BOOLEAN_EXTRA, isUpdated);
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	/**
	 * Error Events
	 * @deprecated
	 */
	public static void sendOnErrorIntent(Context context, TaskType type, String message) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_ERROR_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA, message);
		intent.putExtra(ILivePersonIntentExtras.LP_ON_ERROR_TASK_TYPE_INTENT_INT_EXTRA, type.ordinal());
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

	/**
	 * Error Events
	 */
	public static void sendOnErrorIntent(Context context, LpError lpError, String message) {
		Intent intent = new Intent(ILivePersonIntentAction.LP_ON_ERROR_TYPE_INTENT_ACTION);
		intent.putExtra(ILivePersonIntentExtras.LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA, message);
		intent.putExtra(ILivePersonIntentExtras.LP_ON_ERROR_INTENT_INT_EXTRA, lpError.ordinal());
		LocalBroadcastManager.getInstance(context.getApplicationContext()).sendBroadcast(intent);
	}

//endregion == INTENT SENDERS ==

//region == INTENT EXTRA GETTERS ==

	// Conversation Extras
	public static LPConversationData getLPConversationData(Intent intent) {
		return intent.getParcelableExtra(ILivePersonIntentExtras.LP_CONVERSATION_DATA_INTENT_PARCELABLE_EXTRA);
	}

	// Agent Extras
	public static boolean getAgentTypingValue(Intent intent) {
		return intent.getBooleanExtra(ILivePersonIntentExtras.LP_AGENT_IS_TYPING_INTENT_BOOLEAN_EXTRA, false);
	}

	public static AgentData getAgentData(Intent intent) {
		return intent.getParcelableExtra(ILivePersonIntentExtras.LP_AGENT_DATA_INTENT_PARCELABLE_EXTRA);
	}

	public static boolean getOfflineHoursOn(Intent intent) {
		return intent.getBooleanExtra(ILivePersonIntentExtras.LP_IS_OFFLINE_HOURS_ON_INTENT_BOOLEAN_EXTRA, false);
	}

	// User Extras
	public static PermissionType getPermissionType(Intent intent) {
		int permissionType = intent.getIntExtra(ILivePersonIntentExtras.LP_PERMISSION_TYPE_EXTRA, -1);

		if (permissionType >= 0 && permissionType < PermissionType.values().length) {
			return PermissionType.values()[permissionType];
		} else {
			return null;
		}
	}

	public static boolean getPermissionDoNotShowAgainMarked(Intent intent) {
		return intent.getBooleanExtra(ILivePersonIntentExtras.LP_PERMISSION_DO_NOT_SHOW_AGAIN_EXTRA, false);
	}

	public static String getLinkUri(Intent intent) {
		return intent.getStringExtra(ILivePersonIntentExtras.LP_LINK_URI_EXTRA);
	}

	// Survey Extras
	public static String getConversationID(Intent intent) {
		return intent.getStringExtra(ILivePersonIntentExtras.LP_CONVERSATION_ID_INTENT_STRING_EXTRA);
	}

	public static int getCsatStarRating(Intent intent) {
		return intent.getIntExtra(ILivePersonIntentExtras.LP_CSAT_STAR_RATING_INTENT_INT_EXTRA, -1);
	}

	// Connection Extras
	public static boolean getConnectedValue(Intent intent) {
		return intent.getBooleanExtra(ILivePersonIntentExtras.LP_IS_CONNECTED_INTENT_BOOLEAN_EXTRA, false);
	}

	// Error Extras
	public static TaskType getOnErrorTaskType(Intent intent) {
		int taskTypeInt = intent.getIntExtra(ILivePersonIntentExtras.LP_ON_ERROR_TASK_TYPE_INTENT_INT_EXTRA, -1);
		if (taskTypeInt >= 0 && taskTypeInt < TaskType.values().length) {
			return TaskType.values()[taskTypeInt];
		} else {
			return null;
		}
	}

	@NonNull
	public static LpError getErrorType(Intent intent) {
		int errorTypeIndex = intent.getIntExtra(ILivePersonIntentExtras.LP_ON_ERROR_INTENT_INT_EXTRA, -1);
		if (errorTypeIndex >= 0 && errorTypeIndex < LpError.values().length) {
			return LpError.values()[errorTypeIndex];
		} else {
			return LpError.UNKNOWN;
		}
	}

	public static String getOnErrorMessage(Intent intent) {
		return intent.getStringExtra(ILivePersonIntentExtras.LP_ON_ERROR_MESSAGE_INTENT_STRING_EXTRA);
	}

//endregion == INTENT EXTRA GETTERS ==

	// Comprehensive Intent Filter
	public static IntentFilter getIntentFilterForAllEvents() {
		IntentFilter filter = new IntentFilter();
		filter.addAction(ILivePersonIntentAction.LP_ON_ERROR_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_ERROR_TYPE_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_TOKEN_EXPIRED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONVERSATION_STARTED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONVERSATION_RESOLVED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONVERSATION_FRAGMENT_CLOSED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONNECTION_CHANGED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_AGENT_TYPING_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_AGENT_DETAILS_CHANGED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CSAT_LAUNCHED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CSAT_DISMISSED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CSAT_SKIPPED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CSAT_SUBMITTED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONVERSATION_MARKED_AS_URGENT_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_CONVERSATION_MARKED_AS_NORMAL_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_OFFLINE_HOURS_CHANGES_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_AGENT_AVATAR_TAPPED_INTENT_ACTION);
		filter.addAction(ILivePersonIntentAction.LP_ON_USER_DENIED_PERMISSION);
		filter.addAction(ILivePersonIntentAction.LP_ON_USER_ACTION_ON_PREVENTED_PERMISSION);
		filter.addAction(ILivePersonIntentAction.LP_ON_STRUCTURED_CONTENT_LINK_CLICKED);
		filter.addAction(ILivePersonIntentAction.LP_ON_UNAUTHENTICATED_USER_EXPIRED_INTENT_ACTION);
		return filter;
	}
}
