package com.liveperson.messaging.background;

import android.content.Context;
import android.graphics.Bitmap;
import androidx.annotation.NonNull;
import androidx.exifinterface.media.ExifInterface;

import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.filesharing.BaseUploadTask;
import com.liveperson.messaging.background.filesharing.UploadFileTaskBundle;
import com.liveperson.messaging.background.filesharing.image.UploadImageTaskBundle;
import com.liveperson.messaging.commands.SendImageCommand;
import com.liveperson.messaging.exception.FileSharingException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000B0;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000B1;

/**
 * Created by shiranr on 7/24/16.
 */
public class UploadImageTask  extends BaseUploadTask {

    private static final String TAG = "UploadImageTask";
    /**
     * Created this final int when removing usage of
     * thumbnail_longer_dimension_resize,use 100 as default
     */
    private static final int THUMBNAIL_LONGER_DIMENSION_RESIZE = 100;
    protected String mBase64 = null;

    protected UploadImageTaskBundle uploadImageTaskParams;
	private byte[] mOriginalFileByteArray = null;

    /**
     * creating bitmaps (thumbnail and full size)
     * @param params
     */
    public UploadImageTask(UploadImageTaskBundle params, Integer uploadTimeout) throws FileSharingException {
    	super(uploadTimeout);

		int orientation = 0;
		ExifInterface exif;

		if(params == null){
			throw new FileSharingException("Params is null");
		}

        uploadImageTaskParams = params;

	    LPLog.INSTANCE.d(TAG + Thread.currentThread(), "imageTypeExtension = " + uploadImageTaskParams.getFileTypeExtension() + ", imageContentType = " + uploadImageTaskParams.getFileContentType() );

        try {
            // Save the original orientation because picasso losing this information during image resize
			try {
				exif = new ExifInterface(uploadImageTaskParams.getFilePath());
				orientation = exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
			} catch (FileNotFoundException e) {
				LPLog.INSTANCE.w(TAG, "UploadImageTask: cannot get orientation from exif. Using default (0)", e);
			}

            final Bitmap thumbnailBitmap = processThumbnailImage(orientation, uploadImageTaskParams.isImageFromCamera(), uploadImageTaskParams.shouldAddBlurEffect());
            final Bitmap originalBitmap = processOriginalImage(orientation, uploadImageTaskParams.isImageFromCamera());

            final String thumbnailPath = saveThumbnailImage(thumbnailBitmap);
            final String fullImagePath =  saveOriginalImage(originalBitmap);

            LPLog.INSTANCE.d(TAG, "UploadImageTask: Image and thumbnail created. Continue to send image to server");

            createSendMessageCommand(thumbnailPath, fullImagePath);


        } catch (IOException e) {
            LPLog.INSTANCE.e(TAG + Thread.currentThread(), ERR_000000B0, "error resizing or saving image", e);
            onUploadFailed(e);
            throw new FileSharingException("error resizing or saving image");
        }

    }

	@Override
	protected UploadFileTaskBundle getUploadTaskBundle() {
		return uploadImageTaskParams;
	}

	@Override
	protected byte[] getFileByteArray() {
		return mOriginalFileByteArray;
	}

	@Override
	public int getTaskId() {
		return uploadImageTaskParams.getTaskId();
	}

	@Override
	public String getThumbnailBase64() {
		return mBase64;
	}

	@NonNull
    protected Bitmap processOriginalImage(int orientation, boolean fromCamera) throws IOException {
        Bitmap originalBitmap;// Resize the full image
        int fullImageLongerDimensionResize = Configuration.getInteger(R.integer.full_image_longer_dimension_resize);
        Context context = Infra.instance.getApplicationContext();
        originalBitmap = ImageUtils.createResizedBitmap(context, uploadImageTaskParams.getFileUri(), fullImageLongerDimensionResize, orientation, fromCamera);
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "originalBitmap size (w, h): " + originalBitmap.getWidth() + ", " + originalBitmap.getHeight());
        return originalBitmap;
    }

    @NonNull
    protected String saveOriginalImage(Bitmap originalBitmap) throws FileSharingException {
        // Create a byte array of the image. If the image is NOT from the camera, save it to local folder
        final String fullImagePath = convertAndSaveFullImage(originalBitmap, uploadImageTaskParams.getFilePath(), uploadImageTaskParams.isImageFromCamera(), uploadImageTaskParams.getFileTypeExtension());
        if (fullImagePath == null) { // File is too large (after compression)
            LPLog.INSTANCE.d(TAG, "UploadImageTask: file is too large after compression");
            throw new FileSharingException("File is too large after compression");
        }
        return fullImagePath;
    }

    @NonNull
    protected Bitmap processThumbnailImage(int orientation, boolean fromCamera, boolean addBlurEffect) throws IOException {
        Bitmap thumbnailBitmap;// Create a thumbnailBitmap
        int blurRadius = Configuration.getInteger(R.integer.lp_image_thumbnail_blur_radius);
        Context context = Infra.instance.getApplicationContext();
        thumbnailBitmap = ImageUtils.createResizedBitmap(context, uploadImageTaskParams.getFileUri(), THUMBNAIL_LONGER_DIMENSION_RESIZE, orientation, fromCamera);

        if (addBlurEffect) {
            try {
                thumbnailBitmap = ImageUtils.createBlurredBitmap(blurRadius, thumbnailBitmap);
            } catch (Exception e) {
                LPLog.INSTANCE.w(TAG, "Failed to add blur effect to an image. Radius: " + blurRadius + " Error: " + e);
            }
        }
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "thumbnailBitmap size: " + thumbnailBitmap.getWidth() + ", " + thumbnailBitmap.getHeight());
        return thumbnailBitmap;
    }

    @NonNull
    protected String saveThumbnailImage(Bitmap thumbnailBitmap) throws FileSharingException {
        // Create and save a thumbnail to the local folder
        final String thumbnailPath = saveThumbnail(thumbnailBitmap, uploadImageTaskParams.getFileTypeExtension());
        if (thumbnailPath == null) {
            LPLog.INSTANCE.e(TAG + Thread.currentThread(), ERR_000000B1, "run: Could not save image thumbnail to disk");
            throw new FileSharingException("Could not save thumbnailBitmap image to disk");
        }
        return thumbnailPath;
    }

    private String saveThumbnail(Bitmap mThumbnail, String imageTypeExtension) {

		// We support only JPG and PNG image types, so if the imageTypeExtension is not one of those we change it to jpg
		if (!imageTypeExtension.equals(ImageUtils.JPG) && !imageTypeExtension.equals(ImageUtils.PNG)) {
			imageTypeExtension = ImageUtils.JPG;
		}

		byte [] thumbnailByteArray = ImageUtils.getOutputStreamFromBitmap(mThumbnail, 100, imageTypeExtension);

		// If thumbnail was not created successfully return null
		if (thumbnailByteArray == null) {
			return null;
		}

		// Encode thumbnail's base64 string
		mBase64 = ImageUtils.bitmapToBase64(thumbnailByteArray);
        LPLog.INSTANCE.d(TAG + Thread.currentThread(), "run: Thumbnail Base64: " + mBase64);

        // Save the bitmap to disk and get its path
        return ImageUtils.saveBitmapToDisk(Infra.instance.getApplicationContext(), thumbnailByteArray, uploadImageTaskParams.getBrandId(), ImageUtils.ImageFolderType.PREVIEW, imageTypeExtension);
    }

	/**
	 * Create an output stream from the given bitmap.
	 * @param mOriginalBitmap
	 * @param originalImagePath
	 * @param saveOnOriginalPath  @return
	 * @param imageTypeExtension
	 */
	private String convertAndSaveFullImage(Bitmap mOriginalBitmap, String originalImagePath, boolean saveOnOriginalPath, String imageTypeExtension) {

        createOriginalImageByteArray(mOriginalBitmap, imageTypeExtension);


        // If the full image was copied successfully to private folder return null
		if (mOriginalFileByteArray == null) {
			return null;
		}

		LPLog.INSTANCE.d(TAG, "convertAndSaveFullImage: size of full image file (after compression): " + mOriginalFileByteArray.length);

		// Check if the image after compression is larger than allowed
		int maxImageSizeKb = Configuration.getInteger(R.integer.max_image_size_kb);
		if (mOriginalFileByteArray.length > maxImageSizeKb * 1000) {
			return null;
		}

		if(saveOnOriginalPath) {
			return ImageUtils.saveBitmapToDisk(mOriginalFileByteArray, new File(originalImagePath));
		}
		else{
			return ImageUtils.saveBitmapToDisk(Infra.instance.getApplicationContext(), mOriginalFileByteArray, uploadImageTaskParams.getBrandId(), ImageUtils.ImageFolderType.FULL, imageTypeExtension);
		}
	}

    protected void createOriginalImageByteArray(Bitmap mOriginalBitmap, String imageTypeExtension) {
        // Get the compression rate for the full image from the resources
        int compressionRate = Configuration.getInteger(R.integer.full_image_compression_rate);

        LPLog.INSTANCE.d(TAG, "convertAndSaveFullImage: compression rate for full image: " + compressionRate);

        // Get byte array of the full image bitmap
        mOriginalFileByteArray = ImageUtils.getOutputStreamFromBitmap(mOriginalBitmap, compressionRate, imageTypeExtension);
    }

    protected void createSendMessageCommand(String thumbnailPath, String fullImagePath) {
        sendMessageCommand = new SendImageCommand(MessagingFactory.getInstance().getController(), uploadImageTaskParams.getTargetId(), uploadImageTaskParams.getBrandId(),
                uploadImageTaskParams.getFileContentType(), thumbnailPath, fullImagePath,uploadImageTaskParams.getFileTypeExtension(), uploadImageTaskParams.getMessage());
        setSendMessageCommandCallback();
    }
}
