package com.liveperson.messaging.commands;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LinkUtils;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.socket.requests.NewConversationRequest;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;
import com.liveperson.messaging.offline.OfflineMessagingManager;
import com.liveperson.messaging.offline.api.OfflineMessagesRepository;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000A7;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A8;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A9;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000016B;
import static com.liveperson.messaging.model.MessagingChatMessage.MessageState.PENDING;
import static com.liveperson.messaging.model.MessagingChatMessage.MessageState.QUEUED;
import static com.liveperson.messaging.model.MessagingChatMessage.MessageState.OFFLINE;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Set;

/**
 * Created by Ilya Gazman on 11/11/2015.
 * <p/>
 * A command for sending message
 */
public class SendMessageCommand implements Command {

    private static final String TAG = "SendMessageCommand";

    protected final Messaging mController;

    protected String mBrandId;
    protected String mTargetId;
    protected MaskedMessage mMessage;
    protected String mConsumerId;
    protected String mEventId;
	protected DeliveryStatusUpdateInfo mInfo;

    /**
     * Creates a command for sending messages
     *
     * @param controller
     * @param message
     */
    public SendMessageCommand(Messaging controller, String targetId, String brandId, MaskedMessage message) {
        mController = controller;
        mBrandId = brandId;
        mTargetId = targetId;
        mMessage = message;
    }

    /**
     * Creates a command for sending messages
     *
     * @param controller
     * @param message
	 * @param info a {@link DeliveryStatusUpdateInfo} object with metadata to be sent with the publish
     */
    public SendMessageCommand(Messaging controller, String targetId, String brandId, MaskedMessage message, @Nullable DeliveryStatusUpdateInfo info) {
		this(controller, targetId, brandId, message);
        mInfo = info;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.d(TAG, "execute");
        Dialog activeDialog = mController.amsDialogs.getActiveDialog();
        Conversation conversation = mController.amsConversations.getConversationFromTargetIdMap(mTargetId);
        mConsumerId = mController.amsUsers.getConsumerId(mTargetId);
        OfflineMessagingManager manager = mController.getOfflineManager();
        OfflineMessagesRepository offlineMessagesRepository = mController.amsMessages.getOfflineMessagesRepository();
        if (manager.isOfflineModeEnabled()
                && canSendWhileOffline()
                && !manager.isFullySynced()
        ) {
            DataBaseCommand<Dialog> command = manager.getOrCreateOfflineDialog();
            command.setPostQueryOnBackground(offlineDialog -> {
                final String dialogId;
                if (offlineDialog == null) {
                    dialogId = Dialog.OFFLINE_DIALOG_ID;
                } else {
                    dialogId = offlineDialog.getDialogId();
                    if (offlineDialog.getLastServerSequence() == -1) {
                        LPLog.INSTANCE.d(TAG, "Adding metadata for offline message.");
                        formMetadataOnNewConversation(mBrandId);
                        offlineDialog.setLastServerSequence(offlineDialog.getLastServerSequence() + 1);
                    }
                }
                addMessageToDB(dialogId, mMessage, OFFLINE);
            });
            command.execute();
        } else if (conversation == null
                || conversation.getState() == ConversationState.CLOSE
                || conversation.getState() == ConversationState.LOCKED
        ) {
            if (manager.isOfflineModeEnabled() && offlineMessagesRepository.areOfflineMessagesExists(mBrandId)) {
                addMessageToDB(Dialog.OFFLINE_DIALOG_ID, mMessage, OFFLINE);
            } else {
                formMetadataOnNewConversation(mBrandId);
                // Create and persist a temp conversation + temp dialog in DB
                sendOnNewConversation();
            }
        } else if (activeDialog == null) {
            LPLog.INSTANCE.w(TAG, "'Zombie Conversation' state occurred (One conversation is open without open dialogs) while the consumer is trying to send a message! Closing this conversation and open a new one....");
            ResolveConversationCommand resolveConversationCommand = new ResolveConversationCommand(mController.amsConversations, mTargetId, mController.mAccountsController.getConnectionUrl(mBrandId));
            resolveConversationCommand.setCallback(new ICallback<String, Throwable>() {
                @Override
                public void onSuccess(String value) {
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                }

                @Override
                public void onError(Throwable error) {
                    LPLog.INSTANCE.e(TAG, ERR_000000A7, "Failed to resolve conversation before creating a new one, continuing with creating anyway", error);
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                }
            });
            resolveConversationCommand.execute();
        } else {
            LPLog.INSTANCE.d(TAG, "Send message - activeDialog = " + activeDialog.getDialogId() + ", " + activeDialog.getState());
            switch (activeDialog.getState()) {
                case CLOSE:
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                    break;
                case LOCKED:
                    // _TODO: 11/12/2015 What is this state?
                    // _TODO: 15/05/2018 It's a state from the UMS but it's not in use... for 3 years
                    break;
                case OPEN:
                case PENDING:
                case QUEUED:
                    if (manager.isOfflineModeEnabled() && offlineMessagesRepository.areOfflineMessagesExists(mBrandId)) {
                        addMessageToDB(Dialog.OFFLINE_DIALOG_ID, mMessage, OFFLINE);
                    } else {
                        addMessageToDBAndSend(activeDialog.getDialogId(), mMessage, PENDING);
                    }
                    break;
            }
        }
    }

    /**
     * Forming WelcomeMessage metadata if welcome message is being stored in sharedpref
     * @param brandId brand id
     */
    protected void formMetadataOnNewConversation(String brandId) {
        try {
            String strMetadata = this.mController.getWelcomeMessageMetadata(brandId);
            if (strMetadata != null && !strMetadata.isEmpty()) {
                LPLog.INSTANCE.d(TAG, "formMetadataOnNewConversation");
                JSONObject wmMetadata = new JSONObject(strMetadata);
                if (mInfo != null && mInfo.getMetadata() != null) {
                    mInfo.getMetadata().put(wmMetadata);
                } else {
                    mInfo = new DeliveryStatusUpdateInfo(new JSONArray().put(wmMetadata));
                }
                // set to null after used it
                this.mController.setWelcomeMessageMetadata(null, brandId);
            }
        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_0000016B, "Failed to form metadata of welcome message", e);
        }
    }

    /**
     * Creates and persists a temp conversation + temp dialog in DB, adds a message to DB and sends a create-conversation request.
     */
    private void sendOnNewConversation() {
        NewConversationRequest newConversationRequest = createPendingConversationAndDialog();
        sendCreateConversationRequest(newConversationRequest);
        // This must come AFTER creating conversation request!
        addMessageToDBAndSend(Dialog.TEMP_DIALOG_ID, mMessage, PENDING);
	    LPLog.INSTANCE.d(TAG, "sendOnNewConversation: " + LPLog.INSTANCE.mask(mMessage));
    }

    /**
     * Creates pending conversation
     */
    protected NewConversationRequest createPendingConversationAndDialog() {
        final NewConversationRequest newConversationRequest = createNewConversationRequest();
        mController.amsConversations.createPendingConversation(mTargetId, mBrandId, newConversationRequest.getRequestId());
        mController.amsDialogs.createPendingDialog(mTargetId, mBrandId, newConversationRequest.getRequestId());
        return newConversationRequest;
    }

    private NewConversationRequest createNewConversationRequest() {
        return createNewConversationRequest(Conversation.TEMP_CONVERSATION_ID, Dialog.TEMP_DIALOG_ID, null);
    }

    protected NewConversationRequest createNewConversationRequest(String tempConversationId, String tempDialogId, Long requestId) {
        final NewConversationRequest newConversationRequest = new NewConversationRequest(mController, mTargetId, mBrandId, tempConversationId, tempDialogId);
        if (requestId != null) {
            newConversationRequest.setRequestId(requestId);
        }

        mController.amsConversations.enqueuePendingConversationRequestId(newConversationRequest.getRequestId());

        return newConversationRequest;
    }

    public void sendCreateConversationRequest(NewConversationRequest newConversationRequest) {
        SocketManager.getInstance().send(newConversationRequest);
    }

    /**
     * Sends message request
     */
    protected void sendMessage(String conversationId, String dialogId, String message) {
	    SendMessageRequest sendMessageRequest = new SendMessageRequest(mController, mEventId, mTargetId, mBrandId, conversationId, dialogId, message);

	    sendMessage(dialogId, sendMessageRequest);
    }

	/**
	 * Sends message request
	 */
	protected void sendMessage(String dialogId, SendMessageRequest sendMessageRequest) {

		SocketManager.getInstance().send(sendMessageRequest);
		// Add the message to the message queue in order to track if we got ack on it
		mController.amsMessages.mMessageTimeoutQueue.add(MessageTimeoutQueue.MessageType.PUBLISH, (int) sendMessageRequest.getRequestId(),
				mBrandId, dialogId, mEventId);
	}

	protected void sendMessageIfDialogIsOpen() {
		Dialog dialog = mController.amsDialogs.getActiveDialog();

		if (dialog != null) {
			SendMessageRequest sendMessageRequest;
			LPLog.INSTANCE.d(TAG, "sendMessageIfDialogIsOpen: " + dialog.getState());
			if (dialog.getState() == DialogState.OPEN) {
				sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, dialog.getDialogId(), dialog.getConversationId());
				sendMessage(dialog.getDialogId(), sendMessageRequest);
			} else if (dialog.getState() == DialogState.PENDING || dialog.getState() == DialogState.QUEUED) {
				sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, dialog.getDialogId(), dialog.getConversationId());
				dialog.getPendingData().addToPendingRequests(sendMessageRequest);
			} else {
				LPLog.INSTANCE.e(TAG, ERR_000000A8, "sendMessageIfDialogIsOpen: unhandled dialog state:" + dialog.getState());
			}
		} else {
			LPLog.INSTANCE.e(TAG, ERR_000000A9, "sendMessageIfDialogIsOpen: Failed to find an active dialog!");
		}
	}

    /**
     * Method used to adding an offline pending message to database
     * without sending any requests.
     * Such offline messages will be sent to server once connection is established.
     * @param dialogId associated dialog id.
     * @param message masked message typed by user.
     */
    protected void addMessageToDB(String dialogId, MaskedMessage message, MessagingChatMessage.MessageState state) {
        mEventId = UniqueID.createUniqueMessageEventId();
        LPLog.INSTANCE.i(TAG, "addMessageToDB without sending: mEventId = "+ mEventId + " dialog ID = " + dialogId);

        final MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message, state);

        if (mInfo != null && mInfo.getMetadata() != null) {
            chatMessage.setMetadata(mInfo.getMetadata().toString());
        }

        // Add to DB and send
        mController.amsMessages.addMessage(chatMessage, true).execute();
        // If messages are different it means the message was masked. We add a system message for the user
        if (mMessage.isMasked()) {
            String maskedEventId = mEventId + AmsMessages.MASKED_MESSAGE_EVENT_ID_POSTFIX;

            MessagingChatMessage warning = new MessagingChatMessage(chatMessage.getOriginatorId(),
                    message.getMaskedSystemMessage(),
                    chatMessage.getTimeStamp() + 1,
                    chatMessage.getDialogId(),
                    maskedEventId,
                    MessagingChatMessage.MessageType.SYSTEM_MASKED,
                    MessagingChatMessage.MessageState.RECEIVED,
                    AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER,
                    ContentType.text_plain.getText(),
                    EncryptionVersion.NONE
            );

            mController.amsMessages.addMessage(warning, true).execute();
        }
    }

    /**
     * This method is used to add a message to DB. If regexPattern and maskCharacter are provided, the message is checked
     * and masked if the regex pattern exist in the message. Otherwise it is sent as is.
     *
     * @param dialogId
     * @param message
     */
    protected void addMessageToDBAndSend(
            String dialogId,
            MaskedMessage message,
            MessagingChatMessage.MessageState messageState
    ) {
        mEventId = UniqueID.createUniqueMessageEventId();
        LPLog.INSTANCE.i(TAG, "addMessageToDBAndSend: mEventId = "+ mEventId + " dialog ID = " + dialogId);

        final MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message, messageState);

        if (mInfo != null && mInfo.getMetadata() != null) {
            chatMessage.setMetadata(mInfo.getMetadata().toString());
        }

        // Add to DB and send
        mController.amsMessages.addMessage(chatMessage, true)
                .setPreQueryOnBackground(() -> attachMessageToOfflineModeIfNeeded(chatMessage))
                .setPostQueryOnBackground((Long data) -> {
                    LPLog.INSTANCE.i(TAG, "Send message, time: " + chatMessage.getTimeStamp());
                    if (chatMessage.getMessageState() != OFFLINE) {
                        sendMessageIfDialogIsOpen();
                    }
                }).execute();

        // If messages are different it means the message was masked. We add a system message for the user
        if (mMessage.isMasked()) {


            MessagingChatMessage warning = new MessagingChatMessage(
                    chatMessage.getOriginatorId(),
                    message.getMaskedSystemMessage(),
                    chatMessage.getTimeStamp() + 1,
                    chatMessage.getDialogId(),
                    mEventId + AmsMessages.MASKED_MESSAGE_EVENT_ID_POSTFIX,
                    MessagingChatMessage.MessageType.SYSTEM_MASKED,
                    MessagingChatMessage.MessageState.RECEIVED,
                    AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER,
                    ContentType.text_plain.getText(),
                    EncryptionVersion.NONE
            );

            mController.amsMessages.addMessage(warning, true).execute();
        }
    }

    @NonNull
    protected MessagingChatMessage createNewChatMessage(
            String dialogId,
            MaskedMessage message,
            MessagingChatMessage.MessageState state
    ) {
        MessagingChatMessage.MessageType messageType;
        // If a message contains URL, mark it as a Consumer URL
        if (message.isMasked()) {
            if (LinkUtils.containsWebUrls(message.getDbMessage())) {
                messageType = MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
            } else {
                messageType = MessagingChatMessage.MessageType.CONSUMER_MASKED;
            }
        } else {
            if (LinkUtils.containsWebUrls(message.getDbMessage())) {
                messageType = MessagingChatMessage.MessageType.CONSUMER_URL;
            } else {
                messageType = MessagingChatMessage.MessageType.CONSUMER;
            }
        }
        return new MessagingChatMessage(
                mConsumerId,
                message.getDbMessage(),
                System.currentTimeMillis(),
                dialogId,
				mEventId,
                messageType,
                state,
                EncryptionVersion.NONE
        );
    }

    /**
     * Method used to sync offline and pending/queued messages to
     * send and represent messages according to their actual state.
     *
     * If message was sent when socket was connected and offline messages exist,
     * but active dialog wasn't subscribed yet, then this message should be
     * added to db as offline message.
     *
     * If message  sent when socket was connected and offline messages don't exist,
     * but active dialog was subscribed, then this message should be
     * updated with actual timestamp received from related content event.
     *
     * @param chatMessage pending or queued message that was sent
     *                    when user was connected to socket.
     */
    protected void attachMessageToOfflineModeIfNeeded(MessagingChatMessage chatMessage) {
        MessagingChatMessage.MessageState state = chatMessage.getMessageState();
        OfflineMessagingManager offlineManager = mController.getOfflineManager();
        if (offlineManager != null
                && offlineManager.isOfflineModeEnabled()
                && (state.equals(PENDING) || state.equals(QUEUED))
        ) {
           OfflineMessagesRepository repository = mController.amsMessages.getOfflineMessagesRepository();
           Set<String> pendingOfflineMessages = repository.getPendingOfflineMessages(mBrandId);
           pendingOfflineMessages.add(chatMessage.getEventId());
           repository.setPendingOfflineMessages(mBrandId, pendingOfflineMessages);
        }
    }

	@NonNull
	protected SendMessageRequest createMessageRequest(Messaging mController, String mEventId, String mTargetId, String mBrandId, String dialogId, String conversationId) {
		SendMessageRequest sendMessageRequest = new SendMessageRequest(mController, mEventId, mTargetId, mBrandId, dialogId, conversationId);
		sendMessageRequest.setMessageContent(mMessage.getServerMessage());
		sendMessageRequest.setInfo(mInfo);
		return sendMessageRequest;
	}

    public String getEventId() {
        return mEventId;
    }

    /**
     * Method used to determine whether command supports
     * sending messages in offline mode by adding pending messages/files
     * to database.
     * @return true if command should support offline messaging,
     * false otherwise.
     */
    protected boolean canSendWhileOffline() {
        return true;
    }
}
