package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import androidx.annotation.Nullable;

import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.MessagingFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by shiranr on 04/07/2016.
 */
public class AmsFiles extends BaseDBRepository {

	private static final String TAG = "AmsFiles";

	public AmsFiles() {
		super(FilesTable.FILES_TABLE);
	}

	/**
	 * @return DataBaseCommand which returns id of inserted row.
	 */
	public DataBaseCommand<Long> addFile(final long messageRowId, final FileMessage fileMessage){
		return new DataBaseCommand<>(() -> {
			LPLog.INSTANCE.d(TAG, "Adding file: " + fileMessage.getLocalUrl() + " type: " + fileMessage.getFileType());
			ContentValues filesValues = new ContentValues();
			filesValues.put(FilesTable.KEY_PREVIEW, fileMessage.getPreview());
			if (fileMessage.getLocalUrl() != null) {
				filesValues.put(FilesTable.KEY_LOCAL_URL, fileMessage.getLocalUrl());
				filesValues.put(FilesTable.KEY_LOCAL_URL_UPDATE_TIMESTAMP, System.currentTimeMillis());
			}
			filesValues.put(FilesTable.KEY_FILE_TYPE,fileMessage.getFileType());
			filesValues.put(FilesTable.KEY_LOAD_STATUS, fileMessage.getLoadStatus().ordinal());
			filesValues.put(FilesTable.KEY_SWIFT_PATH, fileMessage.getSwiftPath());
			filesValues.put(FilesTable.KEY_RELATED_MESSAGE_ROW_ID,messageRowId);
			return getDB().insert(filesValues);
		});
	}

	/**
	 * Update a specific relativePath by row id.
	 * @param fileRowId
	 * @param relativePath
	 */
	public void updateRelativePath(final Long fileRowId, final String relativePath) {
		DataBaseExecutor.execute(() -> {
			ContentValues filesValues = new ContentValues();
			filesValues.put(FilesTable.KEY_SWIFT_PATH, relativePath);
			getDB().update(filesValues, FilesTable.KEY_ID + " =? ", new String[]{String.valueOf(fileRowId)});
		});
	}

	public void updateStatus(
			final long fileRowId,
			final FilesTable.LoadStatus status
	) {
		updateStatus(fileRowId, status, null);
	}

	public void updateStatus(
			final long fileRowId,
			final FilesTable.LoadStatus status,
			@Nullable final ExecutionCallback<Void> callback
	) {
		DataBaseExecutor.execute(() -> {
			ContentValues filesValues = new ContentValues();
			filesValues.put(FilesTable.KEY_LOAD_STATUS, status.ordinal());
			getDB().update(filesValues, FilesTable.KEY_ID + " =? ", new String[]{String.valueOf(fileRowId)});
			LPLog.INSTANCE.d(TAG, "updateStatus :" + status + " file:"+ fileRowId);
			if (callback != null) {
				callback.onResult();
			}
		});
	}

	public DataBaseCommand<Void> updateLocalPath(final long fileRowId, final String path) {
        return new DataBaseCommand<>(() -> {
			ContentValues filesValues = new ContentValues();
			filesValues.put(FilesTable.KEY_LOCAL_URL, path);
			filesValues.put(FilesTable.KEY_LOCAL_URL_UPDATE_TIMESTAMP, System.currentTimeMillis());
			getDB().update(filesValues, FilesTable.KEY_ID + " =? ", new String[]{String.valueOf(fileRowId)});
			return null;
		});
	}
	/**
	 * Finds and return number of localUrl with the oldest timestamp (localUrlTimestamp)
	 * @param fileTypesList a list containing file type strings.
	 * @return
	 */
	public DataBaseCommand<ArrayList<String>> getMultipleOldestLocalPathFromDB(final String targetId, final int limit, @Nullable final List<String> fileTypesList) {
		return new DataBaseCommand<>(() -> {

			ArrayList<String> oldFileList = new ArrayList<>();

			String localUrl;

			// select localUrl from files, messages, conversations
			// where conversations.targetId='qa33339197'
			// and localUrl <> ""
			// and conversations.conversationId=messages.convID
			// and files.relatedMessageRowID = messages._id
			// order by localUrlTimestamp limit <limit>;

			StringBuilder sb = new StringBuilder();
			ArrayList<Object> optionalParams = new ArrayList<>();
			optionalParams.add(targetId);
			sb.append("select ").append(FilesTable.KEY_LOCAL_URL).append(" from ").append(FilesTable.FILES_TABLE).append(", ")
					.append(MessagesTable.MESSAGES_TABLE).append(", ")
					.append(DialogsTable.TABLE_NAME)
					.append(" where ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.TARGET_ID).append("= ?")
					.append(" and ").append(FilesTable.KEY_LOCAL_URL).append(" <> '' ");

			// If the fileTypesString provided add it to the query
			if (fileTypesList != null && !fileTypesList.isEmpty()) {
				StringBuilder inOperator = new StringBuilder(") IN (");
				for (int i = 0; i < fileTypesList.size(); i++) {
					if (i == 0) {
						inOperator.append(" ? ");
					} else {
						inOperator.append(", ? ");
					}
				}
				optionalParams.addAll(fileTypesList);
				inOperator.append(") ");
				sb.append(" and LOWER(").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_FILE_TYPE).append(inOperator);
			}

			optionalParams.add(limit);
			sb.append(" and ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
					.append("=").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_DIALOG_ID)
					.append(" and ").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID)
					.append("=").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ID)
					.append(" order by ").append(FilesTable.KEY_LOCAL_URL_UPDATE_TIMESTAMP)
					.append(" limit ? ;");

			String sql = sb.toString();

			LPLog.INSTANCE.d(TAG, "query: " + sql);

			Cursor cursor = getDB().rawQuery(sql, optionalParams);

			if (cursor != null) {
				try {
					if (cursor.moveToFirst()) {

						do{
							localUrl = cursor.getString(0);
							LPLog.INSTANCE.d(TAG, "getMultipleOldestLocalPathFromDB: Add to list old file with local url: " + localUrl);
							oldFileList.add(localUrl);

						} while (cursor.moveToNext());

					}
				} finally {
					cursor.close();
				}
			}
			return oldFileList;
		});
	}

	/**
	 * Finds and remove the given localUrl from the relevant record
	 *
	 * @param localUrl
	 * @return
	 */
	public DataBaseCommand<Integer> removeLocalPathFromDB(final String localUrl) {
		return new DataBaseCommand<>(() -> {

			LPLog.INSTANCE.d(TAG, "query: searching and removing localUrl: " + localUrl);

			ContentValues localPathValues = new ContentValues();
			localPathValues.put(FilesTable.KEY_LOCAL_URL, "");
			localPathValues.put(FilesTable.KEY_LOAD_STATUS, FilesTable.LoadStatus.NOT_STARTED.ordinal());

			FileMessage fileByLocalUrl = getFileByLocalUrl(localUrl);
			long messageRelatedRowId = -1;
			if (fileByLocalUrl != null){
				messageRelatedRowId = fileByLocalUrl.getRelatedMessageRowID();
			}
			int numOfRowsAffected = getDB().update(localPathValues,
					FilesTable.KEY_LOCAL_URL + "=?",
					new String[]{localUrl});

			// Update the UI
			MessagingFactory.getInstance().getController().amsMessages.updateMessageFileChanged(messageRelatedRowId);

			LPLog.INSTANCE.d(TAG, "query: removed " + numOfRowsAffected + " records");
			return numOfRowsAffected;

		});

	}

	/**
	 * Get the number of image records that has a localUrl (full image)
	 * @param fileTypesString a string containing file type string separated with commas (e.g. 'jpg','png')
	 * @return
	 */
	public DataBaseCommand<Integer> getNumOfLocalPathFromDB(final String targetId, @Nullable final List<String> fileTypesString){

		return new DataBaseCommand<>(() -> {

			int count = 0;

			// select count(*) from files, messages, conversations
			// where conversations.targetId='qa33339197'
			// and localUrl <> ''
			// and conversations.conversationId=messages.convID
			// and  files.relatedMessageRowID=messages._id;
			StringBuilder sb = new StringBuilder();
			ArrayList<String> optionalParams = new ArrayList<>();
			optionalParams.add(targetId);
			sb.append("select count(*) from ").append(FilesTable.FILES_TABLE).append(", ")
			.append(MessagesTable.MESSAGES_TABLE).append(", ")
			.append(DialogsTable.TABLE_NAME)
			.append(" where ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.TARGET_ID).append("= ? ")
			.append(" and ").append(FilesTable.KEY_LOCAL_URL).append(" <> '' ");


			// If the fileTypesString provided add it to the query
			if (fileTypesString != null && !fileTypesString.isEmpty()) {
				StringBuilder inOperator = new StringBuilder(") IN (");
				for (int i = 0; i < fileTypesString.size(); i++) {
					if (i == 0) {
						inOperator.append(" ? ");
					} else {
						inOperator.append(", ? ");
					}
				}
				optionalParams.addAll(fileTypesString);
				inOperator.append(") ");
				sb.append(" and LOWER(").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_FILE_TYPE).append(inOperator);
			}

			sb.append(" and ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
			.append("=").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_DIALOG_ID)
			.append(" and ").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID)
			.append("=").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ID);

			String sql = sb.toString();
			String[] paramsArray = new String[optionalParams.size()];
			optionalParams.toArray(paramsArray);
			Cursor cursor = getDB().rawQuery(sql, paramsArray);

			if (cursor != null) {
				try {
					if (cursor.moveToFirst()) {

						count = cursor.getInt(0);

						LPLog.INSTANCE.d(TAG, "query: number of records with localUrl: " + count);
					}
				} finally {
					cursor.close();
				}
			}
			return count;
		});
	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 * @param fileRowId
	 * @return
	 */
	public FileMessage getFileByFileRowIdOnDbThread(long fileRowId) {
		Cursor cursor = getDB().rawQuery("SELECT * FROM " + FilesTable.FILES_TABLE + " WHERE "+
						FilesTable.KEY_ID + "=? " ,
				String.valueOf(fileRowId));
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return FileMessage.fromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

	public DataBaseCommand<FileMessage> getFileByFileRowId(final long fileRowId) {
		return new DataBaseCommand<>(() -> getFileByFileRowIdOnDbThread(fileRowId));
	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 * @param messageRowId
	 * @return
	 */
	public FileMessage getFileByMessageRowId(long messageRowId) {
		Cursor cursor = getDB().rawQuery("SELECT * FROM " + FilesTable.FILES_TABLE + " WHERE "+
				FilesTable.KEY_RELATED_MESSAGE_ROW_ID + "=? " ,
				String.valueOf(messageRowId));
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return FileMessage.fromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 * return the message row id for given file's local url
	 * @param localUrl
	 * @return message row id
	 */
	private FileMessage getFileByLocalUrl(String localUrl) {
		Cursor cursor = getDB().query(null,FilesTable.KEY_LOCAL_URL + "=? ", new String[]{localUrl}, null, null, null);
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return FileMessage.fromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

	public interface ExecutionCallback<T> {

		void onResult(T... data);
	}
}
