@file:JvmName("HybridSDKUtils")

package com.liveperson.messaging.hybrid

import com.liveperson.infra.ConversationViewParams
import com.liveperson.infra.ForegroundService
import com.liveperson.infra.LPConversationsHistoryStateToDisplay
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.messaging.R
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.controller.ConnectionsController
import com.liveperson.messaging.hybrid.commands.exceptions.IncorrectConversationHistoryStateException
import com.liveperson.messaging.hybrid.commands.exceptions.SDKNotInitializedException
import com.liveperson.messaging.hybrid.commands.exceptions.SDKNotLaunchedException
import com.liveperson.messaging.hybrid.commands.exceptions.ConnectionNotReadyException

/**
 * Extension function over [Messaging] class used as fail-fast receiver of connection controller.
 * If connection controller is null, then [SDKNotInitializedException] will be thrown.
 */
fun Messaging.requireConnectionController(): ConnectionsController {
    return mConnectionController ?: throw SDKNotInitializedException("SDK is not fully initialized.")
}

/**
 * Extension function over [Messaging] class used as fail-fast receiver of conversation view params
 * used to represent a conversation ui.
 * If connection controller is null, then [SDKNotLaunchedException] will be thrown.
 */
fun Messaging.requireConversationViewParams(): ConversationViewParams {
    return conversationViewParams ?: throw SDKNotLaunchedException("SDK is wasn't launched. Conversation view params missing.")
}

/**
 * Extension function over [Messaging] class used as fail-fast receiver for messaging controller.
 * If controller is null, or it wasn't initialized, or active brand's id is null, then
 * [SDKNotInitializedException] will be thrown.
 */
internal fun Messaging?.requireInitialization(): Messaging {
    if (this == null || !isInitialized || activeBrandId == null) {
        throw SDKNotInitializedException("SDK is not initialized.")
    }
    return this
}

/**
 * Extension function over [Messaging] class used as fail-fast receiver for messaging controller with
 * foregrounded state.
 * If [ForegroundService.isBrandForeground] returns false for an active brand id, then
 * [SDKNotLaunchedException] will be thrown.
 */
fun Messaging.requireForegroundState(): Messaging {
    val foregroundService = ForegroundService.getInstance()
    if (!foregroundService.isBrandForeground(activeBrandId!!)) {
        throw SDKNotLaunchedException("Conversation fragment is in background.")
    }
    return this
}

/**
 * Extension function over [Messaging] class used as fail-fast receiver for messaging controller with
 * connected state.
 * If [ConnectionsController.isUpdated] returns false for an active brand id and offline messaging
 * is disable, then [ConnectionNotReadyException] will be thrown.
 */
fun Messaging.requireConnection(): Messaging {
    val isOfflineMessagingEnabled = Configuration.getBoolean(R.bool.lp_is_offline_messaging_enabled)
    val connectionController = requireConnectionController()
    val brandId = activeBrandId!!
    if (!connectionController.isUpdated(brandId) && !isOfflineMessagingEnabled) {
        throw ConnectionNotReadyException("Connection is not ready to perform an operation.")
    }
    return this
}

/**
 * Extension function over [Messaging] class used as fail-fast receiver for messaging controller with
 * ability to send messages.
 * If [ConversationViewParams.historyConversationsStateToDisplay] is set to
 * [LPConversationsHistoryStateToDisplay.CLOSE], then SDK is not able to send any messages and
 * [IncorrectConversationHistoryStateException] will be thrown.
 */
fun Messaging.requireStateToSendMessage(): Messaging {
    val controller = requireConnection().requireForegroundState()
    val params = controller.requireConversationViewParams()
    if (params.historyConversationsStateToDisplay == LPConversationsHistoryStateToDisplay.CLOSE) {
        throw IncorrectConversationHistoryStateException("Could not perform operation. historyConversationsStateToDisplay is set to false")
    }
    return controller
}