package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.annotation.VisibleForTesting;

import android.text.TextUtils;
import android.text.format.DateUtils;

import android.util.Patterns;

import com.liveperson.api.request.message.BasePublishMessage;
import com.liveperson.api.request.message.FilePublishMessage;
import com.liveperson.api.request.message.FormPublishMessage;
import com.liveperson.api.request.message.FormSubmissionPublishMessage;
import com.liveperson.api.request.message.StructuredContentPublishMessage;
import com.liveperson.api.request.message.TextPublishMessage;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.Event;
import com.liveperson.api.response.model.Participants;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.LPConversationHistoryMaxDaysDateType;
import com.liveperson.infra.LPConversationsHistoryStateToDisplay;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.controller.DBEncryptionService;
import com.liveperson.infra.controller.DBEncryptionServiceUtilsKt;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DBUtilities;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.BaseTable;
import com.liveperson.infra.database.tables.ConversationsTable;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.database.transaction_helper.InsertOrUpdateSQLCommand;
import com.liveperson.infra.database.transaction_helper.InsertSQLCommand;
import com.liveperson.infra.database.transaction_helper.SQLiteCommand;
import com.liveperson.infra.database.transaction_helper.UpdateSQLCommand;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.infra.utils.LinkUtils;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.infra.utils.patterns.PatternsCompat;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.commands.DeliveryStatusUpdateCommand;
import com.liveperson.messaging.commands.pusher.ClearUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.pusher.GetUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.pusher.SendReadAcknowledgementCommand;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.http.MessageTimeoutListener;
import com.liveperson.messaging.offline.OfflineMessagesRepositoryImpl;
import com.liveperson.messaging.offline.OfflineMessagesRepositoryImpl.OfflineMessagesListener;
import com.liveperson.messaging.offline.api.OfflineMessagesRepository;
import com.liveperson.messaging.provider.FileMessageProviderImpl;
import com.liveperson.messaging.provider.FileUploadProgressProviderImpl;
import com.liveperson.messaging.utils.MessagingConst;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000076;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000077;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000078;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000079;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007B;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007C;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000080;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000081;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000082;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000087;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000088;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000152;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000153;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000154;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000016B;
import static com.liveperson.messaging.model.MessagingChatMessage.MessageType.isConsumer;
import static com.liveperson.messaging.model.MessagingChatMessage.MessageType.isConsumerUnmasked;

import org.json.JSONException;
import org.json.JSONObject;

import kotlin.text.StringsKt;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Messages model
 */
public class AmsMessages extends BaseDBRepository implements ShutDown, Clearable, AmsMessagesLoaderProvider {

	public static final int MESSAGES_LIMIT = 25;
	public static final int QUICK_REPLY_MESSAGE_SEQUENCE = -6;

	private static final String TAG = "AmsMessages";


	public enum MessagesSortedBy {
		TargetId, ConversationId, DialogId
	}

	public interface MessagesListener {

		void initMessages(ArrayList<FullMessageRow> searchedMessageList);

		void onQueryMessagesResult(long firstTimestamp, long lastTimestamp);

		void onUpdateMessages(long firstTimestamp, long lastTimestamp);

		void onNewMessage(FullMessageRow fullMessageRow);

		/**
		 * Method used to represent welcome message that would
		 * not be stored in database until conversation is started.
		 * @param row data to represent welcome message.
		 * @param welcomeMessage welcome message to would be represented.
		 * @param callback callback to notify whether welcome message was show.
		 *                 If welcome message wasn't shown its metadata wouldn't be
		 *                 stored.
		 */
		void onNewWelcomeMessage(@NonNull FullMessageRow row, @NonNull LPWelcomeMessage welcomeMessage, @NonNull OnWelcomeMessageShownCallback callback);

		void onUpdateMessage(FullMessageRow fullMessageRow);

		void onUpdateFileMessage(String eventId, long localId, FileMessage fullMessageRow);

		void onInitialMessagesReceived(List<FullMessageRow> rows);

		/**
		 * Method used to notify messages list about removal
		 * of message.
		 * @param eventId event id of message that was removed from
		 *                database.
		 */
		void removeMessageByEventId(String eventId);

		/**
		 * This only being used when clear conversation history button clicked.
		 */
		void removeAllClosedConversations(String targetId);

		/**
		 * Clear all messages. Used when move to background for un-auth account or the SDK version change.
		 */
		void clearAllMessages(String targetId);

		void onHistoryFetched();

		void onExConversationHandled(boolean emptyNotification);

		void onHistoryFetchedFailed();

		void addFirstWelcomeMessage();

		void onAgentReceived(MessagingUserProfile userProfile);

		void showErrorToast(@StringRes int messageRes);

		int getLoadLimit();

		/**
		 * Method used update recent represented welcome message with actual data
		 * of new create dialog to prevent duplicated representation of welcome message.
		 * @param dialogId dialog that contains welcome message.
		 * @param welcomeMessageRow row with actual data from database.
		 */
		void updateWelcomeMessageForDialogId(String dialogId, FullMessageRow welcomeMessageRow);

		/**
		 * Method used update recent temporal represented welcome message with actual data
		 * of new create dialog to prevent duplicated representation of welcome message.
		 * @param welcomeMessageRow row with actual data from database.
		 */
		void updateTempWelcomeMessage(FullMessageRow welcomeMessageRow);

		/**
		 * Method used update to reorder offline message after successfully sending it
		 * to the server.
		 * @param eventId event id of offline message that was susccessfully sent
		 * @param timestamp actual timestamp of sent message.
		 */
		void updateMessageTimestampByEventId(String eventId, long timestamp);

		/**
		 * Callback used to determine whether welcome message representation
		 * validation was passed to save welcome message metadata.
		 */
		interface OnWelcomeMessageShownCallback {

			/**
			 * Method used to notify whether welcome message was
			 * added to conversation screen successfully.
			 *
			 * @see MessagesListener#onNewWelcomeMessage(FullMessageRow, LPWelcomeMessage, OnWelcomeMessageShownCallback)
			 * @param isAdded equals true when welcome message was added successfully
			 *                or false otherwise.
			 */
			void onWelcomeMessageShown(boolean isAdded);

			/**
			 * Method used to notify whether outbound messsage was
			 * represented.
			 *
			 * @see MessagesListener#onNewWelcomeMessage(FullMessageRow, LPWelcomeMessage, OnWelcomeMessageShownCallback)
			 */
			void onOutboundMessageShown();
		}
	}

    private static final int MAX_SQL_VARIABLES = 997;

	public static final int PENDING_MSG_SEQUENCE_NUMBER = -1;
	/**
	 * Sequence number of conversation resolved message which conversation was
	 * resolved by agent or consumer.
	 */
	public static final int RESOLVE_MSG_SEQUENCE_NUMBER = -2;
	public static final int MASKED_CC_MSG_SEQUENCE_NUMBER = -3;
	public static final int WELCOME_MSG_SEQUENCE_NUMBER = -4;
	public static final int OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER = -5;
	public static final int TRANSCENDENT_MESSAGE_SEQUENCE_NUMBER = -7;
	/**
	 * Sequence number of conversation resolved message which conversation was
	 * auto closed.
	 */
	public static final int RESOLVED_AUTO_CLOSED_MSG_SEQUENCE_NUMBER = -8;

	/**
	 * Local timestamp for welcome messages that were represented for offline mode.
	 * Such message should be stuck in the bottom of active conversation until
	 * user sends message/taps on quick reply option.
	 * The maximum unsigned long that could be stored in database should be
	 * less than Long.MAX_VALUE, otherwise sqlite database will return us
	 * a negative timestamp, that is equal to LONG.MIN_VALUE.
	 *
	 * Welcome message could have a quick reply, that's why we need to subtract
	 * 2, because next message could be welcome message's quick reply.
	 */
	public static final long OFFLINE_WELCOME_MESSAGE_TIMESTAMP = Long.MAX_VALUE - 2;

	private static final String TYPE_CONVERSATION_ACCEPTED = "YOU_ARE_CONNECTED_TO";
	private static final String KEY_NOTIFICATION_METADATA_MESSAGE_TYPE = "messageType";

	public static final String WELCOME_MESSAGE_EVENT_ID_POSTFIX = "-wm";
	public static final String RESOLVE_MESSAGE_EVENT_ID_POSTFIX = "-rm";

	public static final String MASKED_MESSAGE_EVENT_ID_POSTFIX = "-mm";

	private boolean shouldAddWelcomeMessage = false;
	private Map<String, MessagingUserProfile> mMessagingProfiles = new HashMap<>();

    private final Messaging mController;
	public final FormsManager mFormsManager;

	private MessagesListener mMessagesListener = null;
	private MessagesListener mNullMessagesListener = new NullMessagesListener();

	// Holds the most recent agent message quick replies data
	private QuickRepliesMessageHolder mQuickRepliesMessageHolder = null;

	private volatile boolean containsMessagesInDatabse = false;
	private volatile boolean containsPCSMessagesInDatabse = false;
	private AtomicLong lastMessageTimeStamp = new AtomicLong(Long.MAX_VALUE);
	/**
	 * Locker used to synchronize updates of temporary
	 * messages in database.
	 *
	 * @see MessagesListener#getTimestampForWelcomeMessage(String, long)
	 * @see MessagesListener#updateMessagesDialogServerID(String, String)
	 */
	private final Object DIALOG_ID_LOCKER = new Object();

	private final OfflineMessagesRepository mOfflineMessagesRepository;

	public AmsMessages(Messaging controller) {
		super(MessagesTable.MESSAGES_TABLE);

		mController = controller;

        mOfflineMessagesRepository = new OfflineMessagesRepositoryImpl(
				PreferenceManager.getInstance(),
				new FileMessageProviderImpl(mController),
				new FileUploadProgressProviderImpl(mController),
				new OfflineMessagesListener() {
					@Override
					public void onUpdateMessages(long firstTimestamp, long lastTimestamp) {
						getMessagesListener().onUpdateMessages(firstTimestamp, lastTimestamp);
					}

					@Override
					public void removeMessageByEventId(@NonNull String eventId) {
						getMessagesListener().removeMessageByEventId(eventId);
					}
				}
		);

		MessageTimeoutListener mMessageTimeoutListener = new MessageTimeoutListener() {
			@Override
			public void onMessageTimeout(String brandId) {
				mController.onMessageTimeout(brandId);
				LPLog.INSTANCE.e(TAG, ERR_00000076, "on message timeout received");
			}

			@Override
			public void onPublishMessageTimeout(String brandId, String eventId, String dialogId) {
				getMessageByEventId(eventId).setPostQueryOnBackground(data -> {
					if (data != null) {
						LPLog.INSTANCE.i(TAG, "onPublishMessageTimeout: "
								+ "\ngetMessageByEventId = " + eventId
								+ "\nmessage: = " + data.getMessage()
								+ "\nstate = " + data.getMessageState());
						// if the message has already been sent and received/read, no need to change it's state
						if (data.getMessageState() != MessagingChatMessage.MessageState.RECEIVED
								&& data.getMessageState() != MessagingChatMessage.MessageState.READ) {
							showErrorToast(data);
							updateMessageState(eventId, MessagingChatMessage.MessageState.ERROR);
							LPLog.INSTANCE.e(TAG, ERR_00000077, "on update message timeout");
						}
					}
					else {
						updateMessageState(eventId, MessagingChatMessage.MessageState.ERROR);
						LPLog.INSTANCE.e(TAG, ERR_00000077, "on update message timeout");
					}
				}).execute();
			}
		};
		mMessageTimeoutQueue = new MessageTimeoutQueue(mMessageTimeoutListener);
		mFormsManager = new FormsManager();
	}

	// A queue for storing messages for a specific timeout so we can mark them as error if not getting ack on them
	public final MessageTimeoutQueue mMessageTimeoutQueue;

	/**
	 * Notify of data change from change of agent profile
	 */
	public DataBaseCommand<Void> updateOnCommand() {

		return new DataBaseCommand<>(() -> null);
	}

	/**
	 * MUST RUN ON DB THREAD!!
	 *
	 * @param brandID
	 * @param limitSize          max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp negative value if there is no need to use this value
	 * @param newerThanTimestamp negative value if there is no need to use this value
	 * @return cursor with messages according to params
	 */
	private Cursor messagesByTarget(String brandID, int limitSize, long olderThanTimestamp, long newerThanTimestamp) {
		ArrayList<Object> params = new ArrayList<>();
		StringBuilder sql = getMessagesForTargetQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.BRAND_ID).append(" = ? ");
		params.add(brandID);

		if (olderThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" <= ? ");
			params.add(olderThanTimestamp);
		}
		if (newerThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" >= ? ");
			params.add(newerThanTimestamp);
		}

		// LE-79838 [Android] History Control API :
		// https://confluence.liveperson.com/display/SeattleMobile/History+Control+API+-+Improved

		ConversationViewParams conversationViewParams = mController.getConversationViewParams();
		long historyConversationsMaxDays = conversationViewParams.getHistoryConversationsMaxDays() * DateUtils.DAY_IN_MILLIS;

		LPConversationsHistoryStateToDisplay historyConversationState = conversationViewParams.getHistoryConversationsStateToDisplay();
		LPConversationHistoryMaxDaysDateType maxDaysDateType = conversationViewParams.getHistoryConversationMaxDaysType();

		LPLog.INSTANCE.i(TAG, "History control API params: \n ConversationsHistoryStateToDisplay: " + historyConversationState.toString()
				+ "\n ConversationHistoryMaxDaysDateType: " + maxDaysDateType.toString()
				+ "\n HistoryConversationsMaxDays: " + conversationViewParams.getHistoryConversationsMaxDays());

		long daysAgo = System.currentTimeMillis() - historyConversationsMaxDays;

		switch (historyConversationState) {
			case ALL:
				if (historyConversationsMaxDays > -1) {
					// Fetch all conversations for OPEN dialog despite value of history days.
					sql.append(" AND ")
							.append("(") // Outer parenthesis

							// Append dialog types
							.append("(")
							.append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.OPEN.ordinal())
							.append(" OR ")
							.append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.LOCKED.ordinal()) // Welcome msg dialog
							.append(")");

					// If value of history days is positive, fetch conversations for CLOSE dialog within value of history days.
					if (historyConversationsMaxDays > 0) {
						sql.append(" OR ")
								.append("(") // Inner parenthesis
								.append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.CLOSE.ordinal());
						setMaxDaysDateTypeToQuery(maxDaysDateType, sql, daysAgo);
						sql.append(")"); // End inner parenthesis
					}
					sql.append(")"); // End outer parenthesis
				}
				break;
			case OPEN:
				// Fetch all conversations only for OPEN and LOCKED (Welcome msg) dialog despite value of history days.
				sql.append(" AND ")
						// Append dialog types
						.append("(")
						.append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.OPEN.ordinal())
						.append(" OR ")
						.append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.LOCKED.ordinal())  // Welcome msg dialog
						.append(")");
				break;
			case CLOSE:
				if (historyConversationsMaxDays == 0) {
					return null;
				} else if (historyConversationsMaxDays > 0) {
					// If value of history days is positive, fetch conversations within value of history days.
					setMaxDaysDateTypeToQuery(maxDaysDateType, sql, daysAgo);
				}
				// Fetch all conversations only for CLOSE dialog
				sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.CLOSE.ordinal());
				break;
		}

		//end of LE-79838 [Android] History Control API
		if (limitSize > 0) {
			params.add(limitSize);
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" DESC ");
			sql.append(" LIMIT ? ");
			sql = new StringBuilder("Select * FROM ( ").append(sql).append(" ) ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		} else {
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		}
		return getDB().rawQuery(sql.toString(), params);
	}

	/**
	 * Append clause to fetch conversations only under specific states: start conversation date or end conversation date.
	 *
	 * @param maxDaysDateType Conversation state: start or end conversation date.
	 * @param query Database query to modify
	 * @param daysAgo History days
	 */
	private void setMaxDaysDateTypeToQuery(LPConversationHistoryMaxDaysDateType maxDaysDateType, StringBuilder query, long daysAgo) {
		switch (maxDaysDateType) {
			case endConversationDate:
				query.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.END_TIMESTAMP).append(" >= ").append(daysAgo);
				break;
			case startConversationDate:
				query.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.START_TIMESTAMP).append(" >= ").append(daysAgo);
				break;
		}
	}

	/**
	 * @param conversationID
	 * @param limitSize      -1 for no limit
	 * @return
	 */
	private Cursor messagesByConversationID(String conversationID, int limitSize) {
		ArrayList<Object> params = new ArrayList<>();
		params.add(conversationID);
		StringBuilder sql = getBasicMessagesQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.DIALOG_ID).append(" = ? ").append(" ORDER BY ")
				.append(MessagesTable.KEY_TIMESTAMP);

		if (limitSize != -1) {
			sql.append(" LIMIT ? ");
			params.add(limitSize);
		}
		return getDB().rawQuery(sql.toString(), params);
	}

	/**
	 * @param dialogID
	 * @param limitSize      -1 for no limit
	 * @return
	 */
	private Cursor messagesByDialogID(String dialogID, int limitSize) {
		ArrayList<Object> params = new ArrayList<>();
		params.add(dialogID);
		StringBuilder sql = getBasicMessagesQuery().append(" WHERE ").append(MessagesTable.MESSAGES_TABLE)
				.append(".").append(MessagesTable.KEY_DIALOG_ID).append(" = ?").append(" ORDER BY ")
				.append(MessagesTable.KEY_TIMESTAMP);

		if (limitSize != -1) {
			sql.append(" LIMIT ? ");
			params.add(limitSize);
		}
		return getDB().rawQuery(sql.toString(), params);
	}

	private StringBuilder getMessagesForTargetQuery() {
		return new StringBuilder()
				.append("select ")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(BaseTable.KEY_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_EVENT_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(MessagesTable.KEY_SERVER_SEQUENCE).append(",")
				.append(MessagesTable.KEY_DIALOG_ID).append(",")
				.append(MessagesTable.KEY_TEXT).append(",")
				.append(MessagesTable.KEY_CONTENT_TYPE).append(",")
				.append(MessagesTable.KEY_MESSAGE_TYPE).append(",")
				.append(MessagesTable.KEY_STATUS).append(",")
				.append(MessagesTable.KEY_TIMESTAMP).append(",")
				.append(UsersTable.KEY_PROFILE_IMAGE).append(",")
				.append(UsersTable.KEY_NICKNAME).append(",")
				.append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_ID).append(" AS ").append(FilesTable.KEY_ID_AS_VALUE).append(",")
				.append(FilesTable.KEY_FILE_TYPE).append(",")
				.append(FilesTable.KEY_LOCAL_URL).append(",")
				.append(FilesTable.KEY_PREVIEW).append(",")
				.append(FilesTable.KEY_LOAD_STATUS).append(",")
				.append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID).append(",")
				.append(FilesTable.KEY_SWIFT_PATH)
				.append(" from ")
				.append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID)
				.append(" left join ").append(FilesTable.FILES_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ID)
				.append("=").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID);
	}

	@NonNull
	private StringBuilder getBasicMessagesQuery() {
		return new StringBuilder().append("select ")
				.append(MessagesTable.MESSAGES_TABLE).append("._id, serverSequence,text,contentType,dialogId,type,status,")
				.append(MessagesTable.MESSAGES_TABLE).append(".eventId,")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(UsersTable.KEY_NICKNAME).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".originatorId,timeStamp,"+MessagesTable.MESSAGES_TABLE+".encryptVer,")
				.append("description,firstName,lastName,phoneNumber,userType,email,profileImage,coverImage from ").append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID);
	}


	/**
	 * Method used to show a welcome message without saving it to database.
	 * @param message message to show.
	 * @param welcomeMessage actual welcome message (dynamic or from conversation params)
	 * @param callback callback required to store metadata if message was represented
	 */
	public void showWelcomeMessage(
			MessagingChatMessage message,
			LPWelcomeMessage welcomeMessage,
			MessagesListener.OnWelcomeMessageShownCallback callback
	) {
		FullMessageRow row = createFullMessageRow(-1, message, -1);
		getMessagesListener().onNewWelcomeMessage(row, welcomeMessage, callback);
	}

	/**
	 * Add a message, even if the conversation is in pending state
	 */
	public DataBaseCommand<Long> addMessage(final MessagingChatMessage message, final boolean updateUI) {
		return new DataBaseCommand<>(() -> {
			long rowId;
			boolean isEmptyEventId = TextUtils.isEmpty(message.getEventId());
			if (isEmptyEventId) {
				LPLog.INSTANCE.i(TAG, "Received new message without event id, generating new one.. ");
				message.setEventId(UniqueID.createUniqueMessageEventId());
				//in case there is no event id we need to check if such message exists
				//in the db by the conversation id and sequence. is exists - update it,
				//if not, insert this message.
				String whereClause = MessagesTable.KEY_DIALOG_ID + " = ? AND " +
						MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
				ContentValues updateValues =  DBEncryptionServiceUtilsKt.encrypt(getContentValuesForMessageUpdate(message));
				ContentValues insertValues = wrapValuesForInsertion(updateValues, message);

				rowId = getDB().insertOrUpdate(
						insertValues,
						updateValues,
						whereClause,
						new String[]{message.getDialogId(), String.valueOf(message.getServerSequence())});
				LPLog.INSTANCE.d(TAG, "Insert or Update message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
			} else {
				// if it is a welcome message of pending conversation then need to prepare metadata
				if (message != null && AmsDialogs.KEY_WELCOME_DIALOG_ID.equals(message.getDialogId())
						&& (message.getServerSequence() == OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER)) {
					prepareWelcomeMessageMetadata(message, null);
				}

				//Check if the message exist
				try (Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null)) {
					if (cursor != null && cursor.getCount() > 0) {
						ContentValues messageValues = getContentValuesForMessageUpdate(message, cursor, false);

						//If needed update the message in DB
						if (messageValues.size() > 0) {
							rowId = getDB().update(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
									new String[]{String.valueOf(message.getEventId())});
							LPLog.INSTANCE.d(TAG, "Adding message: This message was update with message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
						} else {
							rowId = -1;
							LPLog.INSTANCE.d(TAG, "Adding message: Skip add\\update this message since its already exist" + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
						}
					} else {
						// Add the received message to DB
						// update offline message with first sent
						// message from consumer
						if (MessagingChatMessage.MessageType.isConsumer(message.getMessageType())) {
							updateOfflineWelcomeMessage(message.getTimeStamp() - 2, null);
						} else if (MessagingChatMessage.MessageType.isSystemResolved(message.getMessageType())) {
							updateOfflineWelcomeMessage(message.getTimeStamp() - 2, message.getDialogId());
						}
						ContentValues insertValues = getContentValuesForMessage(message);
						ContentValues encrypted = DBEncryptionServiceUtilsKt.encrypt(insertValues);
						rowId = getDB().insertWithOnConflict(encrypted);
						LPLog.INSTANCE.d(TAG, "Adding message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
					}
				} catch (Exception exception){
					rowId = -1;
					LPLog.INSTANCE.e(TAG, ERR_00000088, "addMessage", exception);
				}
			}

			if (updateUI) {
				if (rowId != -1) {
					//new Item

					getMessagesListener().onNewMessage(createFullMessageRow(rowId, message, -1));
				} else {
					//updated item - need to read again from db cause 'message' might contain different data than db
					//we don;t have the row id on updated message - need to update by event id.
					//updateMessageByRowIdOnDbThread(rowId);
					String eventIdToUpdate = message.getEventId();
					if (isEmptyEventId) {
						LPLog.INSTANCE.d(TAG, "Updating message that originally didn't have event id. ");
						eventIdToUpdate = getEventIdForMessage(message.getDialogId(), message.getServerSequence());
					}
					if (!TextUtils.isEmpty(eventIdToUpdate)) {
						updateMessageByEventId(eventIdToUpdate);
					}
				}
			}

			return rowId;
		});
	}

	public DataBaseCommand<Void> addMultipleMessages(
			final ArrayList<ContentEventNotification> responseMessages,
			final Dialog dialog,
			final String originatorId,
			final String brandId,
			final String targetId,
			final String dialogId,
			final String conversationId,
			final long clockDiff,
			final boolean firstNotification,
			final boolean mShouldUpdateUI
	) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {

			public String[] extractLinks(String text) {
				List<String> links = new ArrayList<>();
				String[] l = text.split("\\s+");
				for (int i = 0; i < l.length; i++) {
					String possibleLink = l[i];
					boolean isLink = PatternsCompat.AUTOLINK_WEB_URL.matcher(possibleLink).matches();
					isLink |= Patterns.EMAIL_ADDRESS.matcher(possibleLink).matches();
					if (isLink) {
						links.add(possibleLink);
					}
				}
				return links.toArray(new String[links.size()]);
			}

			private MessagingChatMessage.MessageType checkIfMessageContainsURLandChangeType(MessagingChatMessage.MessageType type, String msg) {
				if (LinkUtils.containsMarkdownHyperlink(msg) && type == MessagingChatMessage.MessageType.AGENT) {
					return MessagingChatMessage.MessageType.AGENT_MARKDOWN_HYPERLINK;
				}
				String[] urls = extractLinks(msg);
				if (urls.length > 0) {
					if (type == MessagingChatMessage.MessageType.CONSUMER) {
						return MessagingChatMessage.MessageType.CONSUMER_URL;
					}
					if (type == MessagingChatMessage.MessageType.CONSUMER_MASKED) {
						return MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
					}
					if (type == MessagingChatMessage.MessageType.AGENT) {
						return MessagingChatMessage.MessageType.AGENT_URL;
					}
				}
				return type;
			}

			@Override
			public Void query() {
				ArrayList<SQLiteCommand> commands;
				if (responseMessages != null) {
					LPLog.INSTANCE.d(TAG, "Start addMultipleMessages. num of commands = " + responseMessages.size());
					ArrayList<FullMessageRow> messageRows = new ArrayList<>();
					List<MessagesStatusHolder> messagesStatuses = new ArrayList<>();

					commands = new ArrayList<>(responseMessages.size());


					MessagingChatMessage.MessageState messageState;
					MessagingChatMessage.MessageType messageType;
					int[] sequenceList;
					ContentValues contentValues;
					StringBuilder whereBuilder;
					String[] whereArgs;
					Set<Integer> controllerMessagesSequence = new HashSet<>();

					int firstSequence = -1;
					int lastSequence = -1;

					int maxAcceptStatusSequence = -1;
					int maxReadStatusSequence = -1;

					int lastAgentMessageSequence = -1;

					for (int index = 0; index < responseMessages.size(); index++) {

						ContentEventNotification notification = responseMessages.get(index);

				   		if (notification.event == null){
							LPLog.INSTANCE.e(TAG, ERR_00000078, "received message with empty event! continuing to next message.. " );
							continue;
						}

						switch (notification.event.type) {
							case ContentEvent: {
								// check metadata for welcome message
								if (notification.sequence == 0) {
									long timestamp = notification.serverTimestamp + clockDiff;
									if (notification.metadata != null && notification.metadata.length() > 0) {
										processMetadata(notification, commands, messageRows);
									} else if (!containsMessagesAfter(timestamp)){
										updateTempDialogId(notification.dialogId, WELCOME_MSG_SEQUENCE_NUMBER, timestamp - 2);
									}
								}

								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPLog.INSTANCE.e(TAG, ERR_00000079, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.fromString(notification.event.contentType);


								if (TextUtils.equals(originatorId, notification.originatorId) ||
										(notification.originatorMetadata != null && notification.originatorMetadata.mRole == Participants.ParticipantRole.CONSUMER)) {
									//consumer message - make sure it's not double
									messageType = MessagingChatMessage.MessageType.getMessageContentTypeForConsumer(notification, contentType);
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									// If this message is not from myself

									// If message from Controller
									if(notification.originatorMetadata != null && notification.originatorMetadata.mRole == Participants.ParticipantRole.CONTROLLER){
										messageType = MessagingChatMessage.MessageType.CONTROLLER_SYSTEM;
									}
									else { // Message from Agent
										messageType = MessagingChatMessage.MessageType.getMessageContentTypeForAgent(notification,contentType);
										mController.amsConversations.resetEffectiveTTR(targetId);
										mController.amsDialogs.resetEffectiveTTR(targetId);
										//saving the sequence number of the last message sent by an agent
										lastAgentMessageSequence = notification.sequence;
									}
									messageState = MessagingChatMessage.MessageState.RECEIVED;
								}

								// Create the relevant message and add to DB

								MessagingChatMessage messagingChatMessage = createMessageInDB(commands, messageRows, messageState, messageType, notification, publishMessage, contentType);
								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messagingChatMessage.getMessageType(), dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
								if (isNotificationFromController(notification)) {
									controllerMessagesSequence.add(notification.sequence);
								}
							}
							break;

							case RichContentEvent: {
								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPLog.INSTANCE.e(TAG, ERR_0000007A, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.text_structured_content;
								messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;


								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//consumer message - make sure it's not double
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									//message from agent
									messageState = MessagingChatMessage.MessageState.RECEIVED;
									mController.amsConversations.resetEffectiveTTR(targetId);
									mController.amsDialogs.resetEffectiveTTR(targetId);
									//saving the sequence number of the last message sent by an agent
									lastAgentMessageSequence = notification.sequence;
								}

								// Create the relevant message and add to DB
								MessagingChatMessage messagingChatMessage = createMessageInDB(commands, messageRows, messageState, messageType, notification, publishMessage, contentType);

								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messagingChatMessage.getMessageType(), dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
							}
							break;
							case AcceptStatusEvent:

								messageState = getReceivedMessageState(notification.event.status);
								int[] eventSequenceList = notification.event.sequenceList;

								if (messageState == null) { //TODO: investigate this line. getReceivedMessageState method marked as NonNull but there are error logs found in Kibana.
									LPLog.INSTANCE.e(TAG, ERR_0000007B, "messageState is null :| shouldn't happen! original status: " + notification.event.status +
											", dialogId = " + dialogId + " , sequence = " + Arrays.toString(eventSequenceList));

									continue;
								}
								ContentEventNotification prevNotification = index - 1 >= 0 ? responseMessages.get(index - 1) : null;

								if (prevNotification != null
										&& isNotificationFromController(prevNotification)
										&& eventSequenceList.length == 1
										&& eventSequenceList[0] == prevNotification.sequence) {
									continue;
								}
								//check who the is the origin of this AcceptStatus Event
								if (!TextUtils.equals(originatorId, notification.originatorId)) {
									//the AcceptStatus is from an Agent
									if(eventSequenceList[0] == lastAgentMessageSequence){
										LPLog.INSTANCE.d(TAG, "AcceptStatusEvent received from agent for agent message. we ignore this event. lastAgentMsgSequence = " +lastAgentMessageSequence);
										break;
									}
								}

								// Sequences of messages which represent controller's
								// message when conversation was accepted by agent or
								// messages that weren't sent by controller at all.
								// This sequence prevents applying of irregular
								// status to messages that were sent before
								// controller's one.
								List<Integer> filteredSequences = new ArrayList<>();

								if (eventSequenceList != null ) {
									List<Integer> sequencesInDatabase = null;
									for (int sequence : eventSequenceList) {
										if (!controllerMessagesSequence.contains(sequence)) {
											if (sequencesInDatabase == null) {
												sequencesInDatabase = getDBMessagesFromController(dialogId, eventSequenceList);
											}
											LPLog.INSTANCE.w(TAG, "DB SEQUENCE " + sequencesInDatabase);
											if (!sequencesInDatabase.contains(sequence)) {
												filteredSequences.add(sequence);
											}
										}
									}
								}

								if (eventSequenceList != null && eventSequenceList.length > 0) {
									int length = eventSequenceList.length;
									int minSequence = eventSequenceList[0];
									int maxSequence = eventSequenceList[length - 1];
									if (minSequence > maxSequence) {
										int temporary = minSequence;
										minSequence = maxSequence;
										maxSequence = temporary;
									}

									sequenceList = eventSequenceList;


									//saving max sequence to update all messages at the end.
									if (messageState == MessagingChatMessage.MessageState.READ  &&
											maxReadStatusSequence < maxSequence && filteredSequences.contains(maxSequence)){
										maxReadStatusSequence = maxSequence;
									}else if (messageState == MessagingChatMessage.MessageState.RECEIVED&&
											maxAcceptStatusSequence < maxSequence && filteredSequences.contains(maxSequence)){
										maxAcceptStatusSequence = maxSequence;
									}else{
										//for viewed STATUS / SUBMITTED / EXPIRED etc..
										for (int i = 0; i < length; i += (MAX_SQL_VARIABLES - 1)) {
											int[] tempArray;
											int size = (length - i > (MAX_SQL_VARIABLES - 1)) ? (MAX_SQL_VARIABLES - 1) : length - i;
											if (size == length) {
												tempArray = sequenceList;
											} else {
												tempArray = new int[MAX_SQL_VARIABLES];
												System.arraycopy(sequenceList, i, tempArray, 0, size);
											}

											contentValues = new ContentValues();
											whereBuilder = new StringBuilder();
											// Create where args array for the conversation ID and all the sequences from the list
											whereArgs = new String[size + 2];

											MessagesStatusHolder holder = createStatementForUpdateMessagesState(
													dialogId,
													tempArray,
													messageState,
													size,
													contentValues,
													whereBuilder,
													whereArgs,
													filteredSequences
											);
											messagesStatuses.add(holder);

											commands.add(new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs));
										}
									}

									//for updating UI later..
									if (firstSequence == -1) {
										firstSequence = minSequence;
									}

                                    //Update UI: only if the last sequence is less than the status sequence - update.
                                    //if not, it will be in the range between firstSequence and lastSequence
                                    if (lastSequence < maxSequence) {
                                        lastSequence = maxSequence;
                                    }
                                }

								break;
						}


					}

					LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", responseMessages.size()  = " + responseMessages.size() );

					// Save the last quick replies (if exist and if conversation is not closed)
					if (mQuickRepliesMessageHolder != null && !MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
						LPLog.INSTANCE.d(TAG, "QuickReplies exist in the received message, write to SharedPrefs");
						mQuickRepliesMessageHolder.writeToSharedPreferences();
					}

					// First we update all the read than all the Accept to minimize the updates requests!
					if (maxReadStatusSequence > -1){
						LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", maxReadStatusSequence = " + maxReadStatusSequence);
						messagesStatuses.add(new MessagesStatusHolder(dialogId, new TreeSet<>(), MessagingChatMessage.MessageState.READ, maxReadStatusSequence));
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.READ, maxReadStatusSequence));
					}

					//if accept is less or equals to read, no need to update
					if (maxAcceptStatusSequence > -1 && maxAcceptStatusSequence > maxReadStatusSequence){
						LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", maxAcceptStatusSequence = " + maxAcceptStatusSequence );
						messagesStatuses.add(new MessagesStatusHolder(dialogId, new TreeSet<>(), MessagingChatMessage.MessageState.RECEIVED, maxAcceptStatusSequence));
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.RECEIVED, maxAcceptStatusSequence));
					}

					// Sorting of messages statuses is needed to prevent situations when messages with status "Read" weren't represented as "Delivered".
					Collections.sort(messagesStatuses, (o1, o2) -> -Integer.compare(o1.getState().ordinal(), o2.getState().ordinal()));
					// synchronized block that checks whether it is
					// first load of UMS dialog or first load if INCA dialog.
					synchronized (AmsMessages.this) {
						if (!mShouldUpdateUI) {
							ConversationViewParams conversationViewParams = mController.getConversationViewParams();
							LPConversationsHistoryStateToDisplay historyConversationState = conversationViewParams.getHistoryConversationsStateToDisplay();
							if (dialog.isOpen()) {
								// Do not present open conversations if display state is CLOSE
								if (historyConversationState != LPConversationsHistoryStateToDisplay.CLOSE) {
									showInitialMessages(brandId, messageRows, messagesStatuses);
								}
							} else if (dialog.getDialogType() == DialogType.POST_SURVEY && !containsPCSMessagesInDB() && getOldestMessageFromReceived(messageRows) > getLastMessageTimestampDB()) {
								if (historyConversationState != LPConversationsHistoryStateToDisplay.OPEN) {
									showInitialMessages(brandId, messageRows, messagesStatuses);
									addResolveMessageToClosedConversation(dialog);
								}
							} else if (!containsMessagesInDB() || getOldestMessageFromReceived(messageRows) > getLastMessageTimestampDB()) {
								// Do not present closed conversations if display state is OPEN
								if (historyConversationState != LPConversationsHistoryStateToDisplay.OPEN) {
									showInitialMessages(brandId, messageRows, messagesStatuses);
									addResolveMessageToClosedConversation(dialog);
								}
							} else if (mController.amsConversations.isConversationOpened(dialog.getConversationId())) {
								if (historyConversationState != LPConversationsHistoryStateToDisplay.CLOSE) {
									Collections.sort(messageRows, FullMessageRow::newerThan);
									applyMessagesStatus(messageRows, messagesStatuses);
									getMessagesListener().onInitialMessagesReceived(messageRows);
									addResolveMessageToClosedConversation(dialog);
								}
							}
						}
					}

					encryptSqlCommandData(commands);
					getDB().runTransaction(commands);

					firstSequence = firstSequence > 1 ? firstSequence : 0;
					// remove all received messages from progress set.
					if (mShouldUpdateUI) {
						requestMessagesUpdates(firstNotification, dialogId, firstSequence, lastSequence, messageRows);
					}

					// Clear pusher's unread count for dialogId.
					sendReadAckToPusherIfRequired(conversationId, dialogId, brandId);
					//sending update on messages status.
					sendReadAckOnMessages(brandId, targetId, originatorId);
				}
				return null;
			}

			/**
			 * Check whether metadata type is welcome message (WelcomeMessage or Proactive) or not
			 * @param jsonMetadata metadata json object
			 * @return true if type is WelcomeMessage or Proactive
			 */
			private boolean isWelcomeMessageMetadata(JSONObject jsonMetadata) {
				if (jsonMetadata == null)
					return false;
				String metadataType = jsonMetadata.optString(MessagingConst.KEY_TYPE);
				if (MessagingConst.VALUE_WELCOME_MESSAGE.equalsIgnoreCase(metadataType) ||
					MessagingConst.VALUE_PROACTIVE.equalsIgnoreCase(metadataType)) {
					return true;
				}
				return false;
			}

			/**
			 * Add welcome message to database
			 */
			private MessagingChatMessage addWelcomeMessageToDB(
					ArrayList<SQLiteCommand> commands,
					ArrayList<FullMessageRow> messages,
					MessagingChatMessage.MessageState messageState,
					MessagingChatMessage.MessageType messageType,
					ContentEventNotification notification,
					BasePublishMessage publishMessage,
					ContentType contentType,
					String metadataType) {

				StringBuilder whereBuilder;
				MessagingChatMessage wcMessage;
				int serverSequence = WELCOME_MSG_SEQUENCE_NUMBER;
				if (MessagingConst.VALUE_PROACTIVE.equalsIgnoreCase(metadataType)) {
					serverSequence = OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER;
				}

				// Timestamp should be 2 millis before first message
				Dialog activeDialog = mController.amsDialogs.getActiveDialog();
				long timeStamp = notification.serverTimestamp + clockDiff - 2;
				String eventId = notification.dialogId + WELCOME_MESSAGE_EVENT_ID_POSTFIX;

				LPLog.INSTANCE.d(TAG, "addWelcomeMessageToDB: " + LPLog.INSTANCE.mask(publishMessage.getMessageText()) + " made-up eventId: " + eventId);

				whereBuilder = new StringBuilder();
				whereBuilder.append("(").append(MessagesTable.KEY_DIALOG_ID).append(" = ? AND ")
						.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" = ? )");


				wcMessage = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());
				wcMessage.setServerSequence(serverSequence);
				wcMessage.setTimeStamp(getTimestampForWelcomeMessage(notification.dialogId, timeStamp));

				ContentValues updateValues = getContentValuesForMessageUpdate(wcMessage);
				ContentValues insertValues = wrapValuesForInsertion(updateValues, wcMessage);

				boolean shouldUpdateWelcomeMessage = false;
				if (activeDialog != null && TextUtils.equals(activeDialog.getDialogId(), wcMessage.getDialogId())) {
					whereBuilder.append(" OR ").append(MessagesTable.KEY_EVENT_ID)
							.append(" = ").append("\"").append(AmsDialogs.KEY_WELCOME_DIALOG_ID + WELCOME_MESSAGE_EVENT_ID_POSTFIX).append("\"");
					updateValues.put(MessagesTable.KEY_EVENT_ID, eventId);
					shouldUpdateWelcomeMessage = true;
				}

				InsertOrUpdateSQLCommand insertOrUpdateSQLCommand = new InsertOrUpdateSQLCommand(
						insertValues,
						updateValues,
						whereBuilder.toString(),
						new String[]{dialogId, String.valueOf(serverSequence)}
				);
				commands.add(insertOrUpdateSQLCommand);

				FullMessageRow welcomeMessageRow = new FullMessageRow(wcMessage, null, null);
				if (shouldUpdateWelcomeMessage) {
					if (containsMessagesAfter(wcMessage.getTimeStamp())) {
						getMessagesListener().updateWelcomeMessageForDialogId(activeDialog.getDialogId(), welcomeMessageRow);
					} else {
						getMessagesListener().updateTempWelcomeMessage(welcomeMessageRow);
					}
				}
				messages.add(welcomeMessageRow);

				return wcMessage;
			}

			/**
			 * Parse metadata of first message to check whether it has attached metadata of welcome message
			 */
			private void processMetadata(@NonNull ContentEventNotification notification, @NonNull ArrayList<SQLiteCommand> commands, @NonNull ArrayList<FullMessageRow> messageRows) {
				if (!isWelcomeMessageExists(notification.dialogId)) {
					for (int i = 0; i < notification.metadata.length(); i++) {
						try {
							JSONObject metadata = notification.metadata.getJSONObject(i);
							if (isWelcomeMessageMetadata(metadata)) {
								String metadataType = metadata.optString(MessagingConst.KEY_TYPE);
								JSONObject event = metadata.optJSONObject(MessagingConst.KEY_EVENT);
								if (event != null) {
									BasePublishMessage publishMessage = null;
									String eventType = event.optString(MessagingConst.KEY_TYPE);
									LPLog.INSTANCE.d(TAG, "eventType: " + eventType);
									ContentType contentType = ContentType.text_plain;
									MessagingChatMessage.MessageState msgState = MessagingChatMessage.MessageState.RECEIVED;
									MessagingChatMessage.MessageType msgType = MessagingChatMessage.MessageType.BRAND;

									if (MessagingConst.VALUE_RICH_CONTENT_EVENT.equalsIgnoreCase(eventType)) {
										contentType = ContentType.text_structured_content;
										msgType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;
										publishMessage = new StructuredContentPublishMessage(event.optString(MessagingConst.KEY_RICH_CONTENT));
									} else if (MessagingConst.VALUE_CONTENT_EVENT.equalsIgnoreCase(eventType)) {
										publishMessage = new TextPublishMessage(event.optString(MessagingConst.KEY_MESSAGE));
									}

									if (publishMessage != null) {
										addWelcomeMessageToDB(commands, messageRows, msgState, msgType,
												notification, publishMessage, contentType, metadataType);
									}
								}
								break;
							}
						} catch (JSONException e) {
							LPLog.INSTANCE.e(TAG, ERR_0000016B, "Failed to parse metadata of welcome message", e);
						}
					}
				} else {
					commands.add(updateWelcomeMessageTimeStamp(notification.dialogId, notification.serverTimestamp + clockDiff - 2));
				}
			}

			@NonNull
			private MessagingChatMessage createMessageInDB(
					ArrayList<SQLiteCommand> commands,
					ArrayList<FullMessageRow> messages,
					MessagingChatMessage.MessageState messageState,
					MessagingChatMessage.MessageType messageType,
					ContentEventNotification notification,
					BasePublishMessage publishMessage,
					ContentType contentType) {
				StringBuilder whereBuilder;
				MessagingChatMessage message;
				String eventId = notification.eventId;
				if (TextUtils.isEmpty(eventId)) {
					eventId = UniqueID.createUniqueMessageEventId();

					LPLog.INSTANCE.d(TAG, "no event id for message: " + LPLog.INSTANCE.mask(publishMessage.getMessageText()) + " creating event id: " + eventId);

					//in case there is no event id we need to check if such message exists
					//in the db by the conversation id and sequence. is exists - update it,
					//if not, insert this message.
					whereBuilder = new StringBuilder();
					whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" = ? AND ")
							.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" = ?");

					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());
					ContentValues updateValues = getContentValuesForMessageUpdate(message);
					ContentValues insertValues = wrapValuesForInsertion(updateValues, message);

					InsertOrUpdateSQLCommand insertOrUpdateSQLCommand = new InsertOrUpdateSQLCommand(
							insertValues,
							updateValues,
							whereBuilder.toString(),
							new String[]{dialogId, String.valueOf(notification.sequence)}
					);

					// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
					AddSqliteCommandListenerForFile(publishMessage, insertOrUpdateSQLCommand, message.getEventId());

					// Add a treatment for form invitation and submission
					AddCommandForForm(publishMessage, message, brandId, commands);

					commands.add(insertOrUpdateSQLCommand);

				} else {

					//we got message with eventId - we insert, if it's exists it will replace.

					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());
					//Check if the message exist
					Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null);
					if (cursor != null && cursor.getCount() > 0) {
						ContentValues messageValues = getContentValuesForMessageUpdate(
								message,
								cursor,
								mOfflineMessagesRepository.isMessagePending(brandId, message.getEventId())
						);

						//If needed update the message in DB
						if (messageValues.size() > 0) {
							LPLog.INSTANCE.d(TAG, "Updating message: This message need to be update with message: " + LPLog.INSTANCE.mask(message));
							UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
									new String[]{String.valueOf(message.getEventId())});

							// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
							AddSqliteCommandListenerForFile(publishMessage, updateSQLCommand, message.getEventId());

							// Add a treatment for form invitation and submission
							AddCommandForForm(publishMessage, message, brandId, commands);

							commands.add(updateSQLCommand);

						} else {
							LPLog.INSTANCE.d(TAG, "Updating message: Skip updating this message since its already exist" + LPLog.INSTANCE.mask(message));
							// Update If any stored welcome message with conversation dialogId
							if (message.getServerSequence() == 0) {
								updateTempDialogId(dialogId, OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER, 0);
							}
						}
					} else {
						//we got message with eventId - we insert, if it's exists it will replace.
						InsertSQLCommand insertSQLCommand = new InsertSQLCommand(getContentValuesForMessage(message));

						// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
						AddSqliteCommandListenerForFile(publishMessage, insertSQLCommand, message.getEventId());

						// Add a treatment for form invitation and submission
						AddCommandForForm(publishMessage, message, targetId, commands);

						commands.add(insertSQLCommand);
					}
				}
				MessagingUserProfile profile = getUserProfile(message.getOriginatorId());
				String profileAvatar = profile != null ? profile.getAvatarUrl() : null;
				messages.add(new FullMessageRow(message, profileAvatar, null));
				return message;
			}

			@NonNull
			private MessagingChatMessage createMessage(String textMessage, MessagingChatMessage.MessageState messageState,
													   MessagingChatMessage.MessageType messageType, ContentEventNotification notification,
													   String eventId, String contentType) {
				MessagingChatMessage message = new MessagingChatMessage(
						notification.originatorId,
						textMessage,
						notification.serverTimestamp + clockDiff,
						dialogId,
						eventId,
						updateMessageTypeIfNeeded(messageType, textMessage),
						messageState,
						notification.sequence,
						contentType,
						EncryptionVersion.NONE);

                LPLog.INSTANCE.d(TAG, "creating message '" + LPLog.INSTANCE.mask(textMessage) + "', seq: " + notification.sequence + ", at time: " + notification.serverTimestamp
						+ ", dialogId: " + dialogId + ", clock diff: " + clockDiff + " = " + (notification.serverTimestamp + clockDiff));
                return message;
            }

			private void AddCommandForForm(BasePublishMessage publishMessage, MessagingChatMessage msg, String brandId, ArrayList<SQLiteCommand> commands) {
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_INVITATION) {
					FormPublishMessage formPublishMessage = (FormPublishMessage) publishMessage;
					LPLog.INSTANCE.d(TAG, "onResult: new form obj to DB getMessage " + LPLog.INSTANCE.mask(formPublishMessage.getMessage()));
					mController.amsMessages.mFormsManager.addForm(formPublishMessage.getInvitationId(),
							new Form(
									msg.getDialogId(),
									conversationId,
									formPublishMessage.getInvitationId(),
									formPublishMessage.getFormId(),
									formPublishMessage.getFormTitle(),
									mController.mAccountsController.getTokenizerUrl(brandId),
									brandId,
									msg.getServerSequence(),
									msg.getEventId()
							));
				}
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_SUBMISSION) {
					FormSubmissionPublishMessage formPublishMessage = (FormSubmissionPublishMessage) publishMessage;
					Form form = mController.amsMessages.mFormsManager.getForm(formPublishMessage.getInvitationId());
					if (form != null) {
						mController.amsMessages.mFormsManager.updateForm(formPublishMessage.getInvitationId(), formPublishMessage.getSubmissionId());
						LPLog.INSTANCE.d(TAG, "Updating message: This message need to be update with message: ");
						ContentValues cv = new ContentValues();
						cv.put(MessagesTable.KEY_STATUS, getReceivedMessageState(DeliveryStatus.SUBMITTED).ordinal());
						UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(cv, MessagesTable.KEY_EVENT_ID + "=?",
								new String[]{String.valueOf(form.getEventId())});
						commands.add(updateSQLCommand);
					}


				}
			}

			/*
			 * Modify type based on masking and url.
			 * */
			private MessagingChatMessage.MessageType updateMessageTypeIfNeeded(MessagingChatMessage.MessageType messageType, String message) {
				// check if we have a link than we need to set the message type to be with URL
				if ((messageType == MessagingChatMessage.MessageType.CONSUMER) ||
						(messageType == MessagingChatMessage.MessageType.CONSUMER_MASKED) ||
						(messageType == MessagingChatMessage.MessageType.AGENT)) {

					messageType = checkIfMessageContainsURLandChangeType(messageType, message);
				}
				return messageType;
			}

			/**
			 * Add a mMessagesListener to the given sqLiteCommand to add the image file after the command is complete
			 *
			 * @param publishMessage
			 * @param sqLiteCommand
			 */
			private void AddSqliteCommandListenerForFile(
					BasePublishMessage publishMessage,
					SQLiteCommand sqLiteCommand,
					final String eventId
			) {

				// Validate the given publishMessage is of type file
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FILE) {

					final BasePublishMessage finalPublishMessage = publishMessage;
					// Listener
					sqLiteCommand.setListener(rowId -> {
						String event = sqLiteCommand.getContentValues().getAsString(MessagesTable.KEY_EVENT_ID);
						String message = sqLiteCommand.getContentValues().getAsString(MessagesTable.KEY_TEXT);
						int version = sqLiteCommand.getContentValues().getAsInteger(MessagesTable.KEY_ENCRYPTION_VERSION);
						String decrypted = DBEncryptionHelper.decrypt(EncryptionVersion.fromInt(version), message);
						String tag = "onInsertComplete";

						if (rowId == DBUtilities.ROW_UPDATED) {
							LPLog.INSTANCE.d(TAG, tag + ": message was updated on DB (and not inserted). No need to add the file to DB");
						} else {
							addFileFromPublishMessageToDB(rowId, eventId, tag, (FilePublishMessage) finalPublishMessage, targetId);
						}
					});
				}
			}

			private void encryptSqlCommandData(List<SQLiteCommand> commands) {
				for (SQLiteCommand command : commands) {
					DBEncryptionServiceUtilsKt.encryptTransactionData(command);
				}
			}
		});
	}

	/**
	 * Update messages of a conversation once it is re-fetched from INCA server. This will make sure we have
	 * PII data in it's masked format.
	 * @param responseMessages conversation response
	 * @param dialogId re-fetched dialog
	 * @return DataBaseCommand
	 */
	public DataBaseCommand<Void> updateMultipleMessages(final ArrayList<ContentEventNotification> responseMessages, final String dialogId) {

		// end query()
		return new DataBaseCommand<>(() -> {
			if (responseMessages != null) {
				LPLog.INSTANCE.d(TAG, "updateMultipleMessages: Start updating messages for dialogId = " + dialogId);

				DataBaseExecutor.execute(() -> {
					for (ContentEventNotification notification : responseMessages) {
						if (notification.event == null || notification.event.type == Event.Types.AcceptStatusEvent) {
							LPLog.INSTANCE.i(TAG, "updateMultipleMessages: Ignore messages with empty or are of type AcceptStatusEvent");
							continue;
						}

						BasePublishMessage publishMessage = null;

						if (notification.event.message != null) {
							publishMessage = notification.event.message;
						}

						// If the publish message is a text and the message itself is empty we ignore this
						if (publishMessage == null ||
								(publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {
							LPLog.INSTANCE.e(TAG, ERR_0000007C, "updateMultipleMessages: Text message received in query messages is empty :| shouldn't happen! " +
									"dialogId = " + dialogId + " , sequence = " + notification.sequence);
							// it's some protocol error, lets ignore it
							continue;
						}

						String sequence = String.valueOf(notification.sequence);
						ContentValues contentValues = new ContentValues();
						final EncryptionVersion messageEncryptionVersion = DBEncryptionService.Companion.getAppEncryptionVersion();
						contentValues.put(MessagesTable.KEY_ENCRYPTION_VERSION, messageEncryptionVersion.ordinal());
						String encryptedMessage = DBEncryptionHelper.encrypt(messageEncryptionVersion, publishMessage.getMessageText());
						// Put message text to be updated
						contentValues.put(MessagesTable.KEY_TEXT, encryptedMessage);

						String whereBuilder = MessagesTable.KEY_DIALOG_ID + " = ? AND " +
								MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
						LPLog.INSTANCE.d(TAG, "updateMultipleMessages: Updating message with sequence: " + sequence);
						getDB().update(contentValues, whereBuilder, new String[]{dialogId, sequence});
					} // end for loop
				});
			}
			return null;
		});
	}

	@NonNull
	@VisibleForTesting
	public synchronized Long getLastMessageTimestampDB() {
		long lastSavedMessage = lastMessageTimeStamp.get();
		if (lastSavedMessage != Long.MAX_VALUE) {
			return lastSavedMessage;
		}
		try (Cursor cursor = requestTimeStampFromDB()) {
			long timestamp;
			if (cursor.moveToFirst()) {
				timestamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
			} else {
				timestamp = Long.MAX_VALUE;
			}
			lastMessageTimeStamp.compareAndSet(Long.MAX_VALUE, timestamp);
			return timestamp;
		} catch (Exception e) {
			LPLog.INSTANCE.i(TAG, FlowTags.QUICK_REPLIES, "Receive last message produces error", e);
			return Long.MAX_VALUE;
		}
	}

	/**
	 * Method used to filter messages from controller stored in database and filtered by sequence
	 * from notification event.
	 *
	 * @param dialogId         required database
	 * @param possibleSequence sequence received from notification
	 * @return list of filtered sequences that were sent by controller and stored in database
	 */
	private synchronized List<Integer> getDBMessagesFromController(String dialogId, int[] possibleSequence) {
		String query = "SELECT "
				+ MessagesTable.KEY_SERVER_SEQUENCE
				+ " FROM "
				+ MessagesTable.MESSAGES_TABLE
				+ " WHERE " + MessagesTable.KEY_DIALOG_ID + " = " + "\"" + dialogId + "\""
				+ " AND " + MessagesTable.KEY_MESSAGE_TYPE + " = " + MessagingChatMessage.MessageType.CONTROLLER_SYSTEM.ordinal()
				+ " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " IN "
				+ "(" + intArrayToString(possibleSequence) + ")";
		try (Cursor cursor = getDB().rawQuery(query)) {
			if (cursor.moveToFirst() && cursor.getCount() > 0) {
				List<Integer> filteredSequence = new ArrayList<>();
				do {
					filteredSequence.add(cursor.getInt(0));
				} while (cursor.moveToNext());
				return filteredSequence;
			} else {
				return Collections.emptyList();
			}
		} catch (Exception exception) {
			LPLog.INSTANCE.w(TAG,"getDBMessagesFromController produces error", exception);
			return Collections.emptyList();
		}
	}

	private boolean containsMessagesAfter(long timestamp) {
		boolean result;
		String query = "SELECT * FROM " + MessagesTable.MESSAGES_TABLE
				+ " WHERE " + MessagesTable.KEY_SERVER_SEQUENCE  + " != " + OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER
				+ " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " != " + WELCOME_MSG_SEQUENCE_NUMBER
				+ " AND " + MessagesTable.KEY_TIMESTAMP + " >= ?"
				+ " LIMIT 1";
		try (Cursor cursor = getDB().rawQuery(query, timestamp)) {
			result = cursor.moveToFirst() && cursor.getCount() > 0;
		} catch (Exception exception) {
			result = false;
		}
		return result;
	}

	/**
	 * Method used to convert integer array to string which separated by coma.
	 * @param param array to convert
	 * @return string representation of array where each element separated by coma
	 */
	private String intArrayToString(int[] param) {
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < param.length; i++) {
			if (i > 0) {
				builder.append(", ");
			}
			builder.append(param[i]);
		}
		return builder.toString();
	}

	private boolean containsMessagesInDB() {
		if (!containsMessagesInDatabse) {
			containsMessagesInDatabse = closedConversationMessagesExists(DialogType.MAIN);
		}
		return containsMessagesInDatabse;
	}

	private boolean containsPCSMessagesInDB() {
		if (!containsPCSMessagesInDatabse) {
			containsPCSMessagesInDatabse = closedConversationMessagesExists(DialogType.POST_SURVEY);
		}
		return containsPCSMessagesInDatabse;
	}

	private Long getOldestMessageFromReceived(List<FullMessageRow> messages) {
		long min = Long.MAX_VALUE;
		for (FullMessageRow row : messages) {
			long timestamp = row.getMessagingChatMessage().getTimeStamp();
			min = Math.min(min, timestamp);
		}
		return min;
	}

	private Boolean closedConversationMessagesExists(DialogType dialogType) {
		String query = "SELECT EXISTS(" + "SELECT " + MessagesTable.MESSAGES_TABLE + "." + MessagesTable.KEY_TIMESTAMP
				+ " FROM " + MessagesTable.MESSAGES_TABLE
				+ " INNER JOIN " + DialogsTable.TABLE_NAME
				+ " ON " + DialogsTable.TABLE_NAME + "." + DialogsTable.Key.DIALOG_ID
				+ " = " + MessagesTable.MESSAGES_TABLE  + "." + MessagesTable.KEY_DIALOG_ID
				+ " WHERE " + DialogsTable.TABLE_NAME + "." + DialogsTable.Key.STATE
				+ " != " + DialogState.OPEN.ordinal()
				+ " AND " + DialogsTable.TABLE_NAME + "." + DialogsTable.Key.DIALOG_TYPE
				+ " = ? "
				+ " ORDER BY " + MessagesTable.KEY_TIMESTAMP
				+ " DESC LIMIT 1" + ")";
		boolean result;
		try (Cursor cursor = getDB().rawQuery(query, dialogType.name())) {
			result = cursor.moveToFirst() && cursor.getInt(0) == 1;
		} catch (Exception exception) {
			result = false;
		}
		return result;
	}

	private void addResolveMessageToClosedConversation(Dialog dialog) {
		Messaging controller = MessagingFactory.getInstance().getController();
		DialogUtils dialogUtils = new DialogUtils(controller);
		if (dialog != null && dialog.getState() == DialogState.CLOSE && isConversationClosed(dialog.getConversationId())) {
			dialogUtils.addClosedDialogDivider(
					dialog.getBrandId(),
					dialog,
					dialog.getAssignedAgentId(),
					dialog.getCloseReason(),
					true,
					null
			);
		}
	}

	private boolean isConversationClosed(String id) {
		try (Cursor cursor = getConversationStateById(id)) {
			if (cursor.moveToFirst()) {
				Conversation conversation = new Conversation(cursor);
				return !conversation.isConversationOpen();
			} else  {
				return false;
			}
		} catch (Exception ex) {
			return false;
		}
	}

	private Cursor requestTimeStampFromDB() {
		return getDB().rawQuery(
				"SELECT "
						+ MessagesTable.KEY_TIMESTAMP
						+ " FROM "
						+ MessagesTable.MESSAGES_TABLE
						+ " ORDER BY "
						+ MessagesTable.KEY_TIMESTAMP
						+ " DESC LIMIT 1"
		);
	}

	/**
	 * Method used to show messages received from UMS (open conversation)
	 * or first closed conversations with database saving and encryption.
	 * @param brandId required bean
	 * @param messages list of messages from UMS/INCA
	 * @param statuses messages statuses holder applied to received messages
	 */
	private void showInitialMessages(String brandId, List<FullMessageRow> messages, List<AmsMessages.MessagesStatusHolder> statuses) {
		Collections.sort(messages, FullMessageRow::newerThan);
		applyMessagesStatus(messages, statuses);
		getMessagesListener().onInitialMessagesReceived(maskMessagesIfNeeded(brandId, messages));
	}

	/**
	 * Method used to mask messages sent from unauthenticated user.
	 * Note: message could be masked only if provided brand
	 * uses unauthenticated flow, client only masking is enabled and message was sent from
	 * consumer (type should be CONSUMER, CONSUMER_URL, CONSUMER_DOC or CONSUMER_IMAGE).
	 * @param brandId required brand id
	 * @param rows messages required to mask.
	 * @return list of masked messages.
	 */
	private List<FullMessageRow> maskMessagesIfNeeded(String brandId, List<FullMessageRow> rows) {
		boolean isUnAuth = MessagingFactory.getInstance()
				.getController()
				.mAccountsController
				.isInUnAuthMode(brandId);
		if (isUnAuth && Configuration.getBoolean(R.bool.enable_client_only_masking)) {
			List<FullMessageRow> maskedRows = new ArrayList<>();
			for (FullMessageRow row : rows) {
				maskedRows.add(maskMessage(brandId, row));
			}
			return maskedRows;
		} else {
			return rows;
		}
	}

	/**
	 * Method used to mask a message by using message validator provided for particular brand.
	 * @param brandId brand id to receive an appropriate message validator.
	 * @param row required to mask. Note: type of message should be CONSUMER,
	 *              CONSUMER_URL, CONSUMER_DOC or CONSUMER_IMAGE
	 * @return message row with masked text.
	 */
	private FullMessageRow maskMessage(String brandId, FullMessageRow row) {
		FullMessageRow maskedRow;
		if (isConsumerUnmasked(row.getMessagingChatMessage().getMessageType())) {
			maskedRow = new FullMessageRow(row);
			MessagingChatMessage chatMessage = maskedRow.getMessagingChatMessage();
			MaskedMessage message = mController.getMaskedMessage(brandId, chatMessage.getMessage());
			chatMessage.setMessage(message.getDbMessage());
		} else {
			maskedRow = row;
		}
		return maskedRow;
	}

	/**
	 * Method used remove all temporal welcome message from database.
	 * @return database command to remove temporal welcome messages.
	 */
	public DataBaseCommand<Boolean> removeTemporalWelcomeMessage() {
		return new DataBaseCommand<Boolean>(() -> {
			String whereClause = MessagesTable.KEY_EVENT_ID + " = ? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
			String[] whereArgs = {
					AmsDialogs.KEY_WELCOME_DIALOG_ID + WELCOME_MESSAGE_EVENT_ID_POSTFIX,
					String.valueOf(AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER)
			};
			try {
				return getDB().removeAll(whereClause, whereArgs) > 0;
			} catch (Exception exception) {
				return false;
			}
		});
	}

	private Cursor getConversationStateById(String id) {
		return getDB().rawQuery(
				"SELECT * FROM "
						+ ConversationsTable.TABLE_NAME
						+ " WHERE "
						+ ConversationsTable.Key.CONVERSATION_ID
						+ " =? "
						+ "LIMIT 1",
				id
		);
	}


	public void onAgentReceived(MessagingUserProfile userProfile) {
		getMessagesListener().onAgentReceived(userProfile);
	}

	/**
	 * Get the QuickReplies JSON from the given notification. This only happen for message from the agent
	 * @param brandId
	 * @param notification
	 * @param messageType
	 * @param dialogId
	 */
	private synchronized void getQuickRepliesFromEvent(String brandId, ContentEventNotification notification, MessagingChatMessage.MessageType messageType, String dialogId) {
		// If conversation is closed don't add QuickReplies
		boolean isDialogClosed = MessagingFactory.getInstance().getController().isDialogClosed(dialogId);
		if(MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
			LPLog.INSTANCE.d(TAG, "getQuickRepliesFromEvent: conversation is closed, not adding QuickReplies message");
			return;
		}

		// Save the message only if from agent
		if ((messageType == MessagingChatMessage.MessageType.AGENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_MARKDOWN_HYPERLINK) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_URL)) {


			QuickRepliesMessageHolder currentQuickRepliesMessageHolder = QuickRepliesMessageHolder.fromContentEventNotification(brandId, notification);

			LPLog.INSTANCE.d(TAG, "getQuickRepliesFromEvent: Message is from agent, try to get QuickReplies string from event");

			// Store the current QR message only if newer
			if (mQuickRepliesMessageHolder == null ||
					((currentQuickRepliesMessageHolder != null) && currentQuickRepliesMessageHolder.newerThan(mQuickRepliesMessageHolder))) {

				LPLog.INSTANCE.d(TAG, FlowTags.QUICK_REPLIES, "QuickReplies message is newer than the current one. New one: " + ((currentQuickRepliesMessageHolder != null) ? currentQuickRepliesMessageHolder.toString() : "null"));
				mQuickRepliesMessageHolder = currentQuickRepliesMessageHolder;
			}
		}
	}

	/**
	 * Get the {@link QuickRepliesMessageHolder} stored in memory. If not exist in memory load from SharedPreferences
	 * @param brandId
	 * @return
	 */
	@Override
	public synchronized QuickRepliesMessageHolder getQuickRepliesMessageHolder(String brandId) {

		// If does not exist in memory load from SharedPreferences
		if (mQuickRepliesMessageHolder == null) {
			mQuickRepliesMessageHolder = QuickRepliesMessageHolder.loadFromSharedPreferences(brandId);
		}

		return mQuickRepliesMessageHolder;
	}

	@Override
	public void resetQuickRepliesMessageHolder() {
		LPLog.INSTANCE.d(TAG, "resetQuickRepliesMessageHolder: resetting QuickRepliesMessageHolder");
		mQuickRepliesMessageHolder = null;
	}

	/**
	 * Remove all CoBrowse message added in database messaging table.
	 */
	@Override
	public void removeCoBrowseMessage() {
		DataBaseExecutor.execute(() -> {
			LPLog.INSTANCE.d(TAG, "removeCoBrowseMessage: Remove cobrowse messages from database");
			getDB().removeAll(MessagesTable.KEY_MESSAGE_TYPE + "=?", new String[]{String.valueOf(MessagingChatMessage.MessageType.COBROWSE.ordinal())});
		});
	}

	//Check if the message state and server sequence need to be updated in the existing message
	@NonNull
	private ContentValues getContentValuesForMessageUpdate(MessagingChatMessage message, Cursor cursor, boolean shouldUpdateTimestamp) {
		MessagingChatMessage existingMessage = getSingleMessageFromCursor(cursor);
		ContentValues messageValues = new ContentValues();

		//Check if we need to update the existing message
		if (existingMessage.getTimeStamp() < message.getTimeStamp() && shouldUpdateTimestamp){
			messageValues.put(MessagesTable.KEY_TIMESTAMP, message.getTimeStamp());
			String logMessage = "Start updating message timestamp. Event id: " + message.getEventId() + "\n"
					+ "Update message timestamp, old val: " + existingMessage.getTimeStamp()
					+ " , new val: " + message.getTimeStamp();
			LPLog.INSTANCE.d(TAG, logMessage);
		} else {
			LPLog.INSTANCE.d(TAG, "Skip update message timestamp, old val: " + existingMessage.getTimeStamp() + " , new val: " + message.getTimeStamp());
		}
		if ((message.getMessageState().ordinal() != existingMessage.getMessageState().ordinal()) &&
				(MessagingChatMessage.MessageState.validChange(existingMessage.getMessageState(), message.getMessageState()))) {
			messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		} else {
			LPLog.INSTANCE.d(TAG, "Skip update message state, old val: " + message.getMessageState() + " , new val: " + existingMessage.getMessageState());
		}
		if (message.getServerSequence() != existingMessage.getServerSequence()) {
			messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		} else {
			LPLog.INSTANCE.d(TAG, "Skip update message server sequence, old val: " + message.getServerSequence() + " , new val: " + existingMessage.getServerSequence());
		}
		if (message.getServerSequence() == WELCOME_MSG_SEQUENCE_NUMBER && !message.getMessage().equals(existingMessage.getMessage())) {
			messageValues.put(MessagesTable.KEY_TEXT, message.getMessage());
		}
		return messageValues;
	}

    /**
     * MUST BE CALLED FROM DB Thread!
     *  @param messageRowId
     * @param tag
	 * @param finalPublishMessage
	 * @param targetId
	 */
	private void addFileFromPublishMessageToDB(
			long messageRowId,
			String eventId,
			String tag,
			FilePublishMessage finalPublishMessage,
			String targetId
	) {

		LPLog.INSTANCE.d(TAG, tag + ": MessagingChatMessage was added. row id: " + messageRowId + ". Adding fileMessage to db.");

        // We first save the base64 as bitmap to disk
        String previewImagePath = ImageUtils.saveBase64ToDisk(Infra.instance.getApplicationContext(), finalPublishMessage.getPreview(), targetId);

		LPLog.INSTANCE.d(TAG, tag + ": preview image saved to location: " + previewImagePath);

		if (previewImagePath != null) {
			FileMessage fileMessage = new FileMessage(previewImagePath, finalPublishMessage.getFileType(), null, finalPublishMessage.getRelativePath(), messageRowId);
			long fileRowId = MessagingFactory.getInstance().getController().amsFiles.addFile(messageRowId, fileMessage).executeSynchronously();
			LPLog.INSTANCE.d(TAG, tag + ": fileMessage was added to db. fileRowId = " + fileRowId);
			if (mMessagesListener != null) {
				mMessagesListener.onUpdateFileMessage(eventId, messageRowId, new FileMessage(fileMessage, fileRowId));
			}
		}
	}

	@NonNull
	private MessagingChatMessage.MessageState getReceivedMessageState(DeliveryStatus status) {
		MessagingChatMessage.MessageState messageState = null;

		switch (status) {
			case ACCEPT:
				messageState = MessagingChatMessage.MessageState.RECEIVED;
				break;
			case READ:
			case ACTION:
				messageState = MessagingChatMessage.MessageState.READ;
				break;
			case VIEWED:
				messageState = MessagingChatMessage.MessageState.VIEWED;
				break;
			case SUBMITTED:
				messageState = MessagingChatMessage.MessageState.SUBMITTED;
				break;
			case ERROR:
			case ABORTED:
				messageState = MessagingChatMessage.MessageState.ERROR;
				break;
		}
		return messageState;
	}

	/**
	 * Create where statement to update list of sequences (messages) state
	 *
	 * @param dialogId
	 * @param sequenceList
	 * @param messageState
	 * @param length
	 * @param contentValues
	 * @param whereBuilder
	 * @param whereArgs
	 */
	private MessagesStatusHolder createStatementForUpdateMessagesState(
			String dialogId,
			int[] sequenceList,
			MessagingChatMessage.MessageState messageState,
			int length,
			ContentValues contentValues,
			StringBuilder whereBuilder,
			String[] whereArgs,
			List<Integer> filteredSequence
	) {
		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		whereArgs[0] = String.valueOf(dialogId);
		whereArgs[1] = String.valueOf(messageState.ordinal());
		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
				append(MessagesTable.KEY_STATUS).append(" <?  AND ").
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" in (");

		// Add all sequences to the where args and to the where clause builder
		TreeSet<Integer> sequenceSet = new TreeSet<>();
		for (int i = 0; i < length; i++) {
			whereArgs[i + 2] = String.valueOf(sequenceList[i]);
			whereBuilder.append("?");
			int sequence = sequenceList[i];
			if (filteredSequence.contains(sequence)) {
				sequenceSet.add(sequenceList[i]);
			}

			// If last item don't add comma
			if (i != length - 1) {
				whereBuilder.append(",");
			}
		}

		whereBuilder.append(")");
		return new MessagesStatusHolder(dialogId, sequenceSet, messageState, -1);
	}

	private UpdateSQLCommand createStatementForUpdateMaxMessagesState(String dialogId,MessagingChatMessage.MessageState messageState, int maxSequence) {
		ContentValues contentValues = new ContentValues();
		StringBuilder whereBuilder = new StringBuilder();

		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		String[] whereArgs = new String[]{
				String.valueOf(dialogId),
				String.valueOf(messageState.ordinal()),
				String.valueOf(maxSequence),
				String.valueOf(PENDING_MSG_SEQUENCE_NUMBER)};

		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ")
				.append("(")
				.append(MessagesTable.KEY_STATUS).append(" < ? ")
				.append(" OR ")
				.append(MessagesTable.KEY_STATUS).append(" == ").append(MessagingChatMessage.MessageState.OFFLINE.ordinal())
				.append(")")
				.append(" AND ")
				.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" <=? AND " )
				.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" >? ");

		return new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs);
	}

	public void updateAllMessagesStateByDialogId(final String dialogId, final MessagingChatMessage.MessageState messageState) {
		DataBaseExecutor.execute(() -> {
			ContentValues contentValues = new ContentValues();
			// The value to be changed
			contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

			int result = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + " = ? ", new String[]{dialogId});
			LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s", result, messageState));

			//update list listener
			updateAllMessagesForDialog(dialogId);
		});
	}


	// Change the messages state of all the closed conversation from pending to close
	private void markPendingMessagesAsFailedOnCloseConv(final String brandId) {

		DataBaseExecutor.execute(() -> {

			// Get images' rowId currently in upload progress.
			// This is required since we don't want to mark these messages as failed
			String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

			String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.CLOSE.ordinal()));
			String where = getPendingMessagesQuery(rowIds, MessagingChatMessage.MessageState.PENDING, MessagingChatMessage.MessageState.QUEUED);

			//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
			Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
			if (cursor != null) {
				try {
					if (cursor.getCount() == 0) {
						//No pending failed messages to update
						return;
					}
					if (cursor.moveToFirst()) {
						do {
							//there are failed messages that need update.
							MessagingChatMessage message = getSingleMessageFromCursor(cursor);
							message.setMessageState(MessagingChatMessage.MessageState.ERROR);
							long rowId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
							updateMessageByRowId(rowId, message);
						} while (cursor.moveToNext());
					}
				} finally {
					cursor.close();
				}

				// Update the DB only if we have result for the query
				ContentValues contentValues = new ContentValues();
				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.ERROR.ordinal());

				//m.status = 3 and c.brandId = "qa81253845" and m.convID = c.conversationId
				int result = getDB().update(contentValues, where, whereArgs);
				LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s", result, MessagingChatMessage.MessageState.ERROR));
			}
		});
	}

	public void resendAllPendingMessages(final String brandId) {

		final long resendMessageTimeout = Configuration.getInteger(R.integer.sendingMessageTimeoutInMinutes);

		// Change the messages state of all the closed conversation from pending to close
		markPendingMessagesAsFailedOnCloseConv(brandId);

		DataBaseExecutor.execute(() -> {

			// Get images' rowId currently in upload progress.
			// This is required since we don't want to mark these messages as failed
			String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

			String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.OPEN.ordinal()));
			String where = getPendingMessagesQuery(rowIds, MessagingChatMessage.MessageState.PENDING, MessagingChatMessage.MessageState.QUEUED);

			//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
			Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
			if (cursor != null) {
				try {
					if (cursor.getCount() == 0) {
						//No pending messages
						return;
					}
					if (cursor.moveToFirst()) {
						ArrayList<String> eventIdsToErrorUpdate = new ArrayList<>();
						Set<String> pendingMessages = mOfflineMessagesRepository.getPendingOfflineMessages(brandId);
						do {
							//Resend all pending messages that didn't pass the timeout
							MessagingChatMessage message = getSingleMessageFromCursor(cursor);
							String dialogId = message.getDialogId();
							if (resendMessageTimeout > 0 && System.currentTimeMillis() < message.getTimeStamp() + TimeUnit.MINUTES.toMillis(resendMessageTimeout)) {
								if (pendingMessages.contains(message.getEventId())) {
									continue;
								}
								LPLog.INSTANCE.d(TAG, "Resend message: " + LPLog.INSTANCE.mask(message));
								long rowId = Messaging.NO_FILE_ROW_ID;

								// Get rowId in order to pull the image fileMessage from the DB
								if (MessagingChatMessage.MessageType.isImage(message.getMessageType())) {
									FileMessage fileMessage = mController.amsFiles.getFileByMessageRowId(message.getLocalId());
									if (fileMessage != null) {
										rowId = fileMessage.getFileRowId();
									}
								}
								mController.resendMessage(message.getEventId(), dialogId, rowId, message.getMessageType(), false);
							} else {
								LPLog.INSTANCE.d(TAG, "Resend timeout - Set message to FAILED state,  resendMessageTimeout:" + resendMessageTimeout + ", message: " + LPLog.INSTANCE.mask(message));
								eventIdsToErrorUpdate.add(message.getEventId());
							}
						} while (cursor.moveToNext());

						if (!eventIdsToErrorUpdate.isEmpty()) {
							// Change message state to failed - resend timeout
							updateMessagesState(eventIdsToErrorUpdate, MessagingChatMessage.MessageState.ERROR);
						}
					}
				} finally {
					cursor.close();
				}
			}
		});
	}

	private String getPendingMessagesQuery(String rowIds, MessagingChatMessage.MessageState... states) {
		/*
		update Messages set status = error where exists for all the rows in this query:
        (select row_id from Messages m, Conversations c where c.state = (m.status = pending or m.status = queue) and c.brandId = "qa81253845" and m.convID = c.conversationId)
         */
		StringBuilder messageStateSubQuery = new StringBuilder();
		for (MessagingChatMessage.MessageState state: states) {
			if (messageStateSubQuery.length() != 0) {
				messageStateSubQuery.append(" or ");
			}
			messageStateSubQuery.append("m.")
					.append(MessagesTable.KEY_STATUS).append("=")
					.append(state.ordinal());
		}

		StringBuilder whereBuilder = new StringBuilder().append(MessagesTable.KEY_ID).append(" in (select m.").append(MessagesTable.KEY_ID).append(" from ")
				.append(MessagesTable.MESSAGES_TABLE).append(" m , ")
				.append(DialogsTable.TABLE_NAME).append(" c ")
				.append("where (").append(messageStateSubQuery)
				.append(") and c.").append(DialogsTable.Key.BRAND_ID).append("=?")
				.append(" and c.").append(DialogsTable.Key.STATE).append("=?")
				.append(" and m.").append(MessagesTable.KEY_DIALOG_ID).append("= c.").append(DialogsTable.Key.DIALOG_ID);

		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			LPLog.INSTANCE.d(TAG, "resendAllPendingMessages: There is upload images in progress, ignore these messages rowId: " + rowIds);
			whereBuilder.append(" and m.").append(MessagesTable.KEY_ID).append(" not in (?)");
		}

		whereBuilder.append(")");// Close the where clause
		String where = whereBuilder.toString();
		LPLog.INSTANCE.d(TAG, "getPendingMessagesQuery: where clause: " + where);
		return where;
	}

	private String[] getPendingMessagesQueryParams(String brandId, String rowIds, String state) {
		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			return new String[]{brandId, state, rowIds};
		} else { // where without image rowId
			return new String[]{brandId, state};
		}
	}

	/**
	 * Update the messageState to a message with the given eventId. The targetId and dialogId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventId
	 * @param messageState
	 */
	public void updateMessageState(final String eventId, final MessagingChatMessage.MessageState messageState) {
		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(() -> {
			// Update the status of the message
			ContentValues contentValues = new ContentValues();
			// The value to be changed
			contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());
			int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + "=?", new String[]{eventId});
			LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventId));
			//update list listener
			updateMessageByEventId(eventId);
		});
	}

	/**
	 * Update the messageState to array of messages based on given eventId. The targetId and conversationId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventIds
	 * @param messageState
	 */
	public void updateMessagesState(final ArrayList<String> eventIds, final MessagingChatMessage.MessageState messageState) {

		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(() -> {

			if (eventIds != null && !eventIds.isEmpty()) {
				// Update the status of the message
				ContentValues contentValues = new ContentValues();

				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

				StringBuilder inOperator = new StringBuilder(" IN (?");
				for (int i = 1; i < eventIds.size(); i++) {
					inOperator.append(",?");
				}
				inOperator.append(")");

				String[] eventIdsArray = new String[eventIds.size()];
				eventIdsArray = eventIds.toArray(eventIdsArray);

				int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + inOperator.toString(), eventIdsArray);

				LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventIds));

				for (String eventId : eventIdsArray) {
					//update list listener
					updateMessageByEventId(eventId);
				}

			} else {
				LPLog.INSTANCE.d(TAG, "updateMessagesState - Skip updated messages , eventID is empty. messageState = " + messageState);
			}
		});
	}

	public void sendReadAckOnMessages(final String brandId, final String targetId, final String originatorId) {

		DataBaseExecutor.execute(() -> {

			// select serverSequence from messages where conversation state != CLOSED || LOCKED and targetId=<targetId> and messageState=ACCEPT
			Cursor cursor = null;
			if (!TextUtils.isEmpty(targetId)) {

				LPLog.INSTANCE.d(TAG, "Get all unread messages for target " + targetId);

				String queryString = String.format("select m.%s, c.%s, c.%s from %s m, %s c where c.%s>? and c.%s=? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
						MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
						DialogsTable.Key.STATE,DialogsTable.Key.TARGET_ID, MessagesTable.KEY_STATUS, MessagesTable.KEY_DIALOG_ID,
						DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

				//select m.serverSequence, c.conversationId from messages m, conversations c where c.messageState>? and c.targetId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
				cursor = getDB().rawQuery(queryString, ConversationState.LOCKED.ordinal() , targetId, MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			} else if (!TextUtils.isEmpty(brandId)) {
				LPLog.INSTANCE.d(TAG, "Get all unread messages for brand " + brandId );

				String queryString = String.format("select m.%s, c.%s, c.%s  from %s m, %s c where c.%s=? and c.%s>? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
						MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
						DialogsTable.Key.BRAND_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
						MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

				//select m.serverSequence, c.conversationId from messages m, conversations c where c.brandId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
				cursor = getDB().rawQuery(queryString, brandId, ConversationState.LOCKED.ordinal() , MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			}

			// If no open conversation found, check If there is a record with conversationState = CLOSED and messageState = received
			if (cursor !=null && !cursor.moveToFirst()) {
				String id = null;
				String queryString = null;
				String rawQuery = "select m.%s, c.%s, c.%s  from %s m, %s c where c.%s=? and c.%s=? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?";
				if ((!TextUtils.isEmpty(targetId))) {
					id = targetId;
					queryString = String.format(rawQuery,
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.TARGET_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
							MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);
				} else if (!TextUtils.isEmpty(brandId)) {
					id = brandId;
					queryString = String.format(rawQuery,
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.BRAND_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
							MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);
				}
				cursor = getDB().rawQuery(queryString, id, ConversationState.CLOSE.ordinal() , MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			}

			HashMap<String, List<Integer>> sequenceMap = new HashMap<>();
			HashMap<String, String> dialogIdsAndConversationIds = new HashMap<>();

			if (cursor != null) {
				try {
					List<Integer> list;

					if (cursor.moveToFirst()) {
						do {
							String dialogId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.DIALOG_ID));
							String conversationId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.CONVERSATION_ID));
							dialogIdsAndConversationIds.put(dialogId, conversationId);

							list = sequenceMap.get(dialogId);
							if (list == null) {
								list = new ArrayList<>();
								sequenceMap.put(dialogId, list);
							}

							int sequence = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));

							list.add(sequence);

						} while (cursor.moveToNext());
					}


				} finally {
					cursor.close();
				}
			} else {
				// Nothing to do
				return;
			}

			for (String dialogId : sequenceMap.keySet()) {
				LPLog.INSTANCE.d(TAG, "Send a read ack to the server for dialog id " + dialogId + " on the following sequences: " + sequenceMap.get(dialogId));

				// We have the sequence list of the messages we need to ack, run the command for updating the server
				String conversationId = dialogIdsAndConversationIds.get(dialogId);
				new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(brandId), (!TextUtils.isEmpty(targetId) ? targetId : brandId), dialogId, conversationId, sequenceMap.get(dialogId)).execute();
			}
		});
	}

	/**
	 * Send read acknowledgement to pusher only If pusher has stored positive count for that dialogId.
	 * @param conversationId
	 * @param dialogId
	 * @param brandId
	 */
	private void sendReadAckToPusherIfRequired(String conversationId, String dialogId, String brandId) {
		if (Configuration.getBoolean(R.bool.lp_pusher_clear_badge_count)) {
			clearPusherUnreadBadgeCount(brandId);
			return;
		}
		try {
			new GetUnreadMessagesCountCommand(mController, brandId, null, null, new ICallback<Integer, Exception>() {
				@Override
				public void onSuccess(Integer count) {
					if (count > 0) {
						Map<String, Integer> countMap = GetUnreadMessagesCountCommand.getUnreadCountMapped();
						if (countMap == null || TextUtils.isEmpty(dialogId)) {
							return;
						}
						Integer idCount = countMap.get(dialogId);
						if (idCount != null && idCount > 0) {
							new SendReadAcknowledgementCommand(mController, brandId, dialogId, conversationId, new ICallback<String, Exception>() {
								@Override
								public void onSuccess(String value) {
									LPLog.INSTANCE.d(TAG, "sendReadAckToPusherIfRequired: Cleared Pusher unread count for dialogId: " + dialogId);
								}

								@Override
								public void onError(Exception exception) {
									LPLog.INSTANCE.e(TAG, ERR_00000080, "sendReadAckToPusherIfRequired: Failed to Clear Pusher unread count for dialogId: " + dialogId);
								}
							}).execute();
						}
					}
				}

				@Override
				public void onError(Exception exception) {
					LPLog.INSTANCE.e(TAG, ERR_00000081, "sendReadAckToPusherIfRequired: Failed to fetch Unread Count Mapping. ", exception);
				}
			}).fetchUnreadCountMapping();
		} catch (Exception error) {
			LPLog.INSTANCE.e(TAG, ERR_00000082, "sendReadAckToPusherIfRequired: Failed to send read acknowledgement to pusher. ", error);
		}
	}

	public void clearPusherUnreadBadgeCount(String brandId) {
		if (Configuration.getBoolean(R.bool.lp_pusher_clear_badge_count)) {
			new ClearUnreadMessagesCountCommand(mController, brandId, null).execute();
		}
	}

	public void setDeliveryStatusUpdateCommand(Form form, DeliveryStatus deliveryStatus) {
		if (form == null) {
			LPLog.INSTANCE.w(TAG, "form not found!");
			return;
		}
		new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(form.getSiteId()), form.getSiteId(), form.getDialogId(), form.getConversationId(), form.getSeqId(), deliveryStatus).execute();
	}

	private ContentValues getContentValuesForMessage(final MessagingChatMessage message) {
		ContentValues messageValues = getContentValuesForMessageUpdate(message);
		messageValues.put(MessagesTable.KEY_EVENT_ID, message.getEventId());
		return messageValues;
	}

	private ContentValues wrapValuesForInsertion(
			ContentValues valuesForUpdated,
			MessagingChatMessage message
	) {
		ContentValues values = new ContentValues(valuesForUpdated);
		values.put(MessagesTable.KEY_EVENT_ID, message.getEventId());
		return values;
	}

	private ContentValues getContentValuesForMessageUpdate(final MessagingChatMessage message) {
		ContentValues messageValues = new ContentValues();
		messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		messageValues.put(MessagesTable.KEY_DIALOG_ID, message.getDialogId());

		// Text is encrypted before saving to DB
		final EncryptionVersion messageEncryptionVersion = DBEncryptionService.getAppEncryptionVersion();
		messageValues.put(MessagesTable.KEY_ENCRYPTION_VERSION, messageEncryptionVersion.ordinal());
		messageValues.put(MessagesTable.KEY_TEXT, message.getMessage());
		messageValues.put(MessagesTable.KEY_CONTENT_TYPE, message.getContentType());
		messageValues.put(MessagesTable.KEY_MESSAGE_TYPE, message.getMessageType().ordinal());
		messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		messageValues.put(MessagesTable.KEY_TIMESTAMP, message.getTimeStamp());
		messageValues.put(MessagesTable.KEY_ORIGINATOR_ID, message.getOriginatorId());
		messageValues.put(MessagesTable.KEY_METADATA, message.getMetadata());
		return messageValues;
	}

	public static MessagingChatMessage getSingleMessageFromCursor(Cursor cursor) {
		long msgId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
		EncryptionVersion encryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ENCRYPTION_VERSION)));

		MessagingChatMessage message = new MessagingChatMessage(
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_ORIGINATOR_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT)),
				cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID)),
				MessagingChatMessage.MessageType.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_MESSAGE_TYPE))],
				MessagingChatMessage.MessageState.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_STATUS))],
				cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_CONTENT_TYPE)),
				encryptionVersion);

		message.setMessageId(msgId);
		String metadata = DBEncryptionHelper.decrypt(encryptionVersion, cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_METADATA)));
		message.setMetadata(metadata);
		return message;
	}

	/**
	 * Method used to determine whether content event notification
	 * is related to conversation.
	 *
	 * If Content event notification is received from auto-bot message
	 * @param notification - Content event notification received from UMS/INCA
	 *
	 * @return true if content event notification is sent from controller and
	 * it's metadata is null or message type (inside metadata)
	 * is not equals to"YOU_ARE_CONNECTED_TO".
	 */
	private static boolean isNotificationFromController(ContentEventNotification notification) {
		return notification != null && Participants.ParticipantRole.CONTROLLER.equals(notification.originatorMetadata.mRole)
				&& (notification.metadata == null || (notification.metadata.optJSONObject(0) != null && !TYPE_CONVERSATION_ACCEPTED.equals(notification.metadata.optJSONObject(0).optString(KEY_NOTIFICATION_METADATA_MESSAGE_TYPE))));
	}

	/**
	 * Method used to apply messages status to received messages
	 * before encoding them and saving to database.
	 *
	 * @param rows required to represent
	 * @param statuses status holder for messages. Note: Statuses are sorted
	 *                   by message state descending to apply correct status.
	 */
	private static void applyMessagesStatus(
			List<FullMessageRow> rows,
			List<MessagesStatusHolder> statuses
	) {
		for (FullMessageRow row : rows) {
			MessagingChatMessage message = row.getMessagingChatMessage();
			if (isConsumer(message.getMessageType())) {
				for (MessagesStatusHolder status : statuses) {
					if (Objects.equals(message.getDialogId(), status.getDialogId())) {
						int messageSequence = message.getServerSequence();
						boolean shouldApply = (status.getSequence().contains(message.getServerSequence())
								|| messageSequence <= status.getMaxSequence())
								&& message.getMessageState().ordinal() < status.getState().ordinal();
						if (shouldApply) {
							message.setMessageState(status.getState());
						}
					}
				}
			}
		}
	}

	/**
	 * Update the Message Server Sequence & Message Status by event id
	 *
	 * @param eventId
	 * @param serverSequence
	 */
	public void updateOnMessageAck(final String eventId, final long serverSequence) {
		DataBaseExecutor.execute(() -> {
			{
				// Update the message server sequence
				final ContentValues contentValuesSequence = new ContentValues();
				contentValuesSequence.put(MessagesTable.KEY_SERVER_SEQUENCE, serverSequence);
				final String whereClauseSequence = MessagesTable.KEY_EVENT_ID + "=?";
				final String[] whereArgsSequence = {String.valueOf(eventId)};
				final int rowsAffected = getDB().update(contentValuesSequence, whereClauseSequence, whereArgsSequence);
				LPLog.INSTANCE.d(TAG, "Update msg server seq query. Rows affected=" + rowsAffected + " Seq=" + serverSequence);
			}

			{
				// Update the message status if relevant
				final ContentValues contentValuesStatus = new ContentValues();
				contentValuesStatus.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.SENT.ordinal());
				final String whereClauseStatus = MessagesTable.KEY_EVENT_ID + "=? AND ("
						+ MessagesTable.KEY_STATUS + "=? OR "
						+ MessagesTable.KEY_STATUS + "=? OR "
						+ MessagesTable.KEY_STATUS + "=?)";
				// Update to SENT but only if current status is PENDING or ERROR
				// (avoid lowering the status if messages are not arriving from the server in order)
				final String[] whereArgsStatus = {
						String.valueOf(eventId),
						String.valueOf(MessagingChatMessage.MessageState.PENDING.ordinal()),
						String.valueOf(MessagingChatMessage.MessageState.ERROR.ordinal()),
						String.valueOf(MessagingChatMessage.MessageState.OFFLINE.ordinal())
				};
				final int rowsAffected = getDB().update(contentValuesStatus, whereClauseStatus, whereArgsStatus);
				LPLog.INSTANCE.d(TAG, "Update msg status to SENT. Rows affected=" + rowsAffected);
			}

			//update list listener
			updateMessageByEventId(eventId);
		});
	}

	public DataBaseCommand<Boolean> transcendentMessagesExists(String dialogId, long timestamp) {
		return new DataBaseCommand<>(() -> {
			Cursor cursor = getDB().rawQuery("SELECT * FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_SERVER_SEQUENCE
					+ "=? AND " + MessagesTable.KEY_TIMESTAMP + "=? AND " + MessagesTable.KEY_DIALOG_ID + "=?", TRANSCENDENT_MESSAGE_SEQUENCE_NUMBER, timestamp, dialogId);

			if (cursor != null) {
				try {
					if (cursor.moveToFirst()) {
						return true;
					}
				} finally {
					cursor.close();
				}
			}
			return false;
		});
	}

	/**
	 * Return last message stored in DB If it has outbound message sequence
	 *
	 * @return message string
	 */
	public DataBaseCommand<String> getLatestOutboundMessage() {
		return new DataBaseCommand<>(() -> {
			Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_TEXT + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_SERVER_SEQUENCE
					+ " =? AND " + MessagesTable.KEY_DIALOG_ID + " =?", OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER, AmsDialogs.KEY_WELCOME_DIALOG_ID);

			if (cursor != null) {
				try {
					if (cursor.moveToFirst()) {
						return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT)));
					}
				} finally {
					cursor.close();
				}
			}
			return null;
		});
	}

	/**
	 * Return last welcome message (could be proactive message as well) stored in DB based on server sequence.
	 *
	 * @return MessagingChatMessage
	 */
	public DataBaseCommand<MessagingChatMessage> getLatestWelcomeMessageByServerSequence(int serverSequence) {
		return new DataBaseCommand<>(() -> {
			Cursor cursor = getDB().query(null, MessagesTable.KEY_SERVER_SEQUENCE
					+ " =? AND " + MessagesTable.KEY_DIALOG_ID + " =?", new String[]{String.valueOf(serverSequence), AmsDialogs.KEY_WELCOME_DIALOG_ID}, null, null, null);
			if (cursor != null) {
				try {
					if (cursor.moveToFirst()) {
						return getSingleMessageFromCursor(cursor);
					}
				} finally {
					cursor.close();
				}
			}
			return null;
		});
	}

	private boolean isWelcomeMessageExists(String dialogId) {
		String query = "SELECT EXISTS(" + "SELECT * FROM " + MessagesTable.MESSAGES_TABLE
					+ " WHERE " + MessagesTable.KEY_DIALOG_ID + " = ? "
					+ " AND "
					+ "(" + MessagesTable.KEY_SERVER_SEQUENCE + " = " + WELCOME_MSG_SEQUENCE_NUMBER
					+ " OR "
					+ MessagesTable.KEY_SERVER_SEQUENCE + " = " + OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER
					+ ")"
					+ ")";
		boolean result;
		try (Cursor cursor = getDB().rawQuery(query, dialogId)) {
			result = cursor.moveToFirst() && cursor.getInt(0) == 1;
		} catch (Exception exception) {
			result =  false;
		}
		return result;
	}

	/**
	 * Method used to generate an appropriate timestamp for first messages
	 * (outbound or welcome message). First messages aren't stored in
	 * database until conversation started.
	 * @param dialogId dialog id required to receive the oldest message.
	 * @param remoteTimestamp timestamp received from notification.
	 * @return oldest message's timestamp for requested dialog or remote timestamp message
	 * if dialog is empty.
	 */
	private long getTimestampForWelcomeMessage(String dialogId, long remoteTimestamp) {
		synchronized (DIALOG_ID_LOCKER) {
			String dialogCondition = MessagesTable.KEY_DIALOG_ID + " = ?";
			Dialog dialog = mController.amsDialogs.getActiveDialog();
			if (dialog != null && dialogId.equals(dialog.getDialogId())) {
				dialogCondition += " OR " + MessagesTable.KEY_DIALOG_ID + " = " + "\"" + Dialog.TEMP_DIALOG_ID + "\"";
			}
			String query = "SELECT " + MessagesTable.KEY_TIMESTAMP + " FROM " + MessagesTable.MESSAGES_TABLE
					+ " WHERE (" + dialogCondition + ")"
					+ " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " != " + AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER
					+ " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " != " + AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER
					+ " ORDER BY " + MessagesTable.KEY_TIMESTAMP + " ASC"
					+ " LIMIT 1";

			long res;
			try (Cursor cursor = getDB().rawQuery(query, dialogId)) {
				if (cursor.moveToFirst() && cursor.getCount() > 0) {
					long firstMessageTimeStamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
					if (remoteTimestamp > firstMessageTimeStamp) {
						res = firstMessageTimeStamp - 1;
					} else {
						res = remoteTimestamp;
					}
				} else {
					res = remoteTimestamp;
				}
			} catch (Exception exception) {
				LPLog.INSTANCE.d(TAG, "getTimestampForWelcomeMessage", exception);
				res = remoteTimestamp;
			}
			return res;
		}
	}

	private UpdateSQLCommand updateWelcomeMessageTimeStamp(String dialogId, long timestamp) {
		String whereClause = MessagesTable.KEY_EVENT_ID + " =? "
				+ "AND " + MessagesTable.KEY_DIALOG_ID + " =? ";
		String[] whereArgs = { dialogId + WELCOME_MESSAGE_EVENT_ID_POSTFIX, dialogId};
		ContentValues contentValues = new ContentValues();
		contentValues.put(MessagesTable.KEY_TIMESTAMP, getTimestampForWelcomeMessage(dialogId, timestamp));
		return new UpdateSQLCommand(contentValues, whereClause, whereArgs);
	}

	/**
	 * Update any existing outbound message in database with new message
	 * @param message new message to update
	 * @return database command
	 */
	public DataBaseCommand<Void> updateLastOutboundMessage(MessagingChatMessage message) {
		return new DataBaseCommand<>(() -> {
			try {
				String whereString = MessagesTable.KEY_SERVER_SEQUENCE + " =? AND " + MessagesTable.KEY_DIALOG_ID + " =?";
				String[] whereArgs = {String.valueOf(OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER), AmsDialogs.KEY_WELCOME_DIALOG_ID};
				ContentValues values = getContentValuesForMessageUpdate(message);
				ContentValues encrypted = DBEncryptionServiceUtilsKt.encrypt(values);
				int affectedRows = getDB().update(encrypted, whereString, whereArgs);
				if (affectedRows > 0) {
					prepareWelcomeMessageMetadata(message, null);
				}
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000153, "Exception updating last outbound message", e);
			}
			return null;
		});
	}

	/**
	 * Remove outbound message from database.
	 */
	public DataBaseCommand<Void> removeLastOutboundMessage() {
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, null, null, null, null, null)) {
				if (cursor.moveToFirst()) {
					int firstItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
					cursor.moveToLast();
					int lastItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
					LPLog.INSTANCE.d(TAG, "removeLastOutboundMessage: removing outbound message");

					int	affectedRows = getDB().removeAll(MessagesTable.KEY_SERVER_SEQUENCE + "=? AND " + MessagesTable.KEY_DIALOG_ID + "=?",
							new String[]{String.valueOf(OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER), AmsDialogs.KEY_WELCOME_DIALOG_ID});
					// If Outbound message was only message in DB (1st time user), add welcome message
					if (firstItemId == lastItemId) {
						if (affectedRows > 0) {
							LPLog.INSTANCE.d(TAG, "removeLastOutboundMessage: Set welcomeMessage to be added");
							shouldAddWelcomeMessage = true; // We don't have message listener available yet. Wait for it.
						}
					}

					if (affectedRows > 0) {
						LPLog.INSTANCE.d(TAG, "removeLastOutboundMessage: affectedRows > 0");
						// Remove welcome message metadata - proactive
						mController.setWelcomeMessageMetadata(null, mController.getActiveBrandId());
					}
				} else {
					mController.amsDialogs.hasWelcomeMessageDialog().setPostQueryOnBackground(hasWelcomeMessageDialog -> {
						if (hasWelcomeMessageDialog) {
							LPLog.INSTANCE.i(TAG, "removeLastOutboundMessage: Found welcome message dialog without welcome message. Add welcomeMessage listener.");
							shouldAddWelcomeMessage = true; // We don't have any message in DB but have welcome message dialog. Add welcome message.
						}
					}).execute();
				}
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000154, "Exception while removing last outbound message", e);
			}
			return null;
		});
	}

	/**
	 * Update welcome message's temp dialog id with conversation dialogId when started new conversation.
	 * Do it only when the dialog is still OPEN.
	 * @param dialogId conversation dialogId
	 * @param tempServerSequence welcome message sequence
	 */
	private void updateTempDialogId(String dialogId, int tempServerSequence, long timestamp) {
		getLatestWelcomeMessageByServerSequence(tempServerSequence).setPostQueryOnBackground(welcomeMessage -> {
			Dialog dialog = mController.amsDialogs.queryDialogById(dialogId).executeSynchronously();
			if (welcomeMessage != null && dialog != null && DialogState.OPEN == dialog.getState()) {
				try {
					final ContentValues contentValues = new ContentValues();
					contentValues.put(MessagesTable.KEY_EVENT_ID, dialogId + WELCOME_MESSAGE_EVENT_ID_POSTFIX);
					contentValues.put(MessagesTable.KEY_DIALOG_ID, dialogId);
					if (timestamp > 0) {
						contentValues.put(MessagesTable.KEY_TIMESTAMP, timestamp);
					}
					String whereClause = MessagesTable.KEY_ID + "=? AND "
							+ MessagesTable.KEY_SERVER_SEQUENCE + "=? AND "
							+ MessagesTable.KEY_DIALOG_ID + "=?";
					final String[] whereArgsSequence = {String.valueOf(welcomeMessage.getLocalId()), String.valueOf(tempServerSequence), AmsDialogs.KEY_WELCOME_DIALOG_ID};

					int updatedRows = getDB().update(contentValues, whereClause, whereArgsSequence);
					LPLog.INSTANCE.d(TAG, "updateTempDialogIds , updatedRows = " + updatedRows);
					if (updatedRows > 0) {
						welcomeMessage.setDialogId(dialogId);
						welcomeMessage.setEventId(dialogId + WELCOME_MESSAGE_EVENT_ID_POSTFIX);
						if (timestamp > 0) {
							welcomeMessage.setTimeStamp(timestamp);
						}
						if (welcomeMessage.getServerSequence() == AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER) {
							getMessagesListener().updateTempWelcomeMessage(createFullMessageRow(welcomeMessage.getLocalId(), welcomeMessage, -1));
						} else {
							getMessagesListener().onUpdateMessage(createFullMessageRow(welcomeMessage.getLocalId(), welcomeMessage, -1));
						}
					}
				} catch (Exception error) {
					LPLog.INSTANCE.e(TAG, ERR_00000152, "updateTempDialogId: Failed to update TEP WELCOME_DIALOG_ID with conversation dialogId: " + error);
				}
			}
		}).execute();
	}

	/**
	 * Update messages serverId for messages that are waiting for create dialog response
	 *
	 * @param serverDialogId      - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessagesDialogServerID(final String serverDialogId, String dialogId) {
		return new DataBaseCommand<>(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_DIALOG_ID, serverDialogId);

			synchronized (DIALOG_ID_LOCKER) {
				int updatedRows = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + " =? ", new String[]{ dialogId });
				updateAllMessagesForDialog(serverDialogId);
				LPLog.INSTANCE.d(TAG, "updateMessagesConversationServerID , updatedRows = " + updatedRows);
			}
			//update list listeners
			return null;
		});
	}

	/**
	 * Update messages serverId  and timestamp for message by row id
	 *
	 * @param messageRowId   - specific row to update
	 * @param dialogId - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessageDialogServerIdAndTime(final long messageRowId, final String dialogId) {

		return new DataBaseCommand<>(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_DIALOG_ID, dialogId);
			int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

			LPLog.INSTANCE.d(TAG, "updateMessageDialogServerIdAndTime , rowId to update = " + messageRowId + ", updated = " + updatedRows);
			//update list listeners
			updateMessageByRowIdOnDbThread(messageRowId);
			return null;
		});
	}

	/**
	 * Update messages state for message by row id
	 *
	 * @param messageRowId - specific row to update
	 * @param state        - new state to set
	 * @return list of the updated messages
	 */
	public void updateMessageState(final long messageRowId, final MessagingChatMessage.MessageState state) {
		DataBaseExecutor.execute(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_STATUS, state.ordinal());
			int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

			LPLog.INSTANCE.d(TAG, "updateMessageState , rowId to update = " + messageRowId + ", updated = " + updatedRows);
			//update list listeners
			updateMessageByRowIdOnDbThread(messageRowId);
		});
	}

	/**
	 * Update message related to file when file updated
	 *
	 * @param messageRowId - specific row to update
	 */
	public void updateMessageFileChanged(final long messageRowId) {
		if (messageRowId < 0) {
			LPLog.INSTANCE.w(TAG, "updateMessageFileChanged cannot be lower than zero! " + messageRowId);
			return;
		}
		DataBaseExecutor.execute(() -> updateMessageByRowIdOnDbThread(messageRowId));
	}

	/***************************
	 * UPDATE UI LISTENER
	 ******************************/

	private MessagesListener getMessagesListener() {
		if (mMessagesListener != null){
			return mMessagesListener;
		}
		return mNullMessagesListener;
	}

	//region AmsMessagesLoaderProvider
    /**
     * Adding MessagesListener to get updates for changes in messages
     *
     * @param messagesListener     listener to notify on updates
     * @param messagesSortedByType way to sort messages by brand or target id
     * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
     */
    @Override
    public void addOnUpdateListener(MessagesListener messagesListener, final MessagesSortedBy messagesSortedByType, final String typeValue) {
        mMessagesListener = messagesListener;
		final int limit = messagesListener.getLoadLimit();
		executePendingListenerTasks();
        DataBaseExecutor.execute(() -> {
			// Query the DB

			ArrayList<FullMessageRow> searchedMessageList = loadMessagesOnDbThread(
					messagesSortedByType,
					typeValue,
					limit,
					-1,
					-1
			);
			List<FullMessageRow> rows = maskMessagesIfNeeded(mController.getActiveBrandId(), searchedMessageList);
			getMessagesListener().initMessages(new ArrayList<>(rows));
		});

    }

    @Override
    public void removeOnUpdateListener() {
        mMessagesListener = null;
    }

    @Override
    public boolean hasListener() {
        return mMessagesListener != null;
    }

	/**
	 * Loading messages by specific rules:
	 *
	 * @param messagesSortedByType way to sort messages by brand or target id
	 * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
	 * @param limit                max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp   negative value if there is no need to use this value
	 * @param newerThanTimestamp   negative value if there is no need to use this value
	 * @return
	 */
	@Override
	public DataBaseCommand<ArrayList<FullMessageRow>> loadMessages(final MessagesSortedBy messagesSortedByType, final String typeValue, final int limit, final long olderThanTimestamp, final long newerThanTimestamp) {
		return new DataBaseCommand<>(() -> {
			// Query the DB
			return loadMessagesOnDbThread(messagesSortedByType, typeValue, limit, olderThanTimestamp, newerThanTimestamp);
		});

	}

	@Nullable
	public ArrayList<FullMessageRow> loadMessagesOnDbThread(MessagesSortedBy messagesSortedByType, String typeValue, int limit, long olderThanTimestamp, long newerThanTimestamp) {
		Cursor cursor = null;
		LPLog.INSTANCE.d(TAG, "Loading " + limit + " messages by " + messagesSortedByType);
		switch (messagesSortedByType) {
			case TargetId:
				cursor = messagesByTarget(typeValue, limit, olderThanTimestamp, newerThanTimestamp);
				break;
			case ConversationId:
				cursor = messagesByConversationID(typeValue, limit);
				break;
			case DialogId:
				cursor = messagesByDialogID(typeValue, limit);
				break;
		}

		String activeBrandId = mController.getActiveBrandId();
		boolean isUnAuth = MessagingFactory.getInstance()
				.getController()
				.mAccountsController
				.isInUnAuthMode(activeBrandId);

		ArrayList<FullMessageRow> searchedMessageList = null;
		if (cursor != null) {
			try {
				searchedMessageList = new ArrayList<>(cursor.getCount());
				if (cursor.moveToFirst()) {
					do {
						FullMessageRow row = new FullMessageRow(cursor);
						if (isUnAuth && Configuration.getBoolean(R.bool.enable_client_only_masking)) {
							searchedMessageList.add(maskMessage(activeBrandId, row));
						} else {
							searchedMessageList.add(row);
						}
					} while (cursor.moveToNext());
				}
			} finally {
				cursor.close();
			}
		}
		return searchedMessageList;
	}

	@Override
	public String getMyUserId(String targetId) {
		return mController.getOriginatorId(targetId);
	}

	@Override
	public MessagingUserProfile loadMessagingUserProfile(String originatorId) {
		return mController.amsUsers.getUserById(originatorId).executeSynchronously();
	}
	//endregion

	@NonNull
	private FullMessageRow createFullMessageRow(long messageRowId, @NonNull MessagingChatMessage message, long fileRowId) {
		if (messageRowId > 0) {
			message.setLocalId(messageRowId);
		}
		MessagingUserProfile messagingUserProfile = getUserProfile(message.getOriginatorId());
		String avatarUrl = messagingUserProfile == null ? "" : messagingUserProfile.getAvatarUrl();
		String agentNickname = messagingUserProfile == null ? "" : messagingUserProfile.getNickname();
		FileMessage fileMessage;
		if (fileRowId != -1) {
			fileMessage = mController.amsFiles.getFileByFileRowIdOnDbThread(fileRowId);
		} else {
			fileMessage = mController.amsFiles.getFileByMessageRowId(messageRowId);
		}
		FullMessageRow fullMessageRow = new FullMessageRow(message, avatarUrl, fileMessage);
		fullMessageRow.setAgentNickName(agentNickname);
		return fullMessageRow;
	}

	private synchronized MessagingUserProfile getUserProfile(String originatorId) {
		MessagingUserProfile profile = mMessagingProfiles.get(originatorId);
		if (profile == null) {
			profile = mController.amsUsers.getUserById(originatorId).executeSynchronously();
			mMessagingProfiles.put(originatorId, profile);
			return profile;
		}
		return profile;
	}

	/**
	 * Clear all closed conversation messages of the given targetId
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearMessagesOfClosedConversations(final String targetId) {

		return new DataBaseCommand<>(() -> {
			//"where messages._id in (select m._id from messages m, dialogs d where d.targetId='21337028' and d.state=0 and d.dialog_id=m.dialogId or m.dialogId=KEY_WELCOME_DIALOG_ID)"
			String whereString = MessagesTable.KEY_ID + " in (select m." + MessagesTable.KEY_ID + " from " + MessagesTable.MESSAGES_TABLE + " m, "
					+ DialogsTable.TABLE_NAME + " d where d." + DialogsTable.Key.TARGET_ID + "=? and d." + DialogsTable.Key.STATE
					+ "=? and d." + DialogsTable.Key.DIALOG_ID+ "=m." + MessagesTable.KEY_DIALOG_ID + " or m." + MessagesTable.KEY_DIALOG_ID + "=?)";
			String[] whereArgs = {targetId, String.valueOf(DialogState.CLOSE.ordinal()), AmsDialogs.KEY_WELCOME_DIALOG_ID}; // Closed conversations for targetId


			int removed = getDB().removeAll(whereString, whereArgs);
			LPLog.INSTANCE.d(TAG, "clearMessagesOfClosedConversations: removed: " + removed + " where: " + whereString + ", whereArgs: " + Arrays.toString(whereArgs));

			//update list listeners
			getMessagesListener().removeAllClosedConversations(targetId);
			return removed;
		});
	}

	/**
	 * Clear all closed conversation messages of the given targetId. This will be triggered when upgrade the SDK version or background app for unauth account.
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearAllMessages(final String targetId) {

		return new DataBaseCommand<>(() -> {

			int removed = getDB().removeAll(null, null);
			LPLog.INSTANCE.d(TAG, "clearAllMessages from messages table");

			//update list listeners
			getMessagesListener().clearAllMessages(targetId);
			return removed;
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
	 * @param firstNotification
	 * @param dialogId
	 * @param firstSequence
	 * @param lastSequence
	 */
    private void updateMessages(boolean firstNotification, String dialogId, int firstSequence, int lastSequence) {
        long firstTimestamp = getTimestampMessage(dialogId, firstSequence);
        long lastTimestamp = getTimestampMessage(dialogId, lastSequence);
		if (firstTimestamp > lastTimestamp) {
			long actualTimestamp = firstTimestamp;
			firstTimestamp = lastTimestamp;
			lastTimestamp = actualTimestamp;
		}
        if (firstNotification){
			LPLog.INSTANCE.d(TAG, "updateMessages first notification event. onQueryMessagesResult ");
	        getMessagesListener().onQueryMessagesResult(firstTimestamp, lastTimestamp);
		}else{
			LPLog.INSTANCE.d(TAG, "updateMessages NOT first notification event. onUpdateMessages ");
			getMessagesListener().onUpdateMessages(firstTimestamp, lastTimestamp);
		}
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param eventId
     */
    private void updateMessageByEventId(String eventId) {
        Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    MessagingChatMessage message = getSingleMessageFromCursor(cursor);
                    int rowId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
                    getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
                }
            } finally {
                cursor.close();
            }
        }

	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private String getEventIdForMessage(String dialogId, int sequence) {
		String eventId = null;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_EVENT_ID + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
				}
			} finally {
				cursor.close();
			}
		}
		return eventId;
	}

	public DataBaseCommand<MessagingChatMessage> getMessageByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPLog.INSTANCE.w(TAG, "getMessageByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null)) {
				if (cursor != null) {
					return getSingleMessageFromCursor(cursor);
				}
			}
			return null;
		});
	}

	/**
	 * @return If resolve message is already exists for the dialog.
	 */
	public DataBaseCommand<Boolean> isResolveMessageForDialogAdded(final String dialogId) {
		if (TextUtils.isEmpty(dialogId)) {
			LPLog.INSTANCE.w(TAG, "getResolveMessageForDialog - dialogId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			// SELECT dialogId FROM messages WHERE dialogId={dialogId} AND serverSequence=-2
			String queryString = "SELECT " + MessagesTable.KEY_DIALOG_ID + " FROM " + MessagesTable.MESSAGES_TABLE
					+ " WHERE " + MessagesTable.KEY_DIALOG_ID
					+ "= ? AND " + MessagesTable.KEY_EVENT_ID + " = ? LIMIT 1";
			try (Cursor cursor = getDB().rawQuery(queryString, dialogId, dialogId + RESOLVE_MESSAGE_EVENT_ID_POSTFIX)) {
				if (cursor != null && cursor.moveToFirst()) {
					String dbDialogId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID));
					return dbDialogId.equals(dialogId);
				}
			} catch (Exception e) {
				LPLog.INSTANCE.w(TAG, e.getMessage(), e);
			}
			return false;
		});
	}

	public DataBaseCommand<Long> getRowIdByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPLog.INSTANCE.w(TAG, "getRowIdByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(new String[]{MessagesTable.KEY_ID}, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null)) {
				if (cursor != null) {
					return cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
				}
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000087, "Exception while getting a rowId by eventId", e);
			}
			return -1L;
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param messageRowId
     */
    private void updateMessageByRowIdOnDbThread(long messageRowId) {
        MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
        if (message != null){
			getMessagesListener().onUpdateMessage(createFullMessageRow(messageRowId, message, -1));
		} else {
			LPLog.INSTANCE.e(TAG, ERR_00000088, "updateMessageByRowIdOnDbThread - message does not exist");
		}
    }

	public DataBaseCommand<Void> updateFileMessageByRowId(final long messageRowId, final long fileRowId) {

		return new DataBaseCommand<>(() -> {
			MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
			if (message == null) {
				message = getMessagesFileRowId(fileRowId);
			}
			FullMessageRow messageRow = createFullMessageRow(messageRowId, message, fileRowId);

			getMessagesListener().onUpdateMessage(messageRow);
			return null;
		});
    }

    @Nullable
	private MessagingChatMessage getMessageByRowIdOnDbThread(long messageRowId) {
		Cursor cursor = getDB().query(null, MessagesTable.KEY_ID + " = ?", new String[]{String.valueOf(messageRowId)}, null, null, null);
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return getSingleMessageFromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

	@Nullable
	private MessagingChatMessage getMessagesFileRowId(long fileRowId) {
    	String request = getBasicMessagesQuery()
				.append( " inner join " + FilesTable.FILES_TABLE + " on ")
				.append(FilesTable.FILES_TABLE + "." + FilesTable.KEY_RELATED_MESSAGE_ROW_ID)
				.append(" = " )
				.append(MessagesTable.MESSAGES_TABLE + "." + MessagesTable.KEY_ID)
				.append(" where " + FilesTable.FILES_TABLE + "." + FilesTable.KEY_ID + " = ?")
				.append(" LIMIT 1")
				.toString();
    	try (Cursor cursor = getDB().rawQuery(request, fileRowId)) {
    		if (cursor.moveToFirst()) {
				return new FullMessageRow(cursor).getMessagingChatMessage();
			} else {
				return null;
			}
		} catch (Exception exception) {
			return null;
		}
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param rowId
     * @param message
     */
    private void updateMessageByRowId(long rowId, MessagingChatMessage message) {
        getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param dialogId
     */
    private void updateAllMessagesForDialog(String dialogId) {
        Cursor cursor = getDB().query(new String[]{"MIN(" + MessagesTable.KEY_TIMESTAMP + ")", "MAX(" + MessagesTable.KEY_TIMESTAMP + ")"},
                MessagesTable.KEY_DIALOG_ID + " = ?", new String[]{dialogId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    long firstMessageTimestampForConversation = cursor.getLong(0);
                    long lastMessageTimestampForConversation = cursor.getLong(1);
                    getMessagesListener().onUpdateMessages(firstMessageTimestampForConversation, lastMessageTimestampForConversation);

				}
			} finally {
				cursor.close();
			}
		}
	}

	private void showErrorToast(MessagingChatMessage message) {
		final int toastTextResId;
		switch (message.getMessageType()) {
			case CONSUMER:
			case CONSUMER_MASKED: toastTextResId = R.string.lp_failed_to_send_message;
				break;
			case CONSUMER_URL:
			case CONSUMER_URL_MASKED: toastTextResId = R.string.lp_failed_to_send_link;
				break;
			case CONSUMER_FORM: toastTextResId = R.string.lp_failed_to_send_secure_form_answer;
				break;
			default: toastTextResId = -1;
				break;
		}
		MessagesListener listener = getMessagesListener();
		if (toastTextResId != -1 && listener != null) {
			listener.showErrorToast(toastTextResId);
		}
	}


	/**
	 * updating messages after fetch history.
	 */
	public void updateFetchHistoryEnded(final boolean updated) {
		DataBaseExecutor.execute(() -> {
			if (updated) {
				getMessagesListener().onHistoryFetched();
			} else {
				getMessagesListener().onHistoryFetchedFailed();
			}
		});
	}

	/**
	 * Updating messages when agent details relevant for conversation was updated.
	 *
	 * @param dialogId
	 */
	public void updateAgentDetailsUpdated(String dialogId) {
		updateAllMessagesForDialog(dialogId);
	}

    public void updateHandledExConversation(boolean emptyNotification) {
        getMessagesListener().onExConversationHandled(emptyNotification);
    }

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private long getTimestampMessage(String dialogId, int sequence) {
		long timestamp = 0;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_TIMESTAMP + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					timestamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
				}
			} finally {
				cursor.close();
			}
		}
		return timestamp;
	}

	@Override
	public void shutDown() {
		mMessageTimeoutQueue.removeAll();
	}

	@Override
	public void clear() {
	}

	public void removeAllMessages(String brandId) {
		getMessagesListener().clearAllMessages(brandId);
	}

	private void executePendingListenerTasks() {
		if (shouldAddWelcomeMessage) {
			shouldAddWelcomeMessage = false;
			getMessagesListener().addFirstWelcomeMessage();
		}
	}

	/**
	 * Prepare WelcomeMessage metadata if whenever we insert or update a welcome message in database
	 * <br> WelcomeMessage metadata may contain: text/plain message, quick replies, rich content (structured content message)
	 * @param welcomeMessage represented welcome message.
	 * @param quickRepliesContent quick replies associated with represented welcome message.
	 */
	public void prepareWelcomeMessageMetadata(final MessagingChatMessage welcomeMessage, @Nullable String quickRepliesContent) {
		LPLog.INSTANCE.d(TAG, "prepareWelcomeMessageMetadata");
		try {
			JSONObject wmMetadata = new JSONObject();
			// type can be WelcomeMessage or Proactive
			if (welcomeMessage.getServerSequence() == OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER) {
				wmMetadata.put(MessagingConst.KEY_TYPE, MessagingConst.VALUE_PROACTIVE);
			} else {
				wmMetadata.put(MessagingConst.KEY_TYPE, MessagingConst.VALUE_WELCOME_MESSAGE);
			}
			JSONObject metadataContent = new JSONObject();

			if (welcomeMessage.getMessageType() == MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT) {
				metadataContent.put(MessagingConst.KEY_TYPE, MessagingConst.VALUE_RICH_CONTENT_EVENT);
				metadataContent.put(MessagingConst.KEY_RICH_CONTENT, new JSONObject(welcomeMessage.getMessage()));
			} else {
				metadataContent.put(MessagingConst.KEY_TYPE, MessagingConst.VALUE_CONTENT_EVENT);
				metadataContent.put(MessagingConst.KEY_CONTENT_TYPE, MessagingConst.VALUE_TEXT_PLAIN);
				metadataContent.put(MessagingConst.KEY_MESSAGE, welcomeMessage.getMessage());
			}

			// Proactive message does not have QR so far
			final String brandId = mController.getActiveBrandId();
			final JSONObject quickRepliesMetadata;
			if (quickRepliesContent != null && StringsKt.isBlank(quickRepliesContent)) {
				quickRepliesMetadata = new JSONObject(quickRepliesContent);
			} else if (welcomeMessage.getServerSequence() != OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER && !TextUtils.isEmpty(brandId)) {
				final QuickRepliesMessageHolder holder = mController.amsMessages.getQuickRepliesMessageHolder(brandId);
				if (holder != null && holder.isQuickReplyActionsPresented()) {
					quickRepliesMetadata = new JSONObject(holder.getQuickRepliesString());
					LPLog.INSTANCE.d(TAG, "welcomeMessage QuickReply is visible");
				} else {
					quickRepliesMetadata = null;
				}
			} else  {
				quickRepliesMetadata = null;
			}

			if (quickRepliesMetadata != null) {
				metadataContent.put(MessagingConst.KEY_QUICK_REPLIES, quickRepliesMetadata);
			}

			wmMetadata.put(MessagingConst.KEY_EVENT, metadataContent);
			mController.setWelcomeMessageMetadata(wmMetadata.toString(), brandId);
		} catch (JSONException e) {
			LPLog.INSTANCE.e(TAG, ERR_0000016B, "Failed to prepare metadata of welcomeMessage", e);
		}
	}

	/**
	 * Method used to clear metadata of welcome message for active brand.
	 */
	public void clearWelcomeMessageMetadata() {
		final String brandId = mController.getActiveBrandId();
		mController.setWelcomeMessageMetadata(null, brandId);
	}

	public OfflineMessagesRepository getOfflineMessagesRepository() {
		return mOfflineMessagesRepository;
	}

	/**
	 * Method used to update an offline welcome message sent for previously
	 * activated dialog from offline.
	 * @param timestamp future timestamp of welcome message.
	 * @param dialogId possible related dialog id.
	 */
	private void updateOfflineWelcomeMessage(long timestamp, @Nullable String dialogId) {
		String whereClause = MessagesTable.KEY_TIMESTAMP + " = ?"
				+ " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
		List<String> arguments = new ArrayList<>();
		arguments.add(String.valueOf(OFFLINE_WELCOME_MESSAGE_TIMESTAMP));
		arguments.add(String.valueOf(WELCOME_MSG_SEQUENCE_NUMBER));
		if (dialogId != null) {
			whereClause += " AND " + MessagesTable.KEY_EVENT_ID + " = ?";
			arguments.add(dialogId + WELCOME_MESSAGE_EVENT_ID_POSTFIX);
		}
		String[] columns = {
				MessagesTable.KEY_EVENT_ID
		};
		String[] whereArgs = arguments.toArray(new String[] { });
		try (Cursor cursor = getDB().query(columns, whereClause, whereArgs, null, null, null)){
			if (cursor == null || !cursor.moveToFirst() || cursor.getCount() < 1) {
				return;
			}
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_TIMESTAMP, timestamp);
			String eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
			int result = getDB().update(contentValues, whereClause, whereArgs);
			if (result > 0) {
				getMessagesListener().updateMessageTimestampByEventId(eventId, timestamp);
			}
		} catch (Throwable exception) {
			LPLog.INSTANCE.d(TAG, "Failed to update offline message", exception);
		}
	}

	/**
	 * Method used to request messages updates after all
	 * received messages and accept status events were received.
	 *
	 * @param firstNotification flag to determine whether events were received adter first
	 *                          subscription or not. Equals true if consumer navigates
	 *                          to conversation screen and false otherwise.
	 * @param dialogId identifier of dialog for which content or accept status events were received
	 * @param firstSequence sequence of first received message that requires updates
	 * @param lastSequence sequence of last received message that requires updates
	 * @param messageRows parsed message row of received content events.
	 */
	private void requestMessagesUpdates(
			boolean firstNotification,
			String dialogId,
			int firstSequence,
			int lastSequence,
			List<FullMessageRow> messageRows
	) {
        Set<String> eventsIds = mOfflineMessagesRepository.getPendingOfflineMessages(mController.getActiveBrandId());
		for (FullMessageRow row : messageRows) {
			MessagingChatMessage chatMessage = row.getMessagingChatMessage();
			LPLog.INSTANCE.d(TAG, "OfflineMessaging. Process offline message: " + chatMessage.getEventId());
			if (isConsumer(chatMessage.getMessageType())) {
				String eventId = chatMessage.getEventId();
				long timestamp = chatMessage.getTimeStamp();
				if (eventsIds.contains(eventId)) {
					eventsIds.remove(eventId);
					updateOfflineMaskedMessage(eventId, timestamp + 1);
					getMessagesListener().updateMessageTimestampByEventId(eventId, timestamp);
				}
			}
		}
		mOfflineMessagesRepository.setPendingOfflineMessages(mController.getActiveBrandId(), eventsIds);
		updateMessages(firstNotification, dialogId, firstSequence, lastSequence);
	}

	private void updateOfflineMaskedMessage(String eventId, long timestamp) {
		LPLog.INSTANCE.d(TAG, "Reorder masked message for offline message: " + eventId);
		ContentValues contentValues = new ContentValues();
		contentValues.put(MessagesTable.KEY_TIMESTAMP, timestamp);
		String whereClause = MessagesTable.KEY_EVENT_ID + " = ?";
		String maskedMessageEventId = eventId + MASKED_MESSAGE_EVENT_ID_POSTFIX;
		String[] whereArgs = { maskedMessageEventId };
		int affectedRows = getDB().update(contentValues, whereClause, whereArgs);
		LPLog.INSTANCE.d(TAG, "Reordering of masked message finished. Rows count: " + affectedRows + ", args: " + Arrays.toString(whereArgs));
	}

	/**
	 * Class that represents the state of message based on held message's sequence.
	 * Class is used to represent the actual state of message that would be represented
	 * before encoding and saving them to database.
	 *
	 * @see #applyMessagesStatus(List, List)
	 * @see #addMultipleMessages(ArrayList, Dialog, String, String, String, String, String, long, boolean, boolean)
	 * @see #createStatementForUpdateMessagesState(String, int[], MessagingChatMessage.MessageState, int, ContentValues, StringBuilder, String[], List)
	 */
	private static class MessagesStatusHolder {

		private final String mDialogId;
		private final TreeSet<Integer> mSequence;
		private final MessagingChatMessage.MessageState mState;
		private final int maxSequence;

		private MessagesStatusHolder(String dialogId, TreeSet<Integer> sequence, MessagingChatMessage.MessageState state, int maxSequence) {
			this.mDialogId = dialogId;
			this.mSequence = sequence;
			this.mState = state;
			this.maxSequence = maxSequence;
		}

		public String getDialogId() {
			return mDialogId;
		}

		public TreeSet<Integer> getSequence() {
			return mSequence;
		}

		public MessagingChatMessage.MessageState getState() {
			return mState;
		}

		public int getMaxSequence() {
			if (mSequence.isEmpty()) {
				return maxSequence;
			} else {
				return mSequence.last();
			}
		}
	}
}
