package com.liveperson.messaging.offline.api

import androidx.annotation.WorkerThread
import com.liveperson.messaging.model.FullMessageRow

interface OfflineMessagesRepository {

    /**
     * Method used to check whether pending offline messages exist
     * in database. If they do exist and offline messaging is enabled
     * SDK should resend them when history is fully synced with server.
     * @param brandId  particular brand associated with consumer that
     *                 could have an active dialog.
     * @return true if database contains any offline messages sent from consumer.
     */
    @WorkerThread
    fun areOfflineMessagesExists(brandId: String): Boolean

    /**
     * Method used for loading offline messages except the ones associated with the event ids passed
     * as the second param
     * @param brandId the id of the brand
     * @param eventIds a set of the event ids associated with the messages to be filtered out
     */
    @WorkerThread
    fun loadOfflineMessagesExcept(brandId: String, eventIds: Set<String>): List<FullMessageRow>

    /**
     * Method for removing a message from the database
     * @param eventId the event id associated with the message to be removed
     */
    @WorkerThread
    fun removeMessage(eventId: String)

    /**
     * Method used for checking whether an offline welcome message already exists in the database
     * @param dialogId the id of the dialog in which the offline welcome message was shown
     */
    @WorkerThread
    fun isOfflineWelcomeMessageExists(dialogId: String): Boolean

    /**
     * Method used for updating the offline messages in the database
     * @param actualDialogId the id of the dialog to which the messages were added
     */
    @WorkerThread
    fun updateOfflineMessagesDialogId(actualDialogId: String)

    /**
     * Queries the database for the first and last message timestamp for the dialog with the
     * supplied id and calls a listener to update the messages on UI
     * @param dialogId the id of the dialog to be queried
     */
    @WorkerThread
    fun triggerUpdateMessagesForDialogId(dialogId: String)

    /**
     * Method used for removing the offline welcome message from the database
     */
    @WorkerThread
    fun removeOfflineWelcomeMessage()

    /**
     * Method used for removing pending offline messages from the database
     * @param eventIds a set containing the event ids associated with the messages to be removed
     */
    @WorkerThread
    fun removePendingOfflineMessages(brandId: String)

    /**
     * Method used for storing the event ids for pending offline messages in shared prefs
     * @param eventIds the event ids associated with the pending offline messages
     */
    @WorkerThread
    fun setPendingOfflineMessages(brandId: String, eventIds: Set<String>)

    /**
     * Method used for retrieving the event ids for pending offline messages from shared prefs
     * @return a set containing the event ids associated with the pending offline messages
     */
    @WorkerThread
    fun getPendingOfflineMessages(brandId: String): Set<String>

    /**
     * Method used for removing the event id associated with the pending offline message from
     * shared prefs
     * @param eventId the event id associated with the pending offline message to be removed
     */
    @WorkerThread
    fun removePendingOfflineMessage(brandId: String, eventId: String)

    /**
     * Method used for checking if a message is pending
     * @param eventId the event id associated with the message to be checked
     * @return true if the event id is found in shared prefs or false otherwise
     */
    @WorkerThread
    fun isMessagePending(brandId: String, eventId: String): Boolean
}