package com.liveperson.messaging.wm

import android.content.ContentProvider
import android.content.ContentValues
import android.content.Context
import android.content.SharedPreferences
import android.content.UriMatcher
import android.database.Cursor
import android.net.Uri
import com.liveperson.messaging.wm.WelcomeMessageBroadcastReceiver.Companion.ACTION_WELCOME_MESSAGE_CHANGED
import com.liveperson.messaging.wm.WelcomeMessageBroadcastReceiver.Companion.sendWelcomeMessageEvent
import com.liveperson.messaging.utils.write
import com.liveperson.messaging.wm.impl.COMMON_KEY
import com.liveperson.messaging.wm.impl.onMatched
import com.liveperson.messaging.wm.impl.onPathMatched
import com.liveperson.messaging.wm.impl.getValueAsCursor
import com.liveperson.messaging.wm.impl.welcomeMessageMimeType
import com.liveperson.messaging.wm.impl.welcomeMessageUriMatcher

/**
 * A content provider for dynamic welcome message.
 * This content provider serves as repository for welcome message
 * and a synchronization unit to interact with welcome message
 * in multi-process environment.
 */
class WelcomeMessageContentProvider : ContentProvider(),
    SharedPreferences.OnSharedPreferenceChangeListener {

    companion object {

        private const val LOCAL_SHARED_PREFERENCES_NAME = "welcome.message.preferences.local"
    }

    private val localSharedPreferences: SharedPreferences by lazy {
        context!!.getSharedPreferences(LOCAL_SHARED_PREFERENCES_NAME, Context.MODE_PRIVATE)
    }

    private val uriMatcher: UriMatcher by lazy {
        context!!.welcomeMessageUriMatcher
    }

    override fun onCreate(): Boolean {
        localSharedPreferences.registerOnSharedPreferenceChangeListener(this)
        return true
    }

    override fun query(
        uri: Uri,
        projection: Array<out String>?,
        selection: String?,
        selectionArgs: Array<out String>?,
        sortOrder: String?
    ): Cursor? = uri.onPathMatched(uriMatcher) { key, type ->
        localSharedPreferences.getValueAsCursor(key, type)
    }

    override fun getType(uri: Uri): String = context!!.welcomeMessageMimeType

    override fun insert(uri: Uri, values: ContentValues?): Uri? {
        return uri.onMatched(uriMatcher) {
            values?.let { content -> localSharedPreferences.edit().write(content).commit() }
            it
        }
    }

    override fun delete(uri: Uri, selection: String?, selectionArgs: Array<out String>?): Int {
        return uri.onPathMatched(uriMatcher) { key, _ ->
            if (key == COMMON_KEY) {
                localSharedPreferences.edit().clear().commit()
            } else {
                localSharedPreferences.edit().remove(key).commit()
            }
        }
            ?.let { isRemoved -> if (isRemoved) 1 else 0 }
            ?: 0
    }

    override fun update(
        uri: Uri,
        values: ContentValues?,
        selection: String?,
        selectionArgs: Array<out String>?
    ): Int = uri.onMatched(uriMatcher) {
        val wasEdited =
            values?.let { content -> localSharedPreferences.edit().write(content).commit() }
                ?: false
        return@onMatched if (wasEdited) 1 else 0
    } ?: 0

    override fun onSharedPreferenceChanged(sharedPreferences: SharedPreferences?, key: String?) {
        context?.sendWelcomeMessageEvent(ACTION_WELCOME_MESSAGE_CHANGED, key)
    }
}